""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """
import base64
import json
import simplejson
import os
import pika
import sys
import ssl
import time
import yaml
import zipfile
from pika.credentials import ExternalCredentials
from os.path import exists

CONFIG_FILES_YAML = [
    '/opt/cyops/configs/integrations/agent/agent_config.ym']

ZIP_PATH = '/opt/cyops-integrations/integrations/log'

cert_auth = '/api/3/picklists/6fe3b5bb-1c50-455e-b58a-7c2154f18643'


class MQUtils:

    def __init__(self,config_paths=None):
        self.created_new_connection = False
        if config_paths:
            self.config_paths = config_paths
        else:
            config_paths = CONFIG_FILES_YAML
        self.config = self.read_yaml_file(yaml_file_path=config_paths)

    def read_yaml_file(self, yaml_file_path):
        config = {}
        for path in yaml_file_path:
            with open(path, 'r') as yaml_file:
                config.update(yaml.load(yaml_file, Loader=yaml.FullLoader))
        return config

    def decrypt(self, value):
        try:
            sys.path.append('/opt/cyops-integrations/')
            from integrations.integrations.PasswordModule import decrypt
        except Exception as e:
            return value
        output = decrypt(value).strip()
        if not output.startswith('Password:'):
            raise Exception('Password decrypt failed with Error: %s for Password : %s' % output)
        else:
            return output[len('Password:'):]

    def create_rmq_connection(self):
        config = {}
        config['userName'] = self.config.get('cyops.rabbitmq.user')
        config['password'] = self.decrypt(self.config.get('cyops.rabbitmq.password'))
        config['heartbeat'] = self.config.get('cyops.rabbitmq.heartbeat')
        config['ssl'] = self.config.get('cyops.rabbitmq.ssl')
        config['server_cert_file'] = self.config.get(
            'cyops.rabbitmq.sslFileName')
        config['host'] = self.config.get('cyops.rabbitmq.host')
        config['sni'] = self.config.get('cyops.rabbitmq.sni', config['host'])
        config['port'] = self.config.get('cyops.rabbitmq.port')
        config['virtualHost'] = self.config.get('cyops.rabbitmq.vhost')
        config['auth_type'] = self.config.get('cyops.rabbitmq.authType')
        if self.config.get('cyops.rabbitmq.clientCertFileName') and exists(
                self.config.get('cyops.rabbitmq.clientCertFileName')):
            config['client_cert_file'] = self.config.get('cyops.rabbitmq.clientCertFileName')
        if self.config.get('cyops.rabbitmq.clientKeyFileName') and exists(
                self.config.get('cyops.rabbitmq.clientKeyFileName')):
            config['client_key_file'] = self.config.get('cyops.rabbitmq.clientKeyFileName')
        return self.get_rmq_connection(config=config)

    def get_rmq_connection(self, config, consumer=False, *args, **kwargs):
        connection = None
        try:

            if config.get('auth_type') == cert_auth:
                credentials = ExternalCredentials()
            else:
                credentials = pika.PlainCredentials(config['userName'], config['password'])
            heartbeat = config.get('heartbeat')
            retry_delay = 10
            connection_attempts = 5
            if config['ssl']:
                context = ssl.create_default_context(
                    cafile=config.get('server_cert_file'))
                if config.get('client_cert_file'): context.load_cert_chain(config.get('client_cert_file'),
                                                                           config.get('client_key_file')
                                                                           )
                ssl_options = pika.SSLOptions(
                    context, server_hostname=config.get('sni'))
                parameters = pika.ConnectionParameters(config['host'],
                                                       config['port'],
                                                       config['virtualHost'],
                                                       credentials,
                                                       heartbeat=heartbeat,
                                                       retry_delay=retry_delay,
                                                       connection_attempts=connection_attempts,
                                                       ssl_options=ssl_options)

            else:
                parameters = pika.ConnectionParameters(config['host'],
                                                       config['port'],
                                                       config['virtualHost'],
                                                       credentials,
                                                       heartbeat=heartbeat,
                                                       retry_delay=retry_delay,
                                                       connection_attempts=connection_attempts,
                                                       )
            connection = pika.BlockingConnection(parameters)
        except Exception as e:
            print("Unable to notify current state to the master")
        return connection

    def check_sme_connection(self):
        try:
            connection = self.create_rmq_connection()
            if connection:
                connection.close()
                print("True")
                return True
        except Exception as e:
            pass
        print("False")
        return False


    def get_config_key(self, message):
        print(self.config.get(message.get('key_name')))
        return True

    def notify_master(self, data):
        try:
            master_id = self.config.get('cyops.instance.masterId')
            exchange_name = 'dexchange.cyops.postman.' + master_id
            routing_key = 'route.postman.instanceupgrade.remoterequest.' + master_id
            connection = self.create_rmq_connection()
            if not connection:
                return
            self.created_new_connection = True
            channel = connection.channel()
            properties_dict = {'content_type': 'application/json'}
            properties = pika.BasicProperties(**properties_dict)
            channel.basic_publish(exchange=exchange_name,
                                  routing_key=routing_key,
                                  body=data,
                                  properties=properties
                                  )
            print("Published message successfully")
            channel.close()
        except Exception as e:
            print('Error while publish %s', str(e))
            raise e
        finally:
            if self.created_new_connection:
                connection.close()

    def notify_status_to_master(self, message):
        agent_id = self.config.get('cyops.instance.agentId')
        master_id = self.config.get('cyops.instance.masterId')
        data = json.dumps(
            {
                "data": {'time': int(time.time()),
                         'message': message.get('status'),
                         'code': message.get('code')},
                'sourceId': agent_id,
                'destinationId': master_id,
                'action': 'agent-upgrade-request-ack'
            })
        self.notify_master(data)

    def zipfolder(self, destination_zip_file_path, source_zip_folder_path):
        zipobj = zipfile.ZipFile(destination_zip_file_path + '.zip', 'w', zipfile.ZIP_DEFLATED)
        rootlen = len(source_zip_folder_path) + 1
        for base, dirs, files in os.walk(source_zip_folder_path):
            for file in files:
                fn = os.path.join(base, file)
                zipobj.write(fn, fn[rootlen:])

    def collect_logs(self, message):
        agent_id = self.config.get('cyops.instance.agentId')
        master_id = self.config.get('cyops.instance.masterId')
        log_path = message.get('log_path', '/var/log/cyops')
        if not os.path.realpath(log_path).startswith('/var/log/cyops'):
            return
        if os.path.exists(log_path):
            try:
                self.zipfolder(ZIP_PATH, log_path)
            except Exception as e:
                print('Exception occurred while trying to zip the log folder' + str(e))
                return
            log_file = open(ZIP_PATH + '.zip', "rb")
            b_log_content = base64.b64encode(log_file.read())
            log_file.close()
            data = simplejson.dumps(
                {"data": {'time': int(time.time()),
                          'b_log_file': b_log_content},
                 'sourceId': agent_id,
                 'destinationId': master_id,
                 'action': 'collect-app-log-ack'
                 })
            self.notify_master(data)
            os.remove(ZIP_PATH + '.zip')
        else:
            print("The given log path does not exists")


if __name__ == '__main__':
    message = sys.argv[1]
    if not message:
        print("Please supply message to be send")
        quit()
    message = json.loads(message)
    mq_utils = MQUtils()
    if message.get('get_config_key'):
        mq_utils.get_config_key(message)
    elif message.get('collect_log'):
        mq_utils.collect_logs(message)
    elif message.get('sme_health_check'):
        MQUtils(['/opt/cyops/configs/integrations/agent/tmp/agent_config.yml']).check_sme_connection()
    else:
        mq_utils.notify_status_to_master(message)
