""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class RestoreIntegrationTruststore(Tasks):
    TASK_STATUS_MSG = "Restore integration truststore"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._restore_integration_truststore()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    # The restore_integration_truststore() function restore the certificate pem
    # file for integration sdk.
    def _restore_integration_truststore(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            var_post_upgrade_msg = step_result['var_post_upgrade_msg']
            pem_file_backup_dir = step_result['pem_file_backup_dir']

            d_format_integrations_truststore = "/opt/cyops/scripts"
            f_format_integrations_truststore = "format-integrations-truststore.py"

            cmd = "/opt/cyops-integrations/.env/bin/python3 -c 'import certifi; print (certifi.where())'"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            pem_file = result['std_out']
            pem_file = str(pem_file).strip()
            if not pem_file:
                report_msg = "Failed to find the certificate pem file for integration sdk"
                self.logger.debug(report_msg)
                self.print_txt(report_msg)
                # Add message to var_post_upgrade_msg
                step_result['var_post_upgrade_msg'] = var_post_upgrade_msg.append(
                    "Failed to find the certificate pem file for integration sdk")
                self.store_result_to_json(
                    self.store_result_path, 'pre-upgrade', 'initialize', None, step_result)
                
                # not failing upgrade here
                return

            if os.path.exists(pem_file_backup_dir):
                dir_content = os.scandir(pem_file_backup_dir)
                for pem in dir_content:
                    if not pem.name.endswith('.pem'):
                        continue
                    os.system(f"echo \"\" >>{pem_file}")
                    os.system(f"cat {pem_file_backup_dir}/{pem.name} >>{pem_file}")

            os.system(f"rm -rf {pem_file_backup_dir}")
            self._cleanup_duplicate_certificates_from_pem(pem_file)

            # f_format_integrations_truststore this file is being bundled with upgraded cyops-common rpm
            cmd = f"python3 {d_format_integrations_truststore}/{f_format_integrations_truststore}"
            self.cmd_line_utilities.execute_cmd(cmd)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
            return
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at restore integration truststore task. Refer logs at '{LOG_FILE}'"
            )

    def _cleanup_duplicate_certificates_from_pem(self, pem_file):
        try:
            cert = ""
            certs = set()
            total_certs = 0
            for line in open(pem_file, 'rb'):
                line = line.decode("utf-8")
                cert += line
                if '-----END CERTIFICATE-----' in line:
                    # remove the empty lines
                    cert = os.linesep.join([s for s in cert.splitlines() if s])
                    certs.add(cert)
                    cert = ""
                    total_certs = total_certs + 1
            duplicate_certs = total_certs - len(certs)
            if duplicate_certs > 0:
                with open(pem_file, 'wb') as pem:
                    for cert in certs:
                        pem.write(cert.encode())
                        pem.write("\n\n".encode())
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at restore integration truststore task. Refer logs at '{LOG_FILE}'"
            )