""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE
from helpers.utils import Utilities

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class RemovePasswordlessSudoForCsadmin(Tasks):
    TASK_STATUS_MSG = "Remove passwordless sudo for csadmin user"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.utilities = Utilities()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        
        # In case of AWS, we lock the password of csadmin user during cloud-init script.
        # As a result, user can not enter the password of csadmin for performing sudo su. 
        # Hence we need to keep passwordless sudo in case of AWS. 
        
        s_hypervisor_provider = self.utilities.get_hypervisor_provider()
        
        if s_hypervisor_provider == "AWS":
            return False
        
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        
        self._remove_passwordless_sudo()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

            
    def _remove_passwordless_sudo(self):
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_forticloud_enterprise = step_result['flag_is_forticloud_enterprise']
        
        try:
            # user should have password restriction to use sudo commands on csadmin. 
            # PSIRT Mantis : https://mantis.fortinet.com/bug_view_page.php?bug_id=1160895
            f_etc_sudoers="/etc/sudoers"
            cmd = f"sed -i '/csadmin/ s/NOPASSWD://g' {f_etc_sudoers}"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            if result['return_code'] != 0:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return
            
            # In case of FortiCloud Instances, extra entry is present in /etc/sudoers.d/90-cloud-init-users
            if flag_is_forticloud_enterprise:
                f_cloud_init_sudoers = "/etc/sudoers.d/90-cloud-init-users"
                cmd = f"sed -i '/csadmin/ s/NOPASSWD://g' {f_cloud_init_sudoers}"
                result = self.cmd_line_utilities.execute_cmd(cmd)
                if result['return_code'] != 0:
                    self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                    return
            
            # Verify syntax of /etc/sudoers file after execution
            cmd = "visudo -c"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            if result['return_code'] != 0:
                print(result['std_err'])
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
            
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred while removing passwordless sudo for csadmin user. Refer logs at '{LOG_FILE}'"
            )