import os
import subprocess
import getpass
from framework.base.tasks import Tasks
from helpers.cmd_utils import CmdUtils
from helpers.logger import Logger
from constants import (
    FRAMEWORK_PATH,
    LOG_FILE,
    REPORT_FILE,
    STEP_RESULT_FILE,
    TASK_STATUS,
    TASK_LOG_STATUS,
    TEXT_COLOR,
    TEXT_DECORATION
)


class ResetRootPassword(Tasks):
    TASK_STATUS_MSG = "Reset Root Password"

    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)
        self.applicable_version = "7.6.5"

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return "Reset the root password based on user CLI input."

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        applicable_version = int(self.applicable_version.replace('.', ''))
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_forticloud_enterprise = step_result['flag_is_forticloud_enterprise']
        if flag_is_forticloud_enterprise: 
            return False
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version and applicable_version > current_version

    def _yes_no_user_input(self):
        yes_list = ["y","Y","Yes","yes","YES"]
        no_list = ["n","N","No","no","NO"]
        i_count=0
        while i_count <=2:
            user_input = input()
            if user_input in yes_list:
                return True
            elif user_input in no_list:
                return False
            else:
                print("You have provided an invalid input. Enter Yes or No")   
                i_count+=1 
        if  i_count > 2:
            print("Max retries reached, exiting the upgrade.")  
            
    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["STARTED"])
        print(f"{TEXT_COLOR['RED']}IMPORTANT!{TEXT_COLOR['RESET']}")
        print(f"{TEXT_COLOR['YELLOW']}WARNING{TEXT_COLOR['RESET']}: After upgrading to FortiSOAR {self.target_upgrade_version},\n         sudo access to csadmin user will be restricted.\n         If you are not having Root user password for this instance, \n         Please reset the root password in order to avoid loosing root access to the VM.")
        
        print(f"{TEXT_COLOR['YELLOW']}WARNING{TEXT_COLOR['RESET']}: Proceed with resetting the root password [y/n]?")
        if not self._yes_no_user_input():
            print("Skipped resetting `root` user password. If you want to reset password, please refer the Redhat/RockyLinux documentation to reset `root` password from VM console")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
            return

        try:
            MAX_ATTEMPTS = 3
            attempt = 0

            while attempt < MAX_ATTEMPTS:
                new_pass = self._get_password_input("Enter new root password: ")
                confirm_pass = self._get_password_input("Confirm new root password: ")

                if new_pass == confirm_pass:
                    break  # success, continue execution

                attempt += 1
                remaining = MAX_ATTEMPTS - attempt
                print(f"Passwords do not match. Attempts remaining: {remaining}")

                if remaining == 0:
                    self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                    return

            success = self._apply_password(new_pass)
            if not success:
                print("Failed to reset password.")
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return

            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])

        except Exception as ex:
            self.logger.exception(f"Exception during ResetRootPassword: {ex}")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])

        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _get_password_input(self, prompt: str) -> str:
        return getpass.getpass(prompt)

    def _apply_password(self, new_password: str) -> bool:
        """
        Apply root password using: echo -e "pass\npass" | passwd root
        No policy enforcement. passwd itself handles errors.
        """
        cmd = "passwd root"
        try:
            proc = subprocess.Popen(
                cmd,
                shell=True,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True
            )

            # Only root is allowed to set root password, so run as root
            input_data = f"{new_password}\n{new_password}\n"
            out, err = proc.communicate(input=input_data)

            if proc.returncode != 0:
                print("Failed to set root password.")
                print(err.strip())
                return False

            print("Root password updated successfully.")
            return True

        except Exception as ex:
            self.logger.exception(f"Failed to set root password: {ex}")
            return False

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        color = TEXT_COLOR["GREEN"] if status == TASK_STATUS["DONE"] else TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message, " ", colored_status)
        print(final_msg)

    def add_banner_in_log_file(self, msg: str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status, ":", msg)
        border_length = len(status_msg)
        new_line_char = "\n" if status == TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE, 'a') as log_file:
                log_file.write(final_msg)
