import subprocess

from framework.base.tasks import Tasks
from helpers.cmd_utils import CmdUtils
from helpers.logger import Logger
from constants import FRAMEWORK_PATH, LOG_FILE, REPORT_FILE, STEP_RESULT_FILE, TASK_STATUS, TASK_LOG_STATUS, TEXT_COLOR, \
    TEXT_DECORATION
import sys
try:
    # This import will fail in case of external SME
    sys.path.append("/opt/cyops-auth")
    from utilities.ha.node import is_primary
except ImportError:
    pass

class RegenerateAppKeys(Tasks):
    """Implementation class for tasks"""

    TASK_STATUS_MSG = "Regenerating APP keys"

    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        """Tags for task"""
        return 'post_upgrade'

    def get_description(self) -> str:
        """Description of a method"""
        return "Regenerating APP keys"

    def is_supported(self) -> bool:
        """To check task is for target upgrade version/platform"""
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        # Check if the key length is already 3072
        cmd = "openssl rsa -in /etc/pki/cyops/jwtprivate.key -text -noout | grep \"Private-Key\" | awk '{print $2}' | tr -d '()'"
        result = self.cmd_line_utilities.execute_cmd(cmd,trap_output=True,use_shell=True)
        is_converted = True
        if result['return_code'] == 0:
            std_out = result['std_out'].strip()
            key_length = int(std_out) if std_out.isdigit() else 0
            if key_length < 3072:
                is_converted = False
        else:
            std_err = result['std_err']
            self.logger.error(f"Error occurred {std_err}")
        return not is_converted and target_upgrade_version >= 764 and self.is_enterprise()

    def execute(self) -> None:
        """To execute main code"""
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["STARTED"])
        cmds = [
            "csadm certs --app-key",
            "rm -rf /opt/cyops-api/var/cache",
            "systemctl restart php-fpm",
            "sudo -u nginx php /opt/cyops-api/bin/console cache:clear",
            "systemctl restart php-fpm",
            "csadm services --restart"
        ]
        for cmd in cmds:
            self.cmd_line_utilities.execute_cmd(cmd, True)
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        """Write validations in this function"""
        return True

    def is_enterprise(self):
        step_result = self.get_step_results("pre-upgrade", "initialize")
        return step_result["flag_is_enterprise"]

    def add_banner_in_log_file(self, msg: str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status, ":", msg)
        border_length = len(status_msg)
        border = '=' * border_length
        new_line_char = "\n" if status == TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        try:
            with open(LOG_FILE, 'a') as log_file:
                log_file.write(final_msg)
        except Exception as e:
            self.logger.error(f"Failed to write log file: {e}")

    def _print_status_msg(self, msg: str, status: str) -> None:
        """Use this function to print status message of task execution"""
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message, " ", colored_status)
        print(final_msg)