""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class ValidateYumRepo(Tasks):
    TASK_STATUS_MSG = "Validate yum repo connectivity"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version > current_version

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        b_validate_yum_repo = self._validate_yum_repo()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        return b_validate_yum_repo

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _remove_config_vm_call_from_bash_profile(self):
        # Enable back control C
        # trap 2
        # Remove config vm call from bash profile
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        f_bash_profile = step_result['f_bash_profile']
        os.system("sed -i '/config-vm.sh$/d' {}".format(f_bash_profile))
        os.system("sed -i '/check_eula.sh$/d' {}".format(f_bash_profile))

    # The validate_custom_yum_url_connection function check whether custom_yum_url variable value (URL) is reachable or not \
    # If the custom_yum_url variable value (URL) not reachable then exit script and avoid to make wrong entry in the repo files.
    def _validate_yum_repo(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            custom_yum_url = step_result['custom_yum_url']
            s_flag = step_result['s_flag']
            s_url = step_result['s_url']
            if custom_yum_url:
                # The custom_yum_url variable export before run the script.
                # The s_url variable value validate in the validate_yum_repo function.
                s_url = custom_yum_url
            
            if not s_url.startswith("https://"):
                s_url = "https://{}".format(s_url)
            
            i_status = None
            cmd = "curl {} {}".format(s_flag, s_url)
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            std_out = result['std_out'].split('\n')[0]
            if std_out:
                l_elements = list((element.strip())
                                for element in (std_out.split(' ')) if element)
                i_status = int(l_elements[1])

            # Check if response will not 200 or i_status variable will be empty
            # then user entered URL will not reachable.
            if not i_status or i_status != 200:
                msg = f"Failed to connect the '{s_url}'. Check connection for '{s_url}' and re-run the command."
                self.store_result_to_json(
                    self.report_path, None, None, 'Verify Yum Repo Connection', {"result": False, "message": msg})
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                print(msg)
                return False
            msg = f"Connection to '{s_url}' repo is successful"
            self.store_result_to_json(
                self.report_path, None, None, 'Verify Yum Repo Connection', {"result": True, "message": msg})
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at validate yum repo task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(self.report_path, None, None,
                                    'Verify Yum Repo Connection', {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"})
