""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys

import psutil
import subprocess
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import (
    LOG_FILE,
    REPORT_FILE,
    STEP_RESULT_FILE
)


TASK_STATUS = {"DONE": "DONE", "FAILED": "FAILED", "SKIPPED": "SKIPPED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}
RECOMMENDED_VCPU_COUNT = 12
RECOMMENDED_RAM_SIZE = 48
MINIMUM_VCPU_COUNT = 8
MINIMUM_RAM_SIZE = 32

class CheckSystemRequirement(Tasks):
    TASK_STATUS_MSG = "Validate required system resources"

    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return "pre-upgrade"

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace(".", ""))
        target_upgrade_version = int(self.target_upgrade_version.replace(".", ""))
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        return target_upgrade_version > current_version and flag_is_enterprise

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        result = self.check_system_requirements()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG, TASK_LOG_STATUS["COMPLETED"])
        return result

    def _yes_no_user_input(self):
        yes_list = ["y","Y","Yes","yes","YES"]
        no_list = ["n","N","No","no","NO"]
        i_count=0
        while i_count <=2:
            user_input = input()
            if user_input in yes_list:
                return True
            elif user_input in no_list:
                return False
            else:
                print("You have provided an invalid input. Enter Yes or No")
                i_count+=1
        print("Max retries reached, exiting the upgrade.")
        return False

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        elif status == TASK_STATUS["SKIPPED"]:
            color = TEXT_COLOR["YELLOW"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)

    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def check_system_requirements(self):
        cpu = CheckSystemRequirement._get_vcpu_count()
        ram = self._get_ram_size()

        # Thresholds
        MIN_CPU, MIN_RAM = MINIMUM_VCPU_COUNT, MINIMUM_RAM_SIZE
        REC_CPU, REC_RAM = RECOMMENDED_VCPU_COUNT, RECOMMENDED_RAM_SIZE

        errors = []

        # Mandatory failures
        if cpu < MIN_CPU:
            errors.append(f"- vCPU count: {cpu} (Minimum required: {MIN_CPU})")

        if ram < MIN_RAM:
            errors.append(f"- RAM size: {ram} GB (Minimum required: {MIN_RAM} GB)")

        # Result evaluation for mandatory requirements
        if not errors:
            # Recommended warning (non-fatal)
            recommended_warnings = []

            if cpu < REC_CPU:
                recommended_warnings.append(
                    f"- vCPU count: {cpu} (Recommended : {REC_CPU})"
                )

            if ram < REC_RAM:
                recommended_warnings.append(
                    f"- RAM size: {ram} GB (Recommended : {REC_RAM} GB)"
                )

            # Store success result
            self.store_result_to_json(
                self.report_path, None, None, self.TASK_STATUS_MSG,
                {"result": True, "message": "Minimum system requirements met."}
            )
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS['DONE'])

            # Print recommended warnings (yellow)
            if recommended_warnings:
                print(f"\n{TEXT_COLOR['YELLOW']}Recommended configuration not met:{TEXT_COLOR['RESET']}")
                for w in recommended_warnings:
                    print(f"  {w}")
            
                # Ask if user wants to continue
                print(f"\n{TEXT_COLOR['YELLOW']}Warning{TEXT_COLOR['RESET']}: "
                    f"Do you still want to proceed with upgrade? [y/n]")

                if not self._yes_no_user_input():
                    sys.exit()

            return True

        # FAILED mandatory requirements
        error_header = f"\n{TEXT_COLOR['RED']}Minimum system requirements not met. Please review:{TEXT_COLOR['RESET']}"
        report_msg = f"{error_header}\n" + "\n".join(errors)

        self.logger.error(report_msg)
        self.store_result_to_json(
            self.report_path, None, None, self.TASK_STATUS_MSG,
            {"result": False, "message": report_msg}
        )
        self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS['FAILED'])
        print(report_msg)

        # Ask user if they still want to proceed
        print(f"\n{TEXT_COLOR['YELLOW']}Warning{TEXT_COLOR['RESET']}: "
              f"Do you still want to proceed with upgrade? [y/n]")

        if not self._yes_no_user_input():
            sys.exit()

        return True
    @staticmethod
    def _get_vcpu_count():
        """
        Returns the number of virtual (logical) CPUs.
        """
        return psutil.cpu_count(logical=True)

    def _get_ram_size(self):
        try:
            cmd = "lsmem --summary=only -b | grep 'Total online memory:' | awk -F': ' '{print $2}' | tr -d ' '"
            mem_bytes = int(subprocess.check_output(cmd, shell=True))
            return mem_bytes / (1024 ** 3)
        except Exception as err:
            self.logger.error("Error occurred while getting RAM size. Error: {0}".format(err))
            return 0