""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import re
import sys
import subprocess
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class Preparation(Tasks):
    TASK_STATUS_MSG = "Prepare system for upgrade"
    PRODUCT_YUM_SERVER_FILE_STATUS_MSG = "Update 'product_yum_server' file for custom yum repo"
    CYOPS_REPO_UPDATE_STATUS_MSG = "Install 'cyops-repo-update' rpm"
    PRE_UPGRADE_PREPARATIONS_STATUS_MSG = "Handle pre-upgrade preparations"
    ENVIRONMENT_FILE_STATUS_MSG = "Update 'environment' file for custom yum repo"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version > current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        b_prep = self._prep()
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        
        if not b_prep:
            sys.exit()

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
        
    def _remove_config_vm_call_from_bash_profile(self):
        # Enable back control C
        # trap 2
        # Remove config vm call from bash profile
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        f_bash_profile = step_result['f_bash_profile']
        os.system("sed -i '/config-vm.sh$/d' {}".format(f_bash_profile))
        os.system("sed -i '/check_eula.sh$/d' {}".format(f_bash_profile))

    # The install_cyops_repo_update() function upgrade the cyops-repo-upgrade rpm package.
    def _install_cyops_repo_update(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            s_repo_update_rpm = step_result['s_repo_update_rpm']
            
            cmd = "rpm --force -Uvh "+s_repo_update_rpm
            result = self.cmd_line_utilities.execute_cmd(cmd)
            return_code = result['return_code']
            if return_code != 0:
                report_msg = "{}ERROR{}: Failed to install {}.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],s_repo_update_rpm)
                self._print_status_msg(self.CYOPS_REPO_UPDATE_STATUS_MSG, TASK_STATUS["FAILED"])
                self.print_txt(report_msg)
                self.store_result_to_json(self.report_path, None, None, 'Install Cyops Repo Update', {"result": False, "message": report_msg})
                self._remove_config_vm_call_from_bash_profile()
                return False
            report_msg = "Successfully installed {}.".format(s_repo_update_rpm)
            self.store_result_to_json(self.report_path, None, None, 'Install Cyops Repo Update', {"result": True, "message": report_msg})
            self.logger.info(report_msg)
            self._print_status_msg(self.CYOPS_REPO_UPDATE_STATUS_MSG, TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.CYOPS_REPO_UPDATE_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )
    
    # If organization has it's own yum repository then \
    # organization can change default repository using the following function.
    # The custom_yum_repository variable export before run the script.
    def _custom_yum_repo(self, custom_yum_url):
        try:
            f_product_yum_repo = "/etc/yum/vars/product_yum_server"
            error_occurred = False
            if os.path.exists(f_product_yum_repo):
                file = None
                try:
                    file = open(f_product_yum_repo, 'w')
                except IOError:
                    self._print_status_msg(self.PRODUCT_YUM_SERVER_FILE_STATUS_MSG, TASK_STATUS["FAILED"])
                    msg = "Failed to update the \"{f_product_yum_repo}\" file.\nCheck if the \"{f_product_yum_repo}\" file has write permission.".format(
                        f_product_yum_repo=f_product_yum_repo)
                    print(msg)
                    err_msg = "Failed to update the \"{f_product_yum_repo}\" file. Check if the \"{f_product_yum_repo}\" file has write permission.".format(
                        f_product_yum_repo=f_product_yum_repo)
                    self.logger.error(f"ERROR: {err_msg}")
                    error_occurred = True
                else:
                    file.write(custom_yum_url)
                finally:
                    if file:
                        file.close()
            else:
                msg = "'{}' does not exits".format(f_product_yum_repo)
                err_msg = "{}ERROR{}:{}".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],msg)
                self.logger.error(msg)
                self._print_status_msg(self.PRODUCT_YUM_SERVER_FILE_STATUS_MSG, TASK_STATUS["FAILED"])
                print(err_msg)
                error_occurred = True
            
            if error_occurred:
                return
            
            msg = "Successfully updated file '{}' with '{}'".format(
                f_product_yum_repo, custom_yum_url)
            self.logger.info(msg)
            self._print_status_msg(self.PRODUCT_YUM_SERVER_FILE_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.PRODUCT_YUM_SERVER_FILE_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )

    # The update_environment_file function updates entry in the file /etc/environment with custom_yum_url variable value.
    # Also, this function calls in cleanup function because cyops rpm package added the entry of \
    # product_yum_server=repo.fortisoar.fortinet.com. So to change the value of product_yum_server variable \
    # update_environment_file function calls in cleanup function.
    def _update_environment_file(self, custom_yum_url):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            f_etc_environment = step_result['f_etc_environment']
            cmd = "grep -qw {} {}".format(custom_yum_url, f_etc_environment)
            ret_code = os.system(cmd)
            if ret_code != 0:
                cmd = "sed -i \"/product_yum_server/c\product_yum_server={}\" {}".format(
                    custom_yum_url, f_etc_environment)
                result = self.cmd_line_utilities.execute_cmd(cmd)
                i_flag_status = result['return_code']
                if i_flag_status != 0:
                    self._print_status_msg(self.ENVIRONMENT_FILE_STATUS_MSG, TASK_STATUS["FAILED"])
                    msg = "Failed to update the \"{f_etc_environment}\" file.\nCheck if the \"{f_etc_environment}\" file has write permission.".format(
                        f_etc_environment=f_etc_environment)
                    print(msg)
                    err_msg = "Failed to update the \"{f_etc_environment}\" file. Check if the \"{f_etc_environment}\" file has write permission.".format(
                        f_etc_environment=f_etc_environment)
                    self.logger.error(f"ERROR: {err_msg}")
                    return
                msg = "Successfully updated file '{}' with '{}'".format(f_etc_environment, custom_yum_url)
                self.logger.info(msg)
                self._print_status_msg(self.ENVIRONMENT_FILE_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.ENVIRONMENT_FILE_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )

    # The following function runs after cyops-repo-update rpm install.
    # Also, update .repo file to specify the rhel repository path.
    # This function will disable all 'fsr-rockylinux-*' repos except 'fsr-epel-9' in 'fsr-os.repo'
    def _update_rhel_repo(self):
        try:
            result = subprocess.run(['yum', 'repolist'], stdout=subprocess.PIPE, stderr=subprocess.DEVNULL, text=True)
            repolist_output = result.stdout

            rockylinux_repos = []
            if repolist_output:
                for line in repolist_output.splitlines():
                    if "fsr-rockylinux-" in line:
                        repo_id = line.split()[0]
                        rockylinux_repos.append(repo_id)
            
            if rockylinux_repos:
                for repo in rockylinux_repos:
                    print(f"Disabling the '{repo}' repo")
                    subprocess.run(['yum-config-manager', '--disable', repo], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.PRE_UPGRADE_PREPARATIONS_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )

    def _update_pip_conf(self, f_pip_conf, custom_yum_url):
        try:
            # The following tickets have details to add update_pip_conf function.
            # https://mantis.fortinet.com/bug_view_page.php?bug_id=0793364
            # https://mantis.fortinet.com/bug_view_page.php?bug_id=0771138
            # https://mantis.fortinet.com/bug_view_page.php?bug_id=0771134
            s_trusted_host = "trusted-host"

            if not os.path.exists(f_pip_conf):
                err_msg = "The '{}' does not found.".format(f_pip_conf)
                self.print_txt(err_msg)
                self.logger.error(f"ERROR: {err_msg}")
                return False
            
            cmd = "grep -wq '{}' {}".format(custom_yum_url, f_pip_conf)
            result = self.cmd_line_utilities.execute_cmd(cmd)
            return_code = result['return_code']
            if return_code == 0:
                msg = "{} already present in {}.".format(
                    custom_yum_url, f_pip_conf)
                self.print_txt(msg)
                return True
            
            cmd = "grep -w \"{}\" {}".format(s_trusted_host, f_pip_conf)
            result = self.cmd_line_utilities.execute_cmd(cmd)
            is_trusted_host_exist = result['return_code']
            
            local_custom_repo_name = custom_yum_url
            local_custom_yum_url = local_custom_repo_name
            if not local_custom_yum_url.startswith("https://"):
                local_custom_yum_url = "https://{}".format(local_custom_yum_url)
            
            # Making pip.conf file mutable for updating the file
            os.system(f"chattr -i {f_pip_conf}")
            # Install the connector if install from offline repo.
            # If trusted-host not exit in pip.conf files then connector installation fails.
            if is_trusted_host_exist != 0:
                with open(f_pip_conf, 'a') as conf_file:
                    conf_file.write('{}={}'.format(s_trusted_host, local_custom_repo_name))
            
            url_pattern = "http[s]:\/\/(?:[a-zA-z0-9]+\.)+com"
            cmd = "grep -w \"extra-index-url\" {}".format(f_pip_conf)
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            return_code = result['return_code']
            s_repo_name = ""
            if return_code == 0:
                std_out = result['std_out']
                s_repo_name = re.findall(url_pattern, std_out)
                s_repo_name = s_repo_name[0] if s_repo_name else ""

            if s_repo_name:
                # custom yum url not found in pip conf file, updating pip conf file
                cmd = "sed -i 's|{}|{}|g' {}".format(s_repo_name, local_custom_yum_url, f_pip_conf)
                result = self.cmd_line_utilities.execute_cmd(cmd)
                return_code = result['return_code']

                if return_code != 0:
                    err_msg = "Failed to update the file {}.".format(f_pip_conf)
                    self.print_txt(err_msg)
                    self.logger.error(f"ERROR: {err_msg}")
                    # Making pip.conf file immutable if failed to update the file
                    os.system(f"chattr +i {f_pip_conf}")
                    return False
            # Making pip.conf file immutable after updating the file
            os.system(f"chattr +i {f_pip_conf}")
            return True
        except Exception as ex:
            # Making pip.conf file immutable if exception occurs
            os.system(f"chattr +i {f_pip_conf}")
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.PRE_UPGRADE_PREPARATIONS_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )

    def _prep(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            custom_yum_url = step_result['custom_yum_url']
            flag_is_rhel = step_result['flag_is_rhel']
            l_pip_conf = step_result['l_pip_conf']
            
            if not self._install_cyops_repo_update():
                return False
            # The following condition checks for offline repo.
            if custom_yum_url:
                self._custom_yum_repo(custom_yum_url)
                self._update_environment_file(custom_yum_url)
                for f_pip_conf in l_pip_conf:
                    # Call for /opt/cyops-integrations/.env/pip.conf to update
                    result = self._update_pip_conf(f_pip_conf, custom_yum_url)
                    if not result:
                        self._remove_config_vm_call_from_bash_profile()
                        return False

            if flag_is_rhel:
                self._update_rhel_repo()
            self._print_status_msg(self.PRE_UPGRADE_PREPARATIONS_STATUS_MSG, TASK_STATUS["DONE"])
            return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.PRE_UPGRADE_PREPARATIONS_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at preparation task. Refer logs at '{LOG_FILE}'"
            )
