""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from framework.utility.messages import ERROR_OCCURED
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE,HTTPS_STATUS_CODE_AND_MESSAGES

SMM_ENDPOINT = "https://localhost/api/3/staging_model_metadatas?$limit=2147483647&$orderby=type&$relationships=true&$export=true"
MM_ENDPOINT = "https://localhost/api/3/model_metadatas?$limit=2147483647&$orderby=type&$relationships=true&$export=true"

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class CheckUnpublishedModules(Tasks):
    TASK_STATUS_MSG = "Verify all modules are published"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version > current_version and not step_result['is_passive_node']

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        b_check_module_publish = True
        
        if flag_is_enterprise:
            b_check_module_publish =self._check_module_publish()
        
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        return b_check_module_publish
        
    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _refine_data(self, data):
        data.pop('@context', None)
        data.pop('@id', None)
        hydra_member = data['hydra:member']

        for each_item in hydra_member:
            if '@id' in each_item:
                each_item.pop('@id', None)
            each_item.pop('@type', None)

            if 'attributes' in each_item:
                attributes_data = each_item['attributes']
                for attributes_item in attributes_data:
                    if '@id' in attributes_item:
                        attributes_item.pop('@id', None)
                    attributes_item.pop('@type', None)
                    attributes_item.pop('sattrib', None)

        hydra_view = data['hydra:view']
        hydra_view.pop('@id', None)
        return data

    def _check_json_diff(self, smm, mm):
        smm_hydra_member = smm['hydra:member']
        mm_hydra_member = mm['hydra:member']

        if len(smm_hydra_member) > len(mm_hydra_member) or len(smm_hydra_member) < len(mm_hydra_member):
            return True
        
        for smm_hydra_member_item in smm_hydra_member:
            for mm_hydra_member_item in mm_hydra_member:
                if smm_hydra_member_item.get('type') == mm_hydra_member_item.get('type'):
                    smm_hydra_member_attrib = smm_hydra_member_item.pop(
                        'attributes', None)
                    mm_hydra_member_attrib = mm_hydra_member_item.pop(
                        'attributes', None)
                    if smm_hydra_member_item == mm_hydra_member_item:
                        # No diff in module
                        if not self._check_attrib_diff(smm_hydra_member_attrib, mm_hydra_member_attrib):
                            # No diff in module attributes
                            continue
                        else:
                            # Diff in module attributes
                            return True
                    else:
                        # Diff in module
                        return True
        return False

    def _check_attrib_diff(self, smm_attrib, mm_attrib):
        if len(smm_attrib) > len(mm_attrib) or len(smm_attrib) < len(mm_attrib):
            return True

        for each_smm_attrib in smm_attrib:
            for each_mm_attrib in mm_attrib:
                if each_smm_attrib.get('name') == each_mm_attrib.get('name'):
                    if each_smm_attrib == each_mm_attrib:
                        continue
                    else:
                        return True
        return False
    
    def _check_module_publish(self):
        try:
            https_status_codes = HTTPS_STATUS_CODE_AND_MESSAGES.keys()
            smm_response = self.task_utilities._make_cyops_request(SMM_ENDPOINT,"GET")
            mm_response = self.task_utilities._make_cyops_request(MM_ENDPOINT,"GET")

            if smm_response.status_code != 200 or mm_response.status_code != 200:
                smm_https_response = HTTPS_STATUS_CODE_AND_MESSAGES[smm_response.status_code] if smm_response.status_code in https_status_codes else f"Https status code : {smm_response.status_code}"
                mm_https_response = HTTPS_STATUS_CODE_AND_MESSAGES[mm_response.status_code] if mm_response.status_code in https_status_codes else f"Https status code : {mm_response.status_code}"
                https_response = smm_https_response if smm_response.status_code != 200 else mm_https_response
                err_msg = ERROR_OCCURED.format("fetching data from 'staging_model_metadatas'", https_response)

                report_msg = "ERROR: Unable to fetch mmd details."
                self.logger.error(err_msg)
                self.print_txt("{}ERROR{}: Unable to fetch mmd details.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"]))
                self.store_result_to_json(
                    self.report_path, None, None, 'Verify Publish Status Of All Modules', {"result": False, "message": report_msg})
                return False

            smm_data = smm_response.json()
            mm_data = mm_response.json()
            smm_refined = self._refine_data(smm_data)
            mm_refined = self._refine_data(mm_data)
            if not self._check_json_diff(smm_refined, mm_refined):
                report_msg = "All modules in current system are in published state"
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
                self.store_result_to_json(
                    self.report_path, None, None, 'Verify Publish Status Of All Modules', {"result": True, "message": report_msg})
                return True
            msg = "Some of the FortiSOAR Modules changes are not published yet.\nYou must publish the pending changes, or revert the changes before starting with the upgrade."
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(msg)
            report_msg = "Some of the FortiSOAR Modules changes are not published yet. You must publish the pending changes, or revert the changes before starting with the upgrade."
            self.logger.error(report_msg)
            self.store_result_to_json(
                self.report_path, None, None, 'Verify Publish Status Of All Modules', {"result": False, "message": report_msg})
            return False
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at check unpublished modules task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(self.report_path, None, None,
                                    'Verify Publish Status Of All Modules', {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"})
