""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
import urllib3
import subprocess
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import LOG_FILE, REPORT_FILE, STEP_RESULT_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

urllib3.disable_warnings()

try:
    sys.path.append("/opt/cyops-auth")
    from utilities.config import config
    from utilities.csconstants import NODE_HARDWARE_KEY
    from handlerworkers.cluster import get_cluster_nodes
    from utilities.ha.common_utils import get_dash_string
    from utilities.ha.node import is_primary, get_other_cluster_node_hostnames
except ImportError:
    pass

class VerifyPresenceOfCluster(Tasks):
    TASK_STATUS_MSG = "Verify presence of cluster"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        # Cluster presence should be checked for upgrade upto 7.6.1 version
        if current_version < 761:
            return True
        # For rolling upgrade, there is no necessity of checking cluster presence. Skipping this file during execution.
        return False

    def execute(self):
        pass

    def validate(self) -> bool:
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        b_handle_cluster_upgrade = True
        
        if flag_is_enterprise:
            b_handle_cluster_upgrade = self._handle_cluster_upgrade()

        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])
        return b_handle_cluster_upgrade

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _remove_config_vm_call_from_bash_profile(self):
        # Enable back control C
        # trap 2
        # Remove config vm call from bash profile
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        f_bash_profile = step_result['f_bash_profile']
        os.system("sed -i '/config-vm.sh$/d' {}".format(f_bash_profile))
        os.system("sed -i '/check_eula.sh$/d' {}".format(f_bash_profile))

    def _print_in_dash_string(self, input_msg):
        dash_string = get_dash_string()
        msg = dash_string + "\n"
        msg = msg + input_msg + "\n"
        msg = msg + dash_string + "\n"
        print(msg)

    def _get_leave_cluster_command(self):
        return "csadm ha leave-cluster"
    
    def _get_join_cluster_command(self):
        return "csadm ha join-cluster"

    def _get_suspend_cluster_command(self):
        return "csadm ha suspend-cluster"

    def _get_resume_cluster_command(self):
        return "csadm ha resume-cluster"

    def _is_device_uuid_changed(self):
        try:
            previous_node_id = config.get('CLUSTER', 'nodeid')
            if NODE_HARDWARE_KEY != previous_node_id:
                return True
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at handle cluster upgrade task. Refer logs at '{LOG_FILE}'"
            )
        return False

    def _handle_primary_node_upgrade(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            is_postgres_upgrade_available = step_result['is_postgres_upgrade_available']
            other_cluster_node_hostnames = get_other_cluster_node_hostnames()
            cluster_dissolve_cmd = self._get_leave_cluster_command() if is_postgres_upgrade_available else self._get_suspend_cluster_command()
            cluster_reform_cmd = self._get_join_cluster_command() if is_postgres_upgrade_available else self._get_resume_cluster_command()
            if (other_cluster_node_hostnames is None) or len(other_cluster_node_hostnames) == 0:
                report_msg = "No other cluster nodes found."
                self.store_result_to_json(
                    self.report_path, None, None, self.TASK_STATUS_MSG, {"result": True, "message": report_msg})
                self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["DONE"])
                return True
            if self._is_device_uuid_changed():
                self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["FAILED"])
                self.print_txt("There is a mismatch in the device UUID of this FortiSOAR instance, possibly\n" +
                               "due to VM cloning, migration or revert. Update the device UUID by running the following\n" +
                               "command and re-run the upgrade.\n" +
                               "sudo csadm license --refresh-device-uuid")
                report_msg = "There is a mismatch in the device UUID of this FortiSOAR instance, possibly due to VM cloning, migration or revert. Update the device UUID by running 'sudo csadm license --refresh-device-uuid' command and re-run the upgrade."
                self.store_result_to_json(
                    self.report_path, None, None, self.TASK_STATUS_MSG, {"result": False, "message": report_msg})
                return False

            err_msg = "{}ERROR{}:\n".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"])
            err_msg = err_msg + "The current node is the primary node of the cluster. Before upgrading,\n"
            err_msg = err_msg + "you must break the cluster by running '" + cluster_dissolve_cmd + "' on the\n"
            err_msg = err_msg + "secondary node(s). Once the upgrade on all the nodes is complete, \n"
            err_msg = err_msg + "reform the cluster using '" + cluster_reform_cmd + "' on the secondary node(s)."
            err_msg = err_msg + "\n\n" + "Secondary Node(s) : {}".format(other_cluster_node_hostnames)
            self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["FAILED"])
            self._print_in_dash_string(err_msg)
            report_msg = "The current node is the primary node of the cluster. Before upgrading, you must break the cluster by running '{}' on the secondary node(s) : {}. Once the upgrade on all the nodes is complete, reform the cluster using '{}' on the secondary node(s).".format(
                cluster_dissolve_cmd, other_cluster_node_hostnames,cluster_reform_cmd)
            self.store_result_to_json(
                self.report_path, None, None, self.TASK_STATUS_MSG, {"result": False, "message": report_msg})

            self._remove_config_vm_call_from_bash_profile()
            return False
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at handle cluster upgrade task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(self.report_path, None, None,
                                    self.TASK_STATUS_MSG, {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"})

    def _handle_secondary_node_upgrade(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            is_postgres_upgrade_available = step_result['is_postgres_upgrade_available']
            other_cluster_node_hostnames = get_other_cluster_node_hostnames()
            cluster_dissolve_cmd = self._get_leave_cluster_command() if is_postgres_upgrade_available else self._get_suspend_cluster_command()
            cluster_reform_cmd = self._get_join_cluster_command() if is_postgres_upgrade_available else self._get_resume_cluster_command()
            print_msg = "{}ERROR{}:\nThis node is part of an HA cluster. Before upgrading,\nyou must break the cluster by running the '{}' command.\nOnce the upgrade is complete, reform the cluster by running\n'{}' command.".format(TEXT_COLOR["RED"],TEXT_COLOR["RESET"],cluster_dissolve_cmd,cluster_reform_cmd)
            report_msg = "This node is part of an HA cluster. Before upgrading you must break the cluster by running the '{}' command. Once the upgrade is complete, reform the cluster by running '{}' command.".format(cluster_dissolve_cmd,cluster_reform_cmd)
            self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["FAILED"])
            self._print_in_dash_string(print_msg)
            self.store_result_to_json(
                self.report_path, None, None, self.TASK_STATUS_MSG, {"result": False, "message": report_msg})

            self._remove_config_vm_call_from_bash_profile()
            return False
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at handle cluster upgrade task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(self.report_path, None, None,
                                    self.TASK_STATUS_MSG, {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"})

    def _handle_cluster_upgrade(self):
        try:
            if is_primary():
                result = self._handle_primary_node_upgrade()
                return result
            result = self._handle_secondary_node_upgrade()
            return result
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG,TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at handle cluster upgrade task. Refer logs at '{LOG_FILE}'"
            )
            self.store_result_to_json(self.report_path, None, None,
                                    self.TASK_STATUS_MSG, {"result": False, "message": f"Execution failed. Please refer '{LOG_FILE}'"})
