""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
from datetime import date
from helpers.logger import Logger
from framework.base.tasks import Tasks
from helpers.cmd_utils import CmdUtils
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class UpdateBanner(Tasks):
    TASK_STATUS_MSG = "Update banner in '/etc/motd'"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._update_banner()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True
    
    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _update_banner(self):
        try:
            step_result = self.get_step_results('pre-upgrade', 'initialize')
            f_os_release = step_result['f_os_release']
            s_release_version = step_result['s_release_version']
            s_pkg = step_result['s_pkg']
            f_motd = "/etc/motd"

            f_os_release = step_result['f_os_release']
            with open(f_os_release, 'r') as os_release:
                result = dict((a.strip(), (b.strip())) for a, b in (element.split(
                    '=') for element in os_release.read().split('\n') if element))
                os_flavor = result['NAME'].strip('\"')
                version_id = result['VERSION_ID'].strip('\"')
            s_current_operating_system = os_flavor
            s_platform_release = version_id
            cmd = f"rpm -q {s_pkg}"
            result = self.cmd_line_utilities.execute_cmd(cmd, True)
            result = result['std_out']
            i_upgraded_cyops_build = result.split("-")[3].split(".")[0]

            today_date = date.today()
            data_to_add_in_motd_file = f"----------------------------------------------------------------\n{s_current_operating_system} Version: {s_platform_release}\nFortiSOAR Version: {s_release_version}-{i_upgraded_cyops_build}\nUpgraded on: {today_date}\n----------------------------------------------------------------"

            with open(f_motd, 'w') as motd_file:
                motd_file.write(data_to_add_in_motd_file)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at update banner task. Refer logs at '{LOG_FILE}'"
            )