""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import shutil
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE
from helpers.utils import Utilities

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class RestrictSudoAccessForCsadminUser(Tasks):
    TASK_STATUS_MSG = "Restrict Sudo Access for Csadmin user"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.utilities = Utilities()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        
        # In case of AWS, we lock the password of csadmin user during cloud-init script.
        # As a result, user can not enter the password of csadmin for performing sudo su. 
        # Hence we need to keep passwordless sudo in case of AWS. 
        
        s_hypervisor_provider = self.utilities.get_hypervisor_provider()
        
        if s_hypervisor_provider == "AWS":
            return False
        
        return target_upgrade_version >= current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        
        self._remove_existing_sudo_access()
        
        self._configure_ssh_group()
        
        # self._restrict_sudo_access()
            
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)
                
    def _configure_ssh_group(self):
        """Remove user from wheel group, create 'fortisoar_ssh' group, add user, update sshd_config."""
        user = "csadmin"
        group = "fortisoar_ssh"
        sshd_config = "/etc/ssh/sshd_config"

        # 1. Remove user from wheel group
        cmd = f"gpasswd -d {user} wheel"
        result = self.cmd_line_utilities.execute_cmd(cmd)
        if result["return_code"] != 0:
            print(f"Warning: Failed to remove {user} from wheel group or user not in wheel.")
        else:
            print(f"Removed {user} from wheel group.")

        # 2. Create group if it doesn't exist
        cmd = f"getent group {group}"
        result = self.cmd_line_utilities.execute_cmd(cmd)
        if result["return_code"] != 0:
            cmd = f"groupadd {group}"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            if result["return_code"] == 0:
                print(f"Group '{group}' created.")
            else:
                print(f"Failed to create group '{group}'.")
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return
        else:
            print(f"Group '{group}' already exists.")

        # 3. Add user to the group
        cmd = f"usermod -aG {group} {user}"
        result = self.cmd_line_utilities.execute_cmd(cmd)
        if result["return_code"] != 0:
            print(f"Failed to add {user} to group '{group}'.")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            return
        else:
            print(f"User '{user}' added to group '{group}'.")

        # 4. Update sshd_config to allow the group
        cmd = f"grep -q '^AllowGroups' {sshd_config}"
        result = self.cmd_line_utilities.execute_cmd(cmd)

        if result["return_code"] != 0:
            # No AllowGroups line, append it
            cmd = f"echo 'AllowGroups {group}' >> {sshd_config}"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            print(f"Added 'AllowGroups {group}' to {sshd_config}.")
        else:
            # AllowGroups exists, check if group already present
            cmd = f"grep -q '{group}' {sshd_config}"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            if result["return_code"] != 0:
                cmd = f"sed -i '/^AllowGroups/ s/$/ {group}/' {sshd_config}"
                result = self.cmd_line_utilities.execute_cmd(cmd)
                print(f"Appended '{group}' to existing AllowGroups.")
            else:
                print(f"Group '{group}' already present in AllowGroups.")

        # 5. Restart sshd
        cmd = "systemctl restart sshd"
        result = self.cmd_line_utilities.execute_cmd(cmd)
        if result["return_code"] != 0:
            print("Failed to restart sshd. Check logs for details.")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            return
        else:
            print("sshd restarted. Test SSH login before closing this session.")


    def _remove_existing_sudo_access(self):
        step_result = self.get_step_results('pre-upgrade', 'initialize')      
        flag_is_forticloud_enterprise = step_result['flag_is_forticloud_enterprise']
        try:
            sudoers_file = "/etc/sudoers"
            backup_file = "/etc/sudoers.bak"
            user = "csadmin"

            # Create a backup first
            cmd = f"cp {sudoers_file} {backup_file}"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            if result['return_code'] != 0:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return

            # Read and filter lines
            with open(sudoers_file, "r") as f:
                lines = f.readlines()

            with open(sudoers_file, "w") as f:
                for line in lines:
                    if user not in line:
                        f.write(line)
                              
            # In case of FortiCloud Instances, extra entry is present in /etc/sudoers.d/90-cloud-init-users
            if flag_is_forticloud_enterprise:
                f_cloud_init_sudoers = "/etc/sudoers.d/90-cloud-init-users"
                cmd = f"rm -f {f_cloud_init_sudoers}"
                result = self.cmd_line_utilities.execute_cmd(cmd)
                if result['return_code'] != 0:
                    self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                    return

            f_sudoersd_csadmin = "/etc/sudoers.d/csadmin"
            # check if file exists
            if os.path.isfile(f_sudoersd_csadmin):
                cmd = f"rm -f {f_sudoersd_csadmin}"
                result = self.cmd_line_utilities.execute_cmd(cmd)

                if result["return_code"] != 0:
                    print(f"failed to remove {f_sudoersd_csadmin}")
                    self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                    return

            # Verify syntax of /etc/sudoers file after execution
            cmd = "visudo -c"
            result = self.cmd_line_utilities.execute_cmd(cmd)
            if result['return_code'] != 0:
                print(result['std_err'])
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
                return
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])


            print(f"Removed all lines containing user '{user}' from {sudoers_file}. Backup saved at {backup_file}.")

            
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred while removing passwordless sudo for csadmin user. Refer logs at '{LOG_FILE}'"
            )