""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
Copyright end """

import sys
import os
import yaml

from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

try:
    sys.path.append("/opt/cyops-auth")
    from utilities.config import config
    from utilities.csconstants import NODE_HARDWARE_KEY
    from handlerworkers.cluster import get_cluster_nodes
    from utilities.ha.common_utils import get_dash_string
    from utilities.ha.node import is_primary, get_other_cluster_node_hostnames
except ImportError:
    pass

try:
    sys.path.append(os.path.abspath("/opt/cyops/scripts/.lib"))
    from PasswordModule import decrypt
except ImportError:
    pass


TASK_STATUS = {"DONE": "DONE", "FAILED": "FAILED", "SKIPPED": "SKIPPED"}
TASK_LOG_STATUS = {"STARTED": "STARTED", "COMPLETED": "COMPLETED"}
TEXT_COLOR = {'GREEN': '\033[92m', 'RED': '\033[91m', 'YELLOW': '\033[93m', 'RESET': '\033[0m'}
TEXT_DECORATION = {'BLINK': '\033[5m', 'BOLD': '\033[1m', 'RESET': '\033[0m'}
TASK_STATUS_MSG = "Update envc table"
CONFIG_FILES_YAML = '/opt/cyops/configs/database/db_config.yml'
PSQL_SEARCH_KEY = 'postgres'
PG_PASS_KEY ='pg_password'
PG_HOST_KEY = 'pg_host'
PG_USER_KEY = 'pg_user'
PG_PORT_KEY = 'pg_port'
PG_EXTERNAL = 'pg_external'


class UpdateEnvcTask(Tasks):

    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(self.current_version)
        self.report_path = REPORT_FILE.format(self.current_version)
        self.psql_config = None

    @property
    def tags(self) -> str:
        return 'post-upgrade'

    def get_description(self) -> str:
        return "Set last_attempt value to null in envc table."

    def is_supported(self) -> bool:
        try:
            current_version = int(self.current_version.replace('.', ''))
            target_upgrade_version = int(self.target_upgrade_version.replace('.', ''))
            return target_upgrade_version >= current_version and self.is_enterprise()
        except ValueError as e:
            self.logger.error(f"Version parsing error: {e}")
            return False

    def execute(self):
        self.add_banner_in_log_file(TASK_STATUS_MSG, TASK_LOG_STATUS["STARTED"])
        query = f"UPDATE envc SET last_attempt = null where hardware_key = '{NODE_HARDWARE_KEY}';"
        result = self.run_db_query(query)
        self.logger.info(f"Run the query for update envc task: {result}")
        self.add_banner_in_log_file(TASK_STATUS_MSG, TASK_LOG_STATUS["COMPLETED"])

    def _get_pgsql_config(self):
        with open(CONFIG_FILES_YAML, 'r') as db_config:
            config = {}
            config.update(yaml.safe_load(db_config))
        self.psql_config = config.get(PSQL_SEARCH_KEY)

    def run_db_query(self, query):
        if self.psql_config is None:
            self._get_pgsql_config()

        s_pg_pass = decrypt(self.psql_config.get(PG_PASS_KEY))
        if not s_pg_pass.startswith('Password:'):
            raise Exception('Password decryption failed')
        s_pg_pass = s_pg_pass[len('Password:'):]
        s_pg_user = self.psql_config.get(PG_USER_KEY)
        s_pg_host = self.psql_config.get(PG_HOST_KEY)
        i_pg_port = self.psql_config.get(PG_PORT_KEY)

        s_pg_db_name = "das"
        cmd = f"env PGPASSWORD=\"{s_pg_pass}\" psql -h {s_pg_host} -p {i_pg_port} -U {s_pg_user} -d {s_pg_db_name} -c \"{query}\""
        result = self.cmd_line_utilities.execute_cmd(cmd, True)
        return result

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        msg = "    " + msg
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        elif status == TASK_STATUS["SKIPPED"]:
            color = TEXT_COLOR["YELLOW"]
        else:
            color = TEXT_COLOR["RED"]

        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status_text = f"{status:^{width}}"
        colored_status = f"{color}{status_text}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message, " ", colored_status)
        print(final_msg)

    def add_banner_in_log_file(self, msg: str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status, ":", msg)
        border_length = len(status_msg)
        border = '=' * border_length
        new_line_char = "\n" if status == TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        try:
            with open(LOG_FILE, 'a') as log_file:
                log_file.write(final_msg)
        except Exception as e:
            self.logger.error(f"Failed to write log file: {e}")

    def is_enterprise(self):
        step_result = self.get_step_results("pre-upgrade", "initialize")
        return step_result["flag_is_enterprise"]
