""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import sys
import subprocess
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from configparser import RawConfigParser, NoOptionError
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}
CONFIG_FILE_PATH = "/opt/cyops-workflow/sealab/sealab/config.ini"
MANAGE_PY_PATH = "/opt/cyops-workflow/sealab/manage.py"
PYTHON_ENV_PATH = "/opt/cyops-workflow/.env/bin/python"

application_config = RawConfigParser()
try:
    application_config.read(CONFIG_FILE_PATH)
except ImportError as ie:
    raise ie

class RetainDataUploadMaxSize(Tasks):
    TASK_STATUS_MSG = "Retain data upload max size."
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version > current_version

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            self._retain_data_upload_maxsize()
        
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _update_config(self, cmd):
        result = os.system(cmd)
        if result != 0:
            err_msg = "Failed to retain data upload max size in the file {}.".format(CONFIG_FILE_PATH)
            self.logger.error(f"ERROR: {err_msg}")
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(err_msg)
            sys.exit()
        msg = "Successfully retained data upload max size in the file {}.".format(CONFIG_FILE_PATH)
        self.logger.info(msg)
        self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        
    
    def _retain_data_upload_maxsize(self):
        try:
            DATA_UPLOAD_MAX_MEMORY_SIZE = application_config.getint('application', 'DATA_UPLOAD_MAX_MEMORY_SIZE')
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except (ValueError, NoOptionError)  as e:    
            DATA_UPLOAD_MAX_MEMORY_SIZE = 2621440
            cmd = (f"sed -i "f"'\\/^\\[application\\]/,/^\\[.*\\]/ s/^DATA_UPLOAD_MAX_MEMORY_SIZE.*/DATA_UPLOAD_MAX_MEMORY_SIZE: {DATA_UPLOAD_MAX_MEMORY_SIZE}/' "f"{CONFIG_FILE_PATH}")
            self._update_config(cmd)
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at retain data upload max size task. Refer logs at '{LOG_FILE}'"
            )
