""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """

import os
import re
import sys
import yaml
import shutil
import requests
import subprocess
from helpers.logger import Logger
from helpers.cmd_utils import CmdUtils
from framework.base.tasks import Tasks
from constants import REPORT_FILE, STEP_RESULT_FILE, LOG_FILE

try:
    sys.path.append(os.path.abspath("/opt/cyops/scripts/.lib"))
    from PasswordModule import decrypt
except ImportError:
    pass

EXCLUDE_LIST_SRC = "/opt/cyops-search/exclude.list"
EXCLUDE_LIST_DEST = "/etc/elasticsearch/exclude.list"
PERMISSIONS = "644"
OWNER = "root:elasticsearch"
CLOSE_CYOPS_INDICES_URL = 'https://localhost:9200/cyops_*/_close'
OPEN_CYOPS_INDICES_URL = 'https://localhost:9200/cyops_*/_open'
CYOPS_INDICES_SETTINGS_URL = 'https://localhost:9200/cyops_*/_settings'
CONFIG_FILES_YAML = '/opt/cyops/configs/database/db_config.yml'
ELASTICSEARCH_KEY = 'elasticsearch'
ES_USER = 'es_user'
ES_PASSWORD = 'secret'
ES_EXTERNAL_KEY = 'es_external'
TASK_STATUS = {"DONE":"DONE", "FAILED":"FAILED", "SKIPPED":"SKIPPED"}
TASK_LOG_STATUS = {"STARTED":"STARTED","COMPLETED":"COMPLETED"}
TEXT_COLOR = {'GREEN':'\033[92m', 'RED':'\033[91m', 'YELLOW':'\033[93m', 'RESET':'\033[0m'}
TEXT_DECORATION = {'BLINK':'\033[5m', 'BOLD':'\033[1m','RESET':'\033[0m'}

class UpdateIndicesMapping(Tasks):
    TASK_STATUS_MSG = "Update ES indices mapping"
    ELASTIC_SERVICE_STATUS = "elasticsearch"
    
    def __init__(self) -> None:
        super().__init__()
        self.logger = Logger.get_logger(__name__)
        self.cmd_line_utilities = CmdUtils()
        self.store_result_path = STEP_RESULT_FILE.format(
            self.target_upgrade_version)
        self.report_path = REPORT_FILE.format(self.target_upgrade_version)
        self.es_config = {}
        self.es_session = None

    @property
    def tags(self) -> str:
        return 'pre-upgrade'

    def get_description(self) -> str:
        return ""

    def is_supported(self) -> bool:
        current_version = int(self.current_version.replace('.', ''))
        target_upgrade_version = int(
            self.target_upgrade_version.replace('.', ''))
        return target_upgrade_version >= current_version and current_version < 765

    def execute(self):
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["STARTED"])
        step_result = self.get_step_results('pre-upgrade', 'initialize')
        flag_is_enterprise = step_result['flag_is_enterprise']
        
        if flag_is_enterprise:
            elastic_service_status = self._get_service_status(self.ELASTIC_SERVICE_STATUS)
            if elastic_service_status == 'active':
                self._copy_exclude_list()
                self._update_indices_mapping()
            else:
                self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["SKIPPED"])
                msg = f"'{self.ELASTIC_SERVICE_STATUS}' service is not up and running. Skipping update ES indices mapping."
                print(msg)
        
        self.add_banner_in_log_file(self.TASK_STATUS_MSG,TASK_LOG_STATUS["COMPLETED"])

    def validate(self) -> bool:
        return True

    def _print_status_msg(self, msg, status):
        reset = TEXT_COLOR["RESET"]
        if status == TASK_STATUS["DONE"]:
            color = TEXT_COLOR["GREEN"]
        elif status == TASK_STATUS["SKIPPED"]:
            color = TEXT_COLOR["YELLOW"]
        else:
            color = TEXT_COLOR["RED"]
        truncated_message = msg[:65] + "..." if len(msg) > 65 else msg
        width = 8
        status = f"{status:^{width}}"
        colored_status = f"{color}{status}{reset}"
        final_msg = "{:<70}{}[{}]".format(truncated_message," ",colored_status)
        print(final_msg)
        
    def add_banner_in_log_file(self, msg:str, status: str) -> None:
        status_msg = " [{:^11}] {} {} ".format(status,":",msg)
        border_length = len(status_msg)
        border = '='*border_length
        new_line_char = "\n" if status==TASK_LOG_STATUS["STARTED"] else "\n\n"
        final_msg = f"{status_msg}{new_line_char}"
        if os.path.exists(LOG_FILE):
            with open(LOG_FILE,'a') as log_file:
                log_file.write(final_msg)

    def _get_es_config(self):
        with open(CONFIG_FILES_YAML, 'r') as db_config:
            config = {}
            config.update(yaml.safe_load(db_config))
        self.es_config = config.get(ELASTICSEARCH_KEY)

    def _is_es_external(self) -> bool:
        return self.es_config.get(ES_EXTERNAL_KEY)

    def _get_session(self) -> str:
        if not self.es_config:
            raise Exception("Failed to get ES config")
        password = decrypt(self.es_config.get(ES_PASSWORD))
        if not password.startswith('Password:'):
            raise Exception('Password decryption failed')
        password = password[len('Password:'):]
        user = self.es_config.get(ES_USER)
        session = requests.Session()
        session.auth = (user, password)
        return session

    def _es_request(self, url, method, data=None):
        try:
            if method == 'PUT':
                response = self.es_session.put(url, verify=False, json=data)
            else:
                response = self.es_session.post(url, verify=False, json=data)
            if not response.ok:
                raise Exception(f"ES request failed {response.status_code}: {response.text}")
        except Exception as ex:
            err_msg = f"ERROR calling ES {method} {url}: {ex}"
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at update ES indices mapping task. Refer logs at '{LOG_FILE}'"
            )
        
    def _get_service_status(self,service_name:str)-> str:
        cmd=f"systemctl status {service_name}"
        result = self.cmd_line_utilities.execute_cmd(cmd, True)
        if result['std_out']:
            match = re.search(r"Active:\s+(\w+)", result['std_out'])
            return match.group(1)
        else:
            return ''

    def _copy_exclude_list(self):
        try:
            if not os.path.exists(EXCLUDE_LIST_DEST) and os.path.exists(EXCLUDE_LIST_SRC):
                shutil.copy(EXCLUDE_LIST_SRC, EXCLUDE_LIST_DEST)
                subprocess.run(["chmod", PERMISSIONS, EXCLUDE_LIST_DEST], check=True)
                subprocess.run(["chown", OWNER, EXCLUDE_LIST_DEST], check=True)
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at update ES indices mapping task. Refer logs at '{LOG_FILE}'"
            )

    def _update_indices_mapping(self) -> None:
        try:
            self._get_es_config()
            # In case of external ES do nothing
            if self._is_es_external():
                report_msg = "ES is external."
                self.print_txt(report_msg)
                return
            self.es_session = self._get_session()

            self._es_request(url=CLOSE_CYOPS_INDICES_URL, method='POST')
            payload = {
                "index": {
                    "analysis": {
                        "char_filter": {
                            "exclude_words_filter": {
                                "mappings_path": "/etc/elasticsearch/exclude.list"
                            }
                        }
                    }
                }
            }
            self._es_request(url=CYOPS_INDICES_SETTINGS_URL, method='PUT', data=payload)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["DONE"])
        except Exception as ex:
            err_msg = "ERROR: {}".format(ex)
            self.logger.exception(err_msg)
            self._print_status_msg(self.TASK_STATUS_MSG, TASK_STATUS["FAILED"])
            print(
                f"Exception occurred at update ES indices mapping task. Refer logs at '{LOG_FILE}'"
            )
        finally:
            if not self._is_es_external():
                self._es_request(url=OPEN_CYOPS_INDICES_URL, method='POST')
