#!/bin/bash
# Copyright start
# Copyright (C) 2008 - 2025 Fortinet Inc.
# All rights reserved.
# FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
# Copyright end
set -e
set -u
set -o pipefail
set -x

my_exit(){
    exit $1
}


version_compare() {
    local s_ver1=$1
    local s_ver2=$2
    local a_ver1=($(echo $s_ver1 | tr '.' "\n"))
    local a_ver2=($(echo $s_ver2 | tr '.' "\n"))
    local i_no_of_ele_ver1=${#a_ver1[@]}
    local i_no_of_ele_ver2=${#a_ver2[@]}
    local i_result=0
    if [ "X$i_no_of_ele_ver1" != "X$i_no_of_ele_ver2" ]; then
        echo -e "version_compare sub-routine needs identical number of elements"
        my_exit 1
    fi

    for ((i = 0; i < $i_no_of_ele_ver1; i++)); do
        if [ ${a_ver1[$i]} -eq ${a_ver2[$i]} ]; then
            continue
        fi
        if [ ${a_ver1[$i]} -gt ${a_ver2[$i]} ]; then
            #version 1 is greater
            i_result=1
        else
            #version 2 is greater
            i_result=2
        fi
        break
    done
    echo $i_result
}


check_kernel_version() {
    echo "------------------------------"
    echo "Operating System Support Check"
    echo "------------------------------"
    current_kernel_version=`rpm -q kernel | tail -1 | cut -d- -f2`
    supported_kernel_version='5.14.0'
    status=$(version_compare $supported_kernel_version $current_kernel_version)
    if [ $? -ne 0 ]; then
        echo "Failed to compare the version"
        exit 1
    fi
    if [ $status -eq 1 ]; then
        # Installed version is less than 7.0.0
        echo "Error: Found kernel version $current_kernel_version."
        echo "Version $product_version agent installer is only supported for Linux kernel versions $supported_kernel_version or higher. You must re-run the installer on a new VM with Redhat/Alma/Rocky Linux 9 as the base Operating System"
        exit 1
    else
        echo "Current Kernel version $current_kernel_version is supported for agent installation"
    fi
}

set_locale() {
  export LANG=en_US.UTF-8
  export LANGUAGE=en_US.UTF-8
  export LC_COLLATE=C
  export LC_CTYPE=en_US.UTF-8
}

install_basic_package() {
  yum install -y which \
                 tar \
                 sudo \
                 policycoreutils \
                 procps-ng
}

check_agent_helper(){
    if [[ ! -f $pwd/agent_helper.py ]]; then
        my_exit 1
    fi
}

move_file_to_integration() {
  if [[ -d $prefix ]]; then
    mv $pwd/agent_helper.py $prefix
  fi
}

disable_existing_postgres_repo() {
  # For Fix of following error
  # All matches were filtered out by modular filtering for argument: postgresql14-server
  sudo dnf -qy module disable postgresql
  yum clean all
}

add_enable_repo() {
  app_repo="[fsr-app]
name=FortiSOAR Application Repository
baseurl=https://$product_yum_server/$product_version/fortisoar/x86_64/
gpgcheck=0
enabled=1
sslverify=false"

  postgres_repo="[fsr-pgdg]
name=FortiSOAR PostgreSQL repository
baseurl=https://$product_yum_server/$product_version/third-party/postgres/pgdg$pg_version/
enabled=1
gpgcheck=1
gpgkey=https://$product_yum_server/$product_version/third-party/postgres/RPM-GPG-KEY-PGDG-$pg_version
sslverify=false"

  echo "$app_repo" >/etc/yum.repos.d/fsr-app.repo
  echo "$postgres_repo" >/etc/yum.repos.d/fsr-third-party.repo

  sudo chmod 644 /etc/yum.repos.d/fsr-app.repo
  sudo chmod 644 /etc/yum.repos.d/fsr-third-party.repo

  yum clean all
}

install_postgres() {
  if ! rpm -qa | grep -q postgresql$pg_version-server; then
    echo "======================================================================================================"
    echo "Installing Postgres Server"
    echo "======================================================================================================"
    yum install -y postgresql$pg_version-server --nogpgcheck
    if [ $? -ne 0 ]; then
      echo "Unable to install PostgreSQL, Exiting the installer"
      echo "Validate connectivity with $product_yum_server or check host entry"
      exit 1
    fi
  fi
}

configure_postgres(){
    #pg setup
    /usr/pgsql-$pg_version/bin/postgresql-$pg_version-setup initdb
    systemctl enable postgresql-$pg_version

    if [ $is_upgrade -ne 1 ]; then
      systemctl start postgresql-$pg_version.service
      # create pg user
      sudo -Hiu postgres psql -c 'DROP USER IF EXISTS cyberpgsql;'
      sudo -Hiu postgres psql -U postgres -c "CREATE USER cyberpgsql WITH SUPERUSER PASSWORD '$pg_password'"
      sudo -Hiu postgres psql -c 'DROP DATABASE IF EXISTS connectors;'
      sudo -Hiu postgres createdb connectors 'cyberpgsql'
    fi

    pg_hba='@pg_hba@'
    >/var/lib/pgsql/$pg_version/data/pg_hba.conf
    echo "$pg_hba" >/var/lib/pgsql/$pg_version/data/pg_hba.conf
    chown postgres:postgres /var/lib/pgsql/$pg_version/data/pg_hba.conf
    chmod 600 /var/lib/pgsql/$pg_version/data/pg_hba.conf

    systemctl restart postgresql-$pg_version.service
}


install_agent() {
    yum install -y $agent_rpm_name --nogpgcheck
    if [ $? -ne 0 ]; then
        exit 1
    fi
    systemctl enable $agent_rpm_name
}

prepare_pip_conf_for_integrations() {
    # setting product_yum_server as default index url on agent for cyops-integration's pip.conf
    # This is required since the latest packages present on the pypi.org may cause conflict with existing connector installation.
    # If user wants to point to pypi.org as extra-index-url for additional packages, steps will be documented.
    f_cyops_integrations_pip_conf="/opt/cyops-integrations/.env/pip.conf"
    local s_extra_index_url="extra-index-url"
    local s_index_url="index-url"
    
    echo "Setting index-url in ${f_cyops_integrations_pip_conf}"
    sed -i "s/${s_extra_index_url}/${s_index_url}/g" $f_cyops_integrations_pip_conf

    if [ $? -ne 0 ]; then
        echo "WARNING: Failed to update the ${f_cyops_integrations_pip_conf} file."
    fi
}

update_agent(){
    if rpm -qa | grep -q $agent_rpm_name; then
        echo "Current version:" $(rpm -q --qf '%{version}' $agent_rpm_name)
        if [ $is_upgrade -eq 1 ]; then
            /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 1, "status": "Pre-validation done, starting agent upgrade."}'
        fi
        yum update -y $agent_rpm_name --nogpgcheck
        if [ $? -ne 0 ]; then
            if [ $is_upgrade -eq 1 ]; then
                /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 4, "status": "Failed to upgrade the agent rpm package"}'
                /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"collect_log": true, "log_path": "/var/log/cyops"}'
            fi
            exit 1
        fi
        systemctl daemon-reload
    fi
}


create_csagent_symlink() {
  ln -sf /opt/cyops-integrations/integrations/cs_sdk.py /usr/bin/csagent
}

remove_repo() {
  rm -f /etc/yum.repos.d/fsr-app.repo
  rm -f /etc/yum.repos.d/fsr-third-party.repo
}

create_config_dir(){

  mkdir -p $d_agent_config
}

change_file_ownership() {
  # reset all file ownership to FortiSOAR user in case new files have been created
  chown -R $agent_user:$agent_group $prefix
  restorecon -R $prefix
  chown -R $agent_user:$agent_group /var/log/cyops/
  restorecon -R $prefix /var/log/cyops

  chown -R $agent_user:$agent_group $d_agent_config
  restorecon -R $d_agent_config
}

enable_agent_config_service(){
    chmod +x /root/fsr_agent/boot.sh
}
init(){
    echo ""
    prefix="/opt/cyops-integrations/"
    d_agent_config="/opt/cyops/configs/integrations/agent/"
    product_yum_server="repo.fortisoar.fortinet.com"
    fortisoar_version="7.6.5"
    product_version="$fortisoar_version"
    pg_version="16"
    agent_rpm_name="cyops-integrations-agent"
    pwd=$(pwd)
    agent_user="fortisoar"
    agent_group="fortisoar"
}

{

  init
  set_locale
  install_basic_package
  # agent_helper.py script is moved
  check_agent_helper
  move_file_to_integration

  #Install postgres
  disable_existing_postgres_repo
  add_enable_repo
  install_postgres
  # configure_postgres do in boot.sh

  # Install cyops integration agent
  install_agent
  create_csagent_symlink
  prepare_pip_conf_for_integrations

  # cleanup
  remove_repo
  create_config_dir
  change_file_ownership

  enable_agent_config_service
}