""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """
import re
import pika

from django import db
from django.db.utils import InterfaceError
from connectors.core.connector import logger
from connectors.environment import expand
from django.conf import settings
from retry import retry
from postman.utils.rmq_utility import prepare_inter_cyops_config, prepare_intra_cyops_config, create_exchange, \
    get_rmq_connection
from postman.utils.callbacks import get_callback


class Consumer:
    def __init__(self, config):
        self.config = config

    def __enter__(self):
        self.callback_map = dict()
        self.connection = self._create_connection()
        self.channel = self.connection.channel()
        # self._create_exchange(self.channel)
        return self

    def __exit__(self, *args):
        self.connection.close()

    def prepare_consumers(self, queue_config, message_received_callback):
        logger.info('Preparing Consumer for basic consume')
        self.callback_map.update({queue_config['routingKey']: message_received_callback})

        self._create_queue(self.channel, queue_config)

        self.channel.queue_bind(queue=queue_config['queueName'],
                                exchange=queue_config['exchangeName'],
                                routing_key=queue_config['routingKey'])

        self.channel.basic_consume(on_message_callback=self._consume_message, queue=queue_config['queueName'])

    def start_consuming(self):
        logger.info('Starting consumer')
        self.channel.start_consuming()

    def stop_consuming(self):
        logger.info('Stopping requested consumer')
        try:
            self.channel.stop_consuming()
            logger.info('Stopped requested consumer successfully')
        except Exception as e:
            logger.error('Error while stopping consumer %s', str(e))

    def _create_exchange(self, channel):
        logger.info('Creating exchange: %s', self.config['exchangeOptions'])
        exchange_options = self.config['exchangeOptions']
        channel.exchange_declare(exchange=self.config['exchangeName'],
                                 exchange_type=self.config['exchangeType'],
                                 passive=exchange_options['passive'],
                                 durable=exchange_options['durable'],
                                 auto_delete=exchange_options['autoDelete'],
                                 internal=exchange_options['internal'])

    def _create_queue(self, channel, queue_config):
        logger.info('Creating queue: %s', queue_config['queueName'])
        queue_options = queue_config['queueOptions']
        channel.queue_declare(queue=queue_config['queueName'],
                              passive=queue_options['passive'],
                              durable=queue_options['durable'],
                              exclusive=queue_options['exclusive'],
                              auto_delete=queue_options['autoDelete'])

    def _create_connection(self):
        logger.info('Creating connection')
        return get_rmq_connection(**{'config': self.config})

    def _consume_message(self, channel, method, properties, body):
        logger.info('Consuming message and calling desired callback')
        callback = self.callback_map.get(method.routing_key)
        if not callback:
            for key, value in self.callback_map.items():
                if re.match(key, method.routing_key):
                    callback = value
                    break
        _retry = 2  # Max attempt in any scenario - Two times
        while _retry > 0:
            try:
                callback(**{'message': body, 'method': method, 'properties': properties})
            except InterfaceError:
                # When data base unexpectedly closes connection due to it's server restart or timeout django does not
                # handle it gracefully and django assumes that connection is never closed and when we access it raises
                # interface error exception as connection already closed.
                logger.warn('Django database connection closed detected retrying.')
                db.connection.close()
                _retry -= 1
                continue
            except Exception as e:
                logger.exception('Exception Trace for callback function %s: ', str(e))
            _retry = 0
        channel.basic_ack(delivery_tag=method.delivery_tag)


@retry(delay=settings.ROUTER_RETRY_DELAY, logger=logger)
def start_consumer(agent_id, *args, **kwargs):
    if agent_id == 'self':
        config = prepare_intra_cyops_config(agent_id)
    else:
        config = prepare_inter_cyops_config(agent_id)
    create_exchange(config, settings.SELF_ID)

    with Consumer(config) as consumer:
        logger.info('Starting consumer for APP %s', config.get('id'))
        for each_queue in config.get('rabbit', {}).get('queues'):
            each_queue = {key: expand({'self_id': settings.SELF_ID}, value) for key, value in each_queue.items()}
            message_received_callback = get_callback(each_queue)
            queue_config = {}
            queue_config.update(each_queue)
            consumer.prepare_consumers(queue_config, message_received_callback)
        consumer.start_consuming()


if __name__ == '__main__':
    pass
