#!/bin/bash
# Copyright start
# Copyright (C) 2008 - 2025 Fortinet Inc.
# All rights reserved.
# FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
# Copyright end


my_exit(){
    exit $1
}

load_env_vars(){
    source /opt/cyops/agent-configs/agent_metadata.env
    agent_shared_key=$(cat "/opt/cyops/agent-configs/agent_shared_key")
    agent_config=$(cat "/opt/cyops/agent-configs/agent_config.yaml")
    ca_cert=$(cat "/opt/cyops/agent-configs/ca_cert.crt")
    client_cert=$(cat "/opt/cyops/agent-configs/client_cert.crt")
    client_key=$(cat "/opt/cyops/agent-configs/client_key.pem")
    connectors=$(cat "/opt/cyops/agent-configs/connectors.json")
    pg_hba=$(cat "/opt/cyops/agent-configs/pg_hba.conf")

    agent_version_info="$agent_id-$(rpm -q --queryformat '%{NAME}-%{VERSION}' $agent_rpm_name)"
    
}

check_env_vars() {
    local required_vars=(
        agent_id
        pg_password
        product_yum_server
        secure_message_exchange_host
        pg_hba
        agent_config
        ca_cert
        connectors
    )

    local optional_vars=(
        client_cert
        client_key
    )

    for var in "${required_vars[@]}"; do
        if [ -z "${!var}" ]; then
            echo "ERROR: Missing required environment variable: $var"
            my_exit 1
        fi
    done

    for var in "${optional_vars[@]}"; do
        if [ -z "${!var}" ]; then
            echo "ERROR: Missing optional environment variable: $var"
        fi
    done
}

import_enc_key() {
  if [ -n "$agent_shared_key" ] && [ -s "$f_key_file" ]; then
    resp=$($python_cmd "$f_manage_password" --import-key "$agent_shared_key")
    if [ $? -ne 0 ]; then
      echo "Failed to import shared encryption key, exiting"
      my_exit 1
    fi
    echo "$resp"
  else
    echo "Skipped to import shared encryption key: missing user input or invalid key file"
  fi
}

generate_enc_key() {
  resp=$($python_cmd "$f_manage_password" --generate-key --type active)
  code=$?
  echo "$resp"
  if [ $code -ne 0 ] && [ $code -ne 2 ]; then
    my_exit 1
  fi
}

check_if_upgrade() {
  is_upgrade=0
  is_container_restart=0
  # check if volume flags directory has flag
  if [ -f "$d_flag_agent/$f_flag_agent_installed" ]; then
    if [[ "$agent_version_info" == "$(cat "$d_flag_agent/$f_flag_agent_installed")" ]]; then
        echo "agent rpm version is same as before, denoting container was restarted."
        is_container_restart=1
        return
    else
        echo "Values are different"
    fi
    is_upgrade=1
    old_agent_id=$(/opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"get_config_key": true, "key_name": "cyops.instance.agentId"}')
    echo "INFO: Existing agent installation with id : $old_agent_id found. "
  fi
}

pre_validate_script() {

  if ! curl -I --silent --connect-timeout 10 https://$product_yum_server -k >/dev/null; then
    echo "======================================================================================================"
    echo "Please ensure connectivity to $product_yum_server for agent installation"
    echo "======================================================================================================"
    if [ $is_upgrade -eq 1 ]; then
      /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": -1, "status": "Unable to reach FortiSOAR yum server"}'
    fi
    exit 1
  fi

  if ! getent hosts $secure_message_exchange_host; then
    echo "=================================================================================================="
    echo "The Secure Message Exchange server $secure_message_exchange_host is not resolvable from this host."
    echo "It is recommended to establish the connectivity to the router before running the agent installer."
    echo "=================================================================================================="
    exit 1
  fi

  if [ $is_upgrade -eq 1 ]; then
    /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 1, "status": "Verified connectivity with FortiSOAR server"}'
  fi
}


configure_postgres(){
    #pg setup
    /usr/pgsql-$pg_version/bin/postgresql-$pg_version-setup initdb
    systemctl enable postgresql-$pg_version

    if [ $is_upgrade -ne 1 ] && [ $is_container_restart -ne 1 ]; then
      systemctl start postgresql-$pg_version.service
      # create pg user
      sudo -Hiu postgres psql -c 'DROP USER IF EXISTS cyberpgsql;'
      sudo -Hiu postgres psql -U postgres -c "CREATE USER cyberpgsql WITH SUPERUSER PASSWORD '$pg_password'"
      sudo -Hiu postgres psql -c 'DROP DATABASE IF EXISTS connectors;'
      sudo -Hiu postgres createdb connectors 'cyberpgsql'
    fi

    # pg_hba='@pg_hba@'
    # >/var/lib/pgsql/$pg_version/data/pg_hba.conf
    if [ $is_container_restart -ne 1 ]; then
        echo "$pg_hba" >/var/lib/pgsql/$pg_version/data/pg_hba.conf
        chown postgres:postgres /var/lib/pgsql/$pg_version/data/pg_hba.conf
        chmod 600 /var/lib/pgsql/$pg_version/data/pg_hba.conf
    fi

    systemctl restart postgresql-$pg_version.service
}

write_config_file() {

  echo "$agent_config" >/opt/cyops/configs/integrations/agent/agent_config.yml
  echo "$ca_cert" >/opt/cyops/configs/integrations/agent/ca_cert.pem
  echo "$client_cert" >/opt/cyops/configs/integrations/agent/client-cert.pem
  echo "$client_key" >/opt/cyops/configs/integrations/agent/client-key.pem

  echo "$connectors" >/opt/cyops-integrations/integrations/connectors.json
}

databse_migrate() {
  if [ $is_container_restart -eq 1 ]; then
    echo "Skipping database migration as it is container restart."
    return
  fi

  if [ $is_upgrade -eq 1 ]; then
    echo "Notifying master: starting database migrations"
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 2, "status": "Starting database migrations"}'
  fi

  # run migration
  cd $prefix/integrations
  sudo -u $agent_user $prefix/.env/bin/python3 manage.py migrate

  if [ $is_upgrade -eq 1 ]; then
    echo "Notifying master: completed database migrations"
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 2, "status": "Completed database migration"}'
  fi
}

create_cache_table() {
  # create cache table
  cd $prefix/integrations
  sudo -u $agent_user $prefix/.env/bin/python3 manage.py createcachetable

}

restart_service() {
  # restart service
  echo "restarting services"
  if [ ! "$1" = "skip_postgres_restart" ]; then
    echo "Skipping PostgreSQL restart on subsequent restart request from installer"
    systemctl restart postgresql-$pg_version
  fi
  systemctl restart $agent_rpm_name
  if [ $? -ne 0 ]; then
    if [ $is_upgrade -eq 1 ]; then
      sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 4, "status": "Agent service failed to start"}'
      sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"collect_log": true, "log_path": "/var/log/cyops"}'
    fi
  fi
}


install_gcc() {
  rpm -q gcc >/dev/null
  if [ $? -ne 0 ]; then
    echo "======================================================================================================"
    echo "attempting to install gcc"
    echo "======================================================================================================"
    yum install -y gcc >/dev/null 2>&1

    if [ 0 -ne $? ]; then
      echo "======================================================================================================"
      echo "Failed to install GCC rpm. Some of the FortiSOAR connectors may have a dependency on gcc."
      echo "It is recommended to install gcc by enabling the CentOS yum repositories on this instance and running 'yum install gcc -y'"
      echo "======================================================================================================"
    else
      echo "successfully installed gcc"
    fi
  fi
}

install_update_connectors() {
  cd /opt/cyops-integrations
  if [ $is_upgrade -eq 1 ]; then
    echo "Notifying master: updated utilities connector"
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 2, "status": "Updating utilities connector"}'
  fi
  echo "Installing specified connectors"
  sudo -u $agent_user /opt/cyops-integrations/.env/bin/python /opt/cyops-integrations/integrations/manage.py install_connector
}

install_connector_dep() {
  echo "installing integration pip packages"
  for requirements in $(find /opt/cyops/configs/integrations/connectors/ -name requirements.txt); do sudo -u fortisoar /opt/cyops-integrations/.env/bin/pip install -r $requirements; done
}

recreate_symlinks_for_system_connectors() {
  if [ $is_upgrade -ne 1 ] && [ $is_container_restart -ne 1 ]; then
      # Skip recreating symlinks for system connectors if the agent container is created for the first time
      return
  fi
  echo "recreating symlinks for installed system connectors"
  sudo -u $agent_user /opt/cyops-integrations/.env/bin/python /opt/cyops-integrations/integrations/manage.py recreate_symlink_for_system_connectors
  echo "Done"
}

collect_logs() {
  echo "Collecting logs" 
  if [ $is_upgrade -eq 1 ]; then
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 3, "status": "Agent upgraded successfully!"}'
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"collect_log": true, "log_path": "/var/log/cyops"}'
  fi
  touch $d_flag_agent/$f_flag_agent_installed
  echo "$agent_version_info" > $d_flag_agent/$f_flag_agent_installed
}


init(){
    echo ""
    prefix="/opt/cyops-integrations/"
    d_flag_agent="/root/.flags"
    f_flag_agent_installed="FLAG_AGENT_INSTALLED"
    python_cmd='/opt/cyops-integrations/.env/bin/python3'
    f_key_file="/opt/cyops/configs/keys/PASSWORD_ENCRYPTION_KEY"
    f_manage_password="/opt/cyops/scripts/manage_passwords.py"
    pg_version="16"
    agent_rpm_name="cyops-integrations-agent"
    pwd=$(pwd)
    agent_user="fortisoar"
    agent_group="fortisoar"
}

# Logger config
f_script_name="agent-boot"
time_stamp=$(date +"%F"-"%s")
mkdir -p /var/log/cyops
f_log="/var/log/cyops/${f_script_name}-${time_stamp}.log"
touch $f_log
f_file_descriptor_3="/dev/fd/3"
exec 3>&1 1>>${f_log} 2>&1

#Main

{

    init

    load_env_vars

    check_env_vars

    generate_enc_key

    import_enc_key

    check_if_upgrade

    pre_validate_script

    configure_postgres

    write_config_file

    databse_migrate

    create_cache_table
    restart_service

    # Install connectors and connector dependencies
    install_gcc
    install_update_connectors
    install_connector_dep
    recreate_symlinks_for_system_connectors

    collect_logs
    restart_service skip_postgres_restart

    echo "Agent Deployed Successfully"
    
} | awk '{ print strftime("[%Y-%m-%d %H:%M:%S]"), $0; fflush(); }' | tee $f_file_descriptor_3

exit 0