""" Copyright start
  Copyright (C) 2008 - 2021 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """
import pika
import ssl
import logging
from fsr_utilities.broker_core.abstract_broker import Broker

logger = logging.getLogger(__name__)


class Rmq(Broker):
    def __init__(self, config):
        self.config = config
        self.connection = self._create_connection(self.config)
        self.channel = self.connection.channel()
        self.callback_map = {}

    # Enter and exit is for context manager capabilities for object of this class
    def __enter__(self):
        return self

    def __exit__(self, *args):
        self._close_connection()

    # a destructor to safely close the all connection before object is deleted
    def __del__(self):
        self._close_connection()

    def _close_connection(self):
        if self.channel and self.channel.is_open: self.channel.close()
        if self.connection and self.connection.is_open: self.connection.close()

    def _create_connection(self, config):
        try:
            credentials = pika.PlainCredentials(config['userName'], config['password'])
            heartbeat = 60
            retry_delay = 2
            connection_attempts = 5
            params = {
                'host': config['host'],
                'port': config['port'],
                'virtual_host': config['virtualHost'],
                'credentials': credentials,
                'heartbeat': heartbeat,
                'retry_delay': retry_delay,
                'connection_attempts': connection_attempts,
            }

            if config['ssl']:
                context = ssl.create_default_context(cafile=config.get('server_cert_file'))
                ssl_options = pika.SSLOptions(context, server_hostname=config.get('sni'))
                params.update({'ssl_options': ssl_options})

            parameters = pika.ConnectionParameters(**params)
            connection = pika.BlockingConnection(parameters)

        except Exception as e:
            logger.exception('message')
            raise e
        return connection

    def create_queue(self, queue_config):
        logger.info('Creating queue: %s', queue_config['queueName'])
        queue_options = queue_config['queueOptions']
        self.channel.queue_declare(queue=queue_config['queueName'],
                                   passive=queue_options['passive'],
                                   durable=queue_options['durable'],
                                   exclusive=queue_options['exclusive'],
                                   auto_delete=queue_options['autoDelete'])

    def create_exchange(self, exchange_config):
        logger.info('Creating exchange: %s', exchange_config['exchangeName'])
        exchange_options = exchange_config['exchangeOptions']
        self.channel.exchange_declare(exchange=exchange_config['exchangeName'],
                                      exchange_type=exchange_config['exchangeType'],
                                      passive=exchange_options['passive'],
                                      durable=exchange_options['durable'],
                                      auto_delete=exchange_options['autoDelete'],
                                      internal=exchange_options['internal'])

    def publish(self, message, publish_config):
        try:
            properties_dict = {
                'delivery_mode': publish_config.get('delivery_mode', 2),
                'content_type': publish_config.get('content_type', 'application/json')
            }
            properties = pika.BasicProperties(**properties_dict)
            self.channel.basic_publish(exchange=publish_config['exchangeName'],
                                       routing_key=publish_config['routingKey'],
                                       body=message,
                                       properties=properties
                                       )
            logger.info("Published message Successfully")
        except Exception as e:
            logger.error('Error while publish %s', str(e))
            raise e

    def start_consumer(self):
        pass

    def stop_consumer(self):
        pass

    def _prepare_consumers(self):
        pass

    def _consume_message(self, channel, method, properties, body):
        pass
