#!/bin/bash
# Copyright start
# Copyright (C) 2008 - 2025 Fortinet Inc.
# All rights reserved.
# FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
# Copyright end

my_exit(){
    exit $1
}

## Validation functions
# The check_root() function check the logged in user is root or not.
check_root() {
    if [ $EUID -ne 0 ]; then
        echo "ERROR: You should have "'"root"'" access to execute the $0 file."
        my_exit 1
    fi
}

check_docker_installed_and_running() {
    if ! command -v docker &>/dev/null; then
        echo "ERROR: Docker is not installed."
        my_exit 1
    fi

    if ! docker compose version &>/dev/null; then
        echo "ERROR: Docker compose plugin is not installed."
        my_exit 1
    fi

    if ! systemctl is-active --quiet docker; then
        echo "ERROR: Docker service is not running."
        my_exit 1
    fi

    echo "Docker is installed and running."
}

check_system_resources() {
    required_cpu=2
    required_mem_mb=4096

    available_cpu=$(nproc)
    allocated_memory_bytes=$(lsmem --summary=only -b \
                                    | grep "Total online memory:" \
                                    | awk -F': ' '{print $2}' \
                                    | tr -d ' ')
    available_mem_mb=$(awk "BEGIN {print int($allocated_memory_bytes / (1024 * 1024))}")


    if (( available_cpu < required_cpu )); then
        echo "ERROR: Not enough CPU cores. Required: $required_cpu, Available: $available_cpu"
        my_exit 1
    fi

    if (( available_mem_mb < required_mem_mb )); then
        echo "ERROR: Not enough memory. Required: ${required_mem_mb}MB, Available: ${available_mem_mb}MB"
        my_exit 1
    fi

    echo "System has sufficient CPU and memory."
}

check_connectivity() {
    if ! curl -I --silent --connect-timeout 10 https://$product_yum_server -k >/dev/null; then
        echo "======================================================================================================"
        echo "ERROR: Please ensure connectivity to $product_yum_server for agent installation"
        echo "======================================================================================================"
       
        my_exit 1
    fi

    populate_extra_hosts_from_etc_hosts

    if ! getent hosts $secure_message_exchange_host; then
        echo "=================================================================================================="
        echo "The Secure Message Exchange server $secure_message_exchange_host is not resolvable from this host."
        echo "It is required to establish the connectivity to the router before running the agent installer."
        echo "=================================================================================================="

        populate_extra_hosts

        if [ $? -ne 0 ]; then 
            echo "ERROR: Failed to add extra_hosts entry for  $secure_message_exchange_host to docker compose file."
            echo "       Exiting the installer"
        fi
    fi 
    if [ $flag_set_extra_hosts -eq 0 ]; then 
        # Removing extra hosts entry from docker compose
        sed -i 's|extra_hosts:|#extra_hosts:|g' "$composefile"       
    fi
    sed -i 's|- "#extra_hosts#"|#- "#extra_hosts#"|g' "$composefile"    
}

check_docker_volume_space() {
    required_kb=$((10 * 1024 * 1024))  # 10 GB in KB

    available_kb=$(df --output=avail /var/lib/docker/volumes 2>/dev/null | tail -n 1)

    if [[ -z "$available_kb" ]]; then
        echo "ERROR: Could not check /var/lib/docker/volumes. Path missing or permission denied."
        my_exit
    fi

    if (( available_kb < required_kb )); then
        echo "ERROR: Not enough space in /var/lib/docker/volumes. Required: 10GB+, Available: $((available_kb / 1024))MB"
        my_exit
    fi

    echo "/var/lib/docker/volumes has enough space."
}

## Utility function

prompt_shared_enc_key() {
  enc=""
  # Checking if shared key exist in the password file(if exist) inside container(if exist) using the identifier
  if [ -n "$f_key_file" ] && docker exec -i "$agent_container_name" sh -c "base64 -d \"$f_key_file\" | grep -q \"$enc_identifier\"" 2>/dev/null; then
    echo "Skipping shared encryption key prompt"
  # Prompting the user for key
  else
      while true; do
        printf "Enter the shared encryption key (cannot be empty): " > /dev/tty
        read -r enc < /dev/tty
        enc="$(echo -e "$enc" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')"  # trim

        if [ -n "$enc" ]; then
          hashed_enc=$(echo -n "$enc" | sha256sum | awk '{print $1}')
          if [ "$enc_hashed" = "$hashed_enc" ]; then
            break
          else
            echo "Encryption key didn't match. Please try again." > /dev/tty
          fi
        fi
      done
  fi
  sed -i "s|#enc_key#|$enc|g" "$composefile"
}

populate_extra_hosts() {
    local secure_message_exchange_ip=""
    local ip_regex="^([0-9]{1,3}\.){3}[0-9]{1,3}$"

    echo "Enter the IP address for host: $secure_message_exchange_host"
    read secure_message_exchange_ip

    # Validate IP format
    if [[ ! "$secure_message_exchange_ip" =~ $ip_regex ]]; then
        echo "Invalid IP format. Please provide a valid IPv4 address (e.g., 192.168.1.10)."
        return 1 
    fi

    # Validate each octet is <= 255
    IFS='.' read -r o1 o2 o3 o4 <<< "$secure_message_exchange_ip"
    for octet in $o1 $o2 $o3 $o4; do
        if (( octet < 0 || octet > 255 )); then
            echo "ERROR: Invalid IP address: Octet '$octet' out of range (0-255)."
            my_exit 1
        fi
    done

    sed -i "s|#extra_hosts#|$secure_message_exchange_host=$secure_message_exchange_ip|g" "$composefile"

    flag_set_extra_hosts=1

    return 0
}

populate_extra_hosts_from_etc_hosts() {
    echo "Populating extra hosts from /etc/hosts"
    local s_hosts=""
    while read -r ip name _; do
        if [[ "$ip" == "127.0.0.1" || "$ip" == "::1" || "$name" == "localhost" ]]; then
            continue
        fi
        s_ip_entry="      - \"$name=$ip\""
        s_hosts="$s_hosts$s_ip_entry\n"
    done < <(getent hosts)

    if [[ -z "$s_hosts" ]]; then
        echo "No valid hosts found via getent"
        echo "Done"
        return 1
    fi
    flag_set_extra_hosts=1
    sed -i "/^[[:space:]]*extra_hosts:/a\\
$s_hosts" "$composefile"

    echo "Done"

}

create_docker_compose_file() {
    local dc_content='@dockercompose@'
    printf "$dc_content" > "$composefile"
}

init() {
    product_yum_server="@product_yum_server@"
    product_version="@product_version@"
    product_build_number="@product_build_number@"
    secure_message_exchange_host="@secure_message_exchange_host@"
    agent_project_name="@project_name@"
    agent_name="@agent_name@"
    agent_project_name="@project_name@"
    agent_container_name="${agent_name}_agent"
    enc_identifier="@enc_identifier@"
    enc_hashed="@enc_hashed@"
    f_key_file="/opt/cyops/configs/keys/PASSWORD_ENCRYPTION_KEY"
    f_fortisoar_agent_docker_image="fortisoar_agent_docker-$product_version-$product_build_number.tgz"

    flag_set_extra_hosts=0
}

pre_install_checks(){
    echo "Running pre-install checks"
    check_root
    check_docker_installed_and_running
    check_system_resources
    check_docker_volume_space
}

install(){

    composefile="docker-compose-$agent_name.yaml"
    create_docker_compose_file

    check_connectivity

    prompt_shared_enc_key

    curl --fail -O https://$product_yum_server/$product_version/$f_fortisoar_agent_docker_image  >/dev/null 2>&1

    if [ $? -ne 0 ]; then 
        echo "ERROR: Unable to download fortisoar agent docker image from $product_yum_server"
        my_exit 1
    fi 

    docker load --input ./$f_fortisoar_agent_docker_image
    if [ $? -ne 0 ]; then 
        echo "ERROR: Unable to load fortisoar agent docker image"
        my_exit 1
    fi 

    docker compose --project-name agent_$agent_project_name -f $composefile up --detach
    if [ $? -ne 0 ]; then 
        echo "ERROR: Unable to create fortisoar agent docker container"
        my_exit 1
    fi 

    rm -f ./$f_fortisoar_agent_docker_image

    echo "Successfully started the FortiSOAR agent container as Docker Compose project: agent_$agent_project_name"
    echo "Connector installation may take a while. You can monitor the agent configuration logs in: /var/log/cyops/"
    
}

main() {

    init

    pre_install_checks

    install

}

main