#!/bin/bash
# Copyright start
# Copyright (C) 2008 - 2025 Fortinet Inc.
# All rights reserved.
# FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
# Copyright end

agent_user="fortisoar"
agent_group="fortisoar"
prefix="/opt/cyops-integrations/"
f_cmd_psql="/usr/bin/psql"
pg_version=16
pg_hba_file="/var/lib/pgsql/$pg_version/data/pg_hba.conf"
product_name="@product_name@"
product_version="@product_version@"
agent_id="@agent_id@"
agent_rpm_name="cyops-integrations-agent"
product_yum_server="repo.fortisoar.fortinet.com"
secure_message_exchange_host="@secure_message_exchange_hostname@"
pwd=$(pwd)
config_update="@config_update@"
s_user_input="@user_input@"
s_identifier="@identifier_str@"
s_hashed="@hashed_str@"
d_keys="/opt/cyops/configs/keys/"
f_key_file="/opt/cyops/configs/keys/PASSWORD_ENCRYPTION_KEY"
f_manage_password='/opt/cyops/scripts/manage_passwords.py'
f_conf_util='/opt/cyops/scripts/confUtil.py'
f_config='/opt/cyops/configs/integrations/agent/agent_config.yml'
python_cmd='/opt/cyops-integrations/.env/bin/python3'

# Code -1 Pre validation failed
# Code 1 means pre-installation
# Code 2 means post installation
# Code 3 means agent upgrade successful
# Code 4 means upgrade failed or service restart failed
# Code 5 means agent config update is in progress
# Code 6 means agent config update successful
# Code 7 means agent config update failed

install_basic_package() {
  yum install -y which tar sudo
}

remove_old_constants_for_fortitip(){
    lower_product_name=$(echo "$product_name" | tr '[:upper:]' '[:lower:]')
  

    if [ "X$lower_product_name" = "Xfortitip" ]; then
        echo "Removing Old constant for $lower_product_name agent"
        rm -f /opt/cyops-integrations/integrations/integrations/OConstant.so
        return
    fi

    echo "Skipping Old constant removal for $lower_product_name agent"
}

generate_enc_key() {
  resp=$($python_cmd "$f_manage_password" --generate-key --type active)
  code=$?
  echo "$resp"
  if [ $code -ne 0 ] && [ $code -ne 2 ]; then
    exit 1
  fi
}


prompt_shared_enc() {
  # Checking if key exist in file using the identifier
  if [ -s "$f_key_file" ] && base64 -d "$f_key_file" 2>/dev/null | grep -q "$s_identifier"; then
    echo "Skipping shared encryption key prompt"
  # Prompting the user for key or using the provided in case of remote upgrade
  else
    if [ -z "$s_user_input" ]; then
      local enc=""

      while true; do
        printf "Enter the shared encryption key (cannot be empty): " > /dev/tty
        read -r enc < /dev/tty
        enc="$(echo -e "$enc" | sed -e 's/^[[:space:]]*//' -e 's/[[:space:]]*$//')"  # trim

        if [ -n "$enc" ]; then
          hashed_enc=$(echo -n "$enc" | sha256sum | awk '{print $1}')
          if [ "$s_hashed" = "$hashed_enc" ]; then
            break
          else
            echo "Encryption key didn't match. Please try again." > /dev/tty
          fi
        fi
      done
      s_user_input="$enc"
    fi
  fi
}

import_shared_enc(){
  if [ -n "$s_user_input" ] && [ -s "$f_key_file" ]; then
    resp=$($python_cmd "$f_manage_password" --import-key "$s_user_input")
    if [ $? -ne 0 ]; then
      echo "Failed to import shared encryption key, exiting"
      exit 1
    fi
    echo "$resp"
  else
    echo "Skipped to import shared encryption key: missing user input or invalid key file"
  fi
}

write_pg_password_in_config(){
  s_pg_passwd_encrypted=$(encrypt_password $pg_password)
  if [ -z "$s_pg_passwd_encrypted"]; then
    echo "Failed to encrypt pg_password."
  fi
  $python_cmd $f_conf_util -f $f_config -k postgres.pg_password -v $s_pg_passwd_encrypted
}

re_encrypt_mq_password(){
  if [ -f "$f_conf_util" ]; then
    enc_mq_password=$($python_cmd "$f_conf_util" -f "$f_config" -g cyops.rabbitmq.password)
  else
    # if upgrading from version where f_conf_util does not exist
    enc_mq_password=$(grep 'cyops.rabbitmq.password:' "$f_config" | awk '{print $2}')
  fi
  plain_mq_password=$(decrypt_password "$enc_mq_password")
  mq_passwd_re_encrypted=$(encrypt_password "$plain_mq_password")
  if [ -z "$mq_passwd_re_encrypted" ]; then
    echo "Failed to re-encrypt mq_password."
  fi
  $python_cmd $f_conf_util -f $f_config -k cyops.rabbitmq.password -v $mq_passwd_re_encrypted
}

decrypt_password() {
  local enc_password="$1"
  local text_password
  text_password=$(PYTHONPATH="/opt/cyops-integrations/integrations" "$python_cmd" -c "from integrations import password_utils; print(password_utils.manage_password('$enc_password', 'decrypt'))")
  echo "$text_password"
}

encrypt_password(){
  local text_password=$1
  enc_password=$(PYTHONPATH="/opt/cyops-integrations/integrations" "$python_cmd" -c "from integrations import password_utils; print(password_utils.manage_password('$text_password', 'encrypt'))")
  echo $enc_password
}

get_random_password() {
    # Generate 30 character random password.
    local random_password=$(date +%s | sha256sum | head -c 30; echo)
    echo "$random_password"
}

set_pg_password(){
    if [ $is_upgrade -eq 1 ]; then
      # If upgrade then get the current pg_password from agent_config.yml
      # and decrypt if it is encrypted, It will be again encrypted while saving in the agent_config.yml file.
      if [ -f "$f_conf_util" ]; then
        enc_pg_password=$($python_cmd "$f_conf_util" -f "$f_config" -g postgres.pg_password)
      else
        # if upgrading from version where f_conf_util does not exist
        enc_pg_password=$(grep 'postgres.pg_password:' "$f_config" | awk '{print $2}')
      fi
      pg_password=$(PYTHONPATH="/opt/cyops-integrations/integrations" "$python_cmd" -c "from integrations import password_utils; print(password_utils.manage_password('$enc_pg_password', 'decrypt'))")
    else
      pg_password=$(get_random_password)
    fi
    if [ -z "$pg_password" ] || [ $pg_password -eq 0 ]; then
      echo "Failed to retrieve pg_password, Exiting the installer."
      exit
    fi
}

update_secret_key(){
    echo "Updating secret keys for cyops-integration-agents"
    /opt/cyops-integrations/.env/bin/python /opt/cyops-integrations/integrations/manage.py write_secret_key
}

identify_bytes_to_skip() {
  i_bytes_to_skip=$(awk '/^__PAYLOAD_STARTS_AFTER_THIS__/ { print NR; exit 0; }' $0)
  c_first_char=$(echo $0 | cut -c 1)
  if [ "$c_first_char" = "/" ]; then
    script=$0
  else
    script=$(pwd)/$0
  fi
  export script
  export i_bytes_to_skip
}

check_if_upgrade() {
  is_upgrade=0
  if rpm -qa | grep -q $agent_rpm_name; then
    is_upgrade=1
  fi
}

extract_archive() {

  if ! which tar; then
    echo "Please ensure the tar command is installed on your machine"
    if [ $is_upgrade -eq 1 ]; then
      /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": -1, "status": "tar command not found on the FortiSOAR agent instance"}'
    fi
    exit 1
  fi

  sed "1,${i_bytes_to_skip}d" $script | tar -xzf - 2>/dev/null
  if [ $? -ne 0 ]; then
    echo "Extract archive failed, exiting"
    exit 1
  fi
}

identify_existing_agent() {
  if [ $is_upgrade -eq 1 ]; then
    old_agent_id=$(/opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"get_config_key": true, "key_name": "cyops.instance.agentId"}')
    if [ "$agent_id" != "$old_agent_id" ]; then
      echo "Found an existing installation of FortiSOAR agent with a different agent ID. The installer will exit."
      echo "Possibly you are running the installer for another agent connected to your FortiSOAR server. Check and download the correct agent installer."
      echo "Or, if you wish to reset the installation with the new agent ID, execute the following commands manually and then re-run the installer script."
      echo -e "\n\n"
      echo "rm -rf /var/lib/pgsql/$pg_version/data/"
      echo "/usr/pgsql-$pg_version/bin/postgresql-$pg_version-setup initdb"

      echo "systemctl restart postgresql-$pg_version.service"

      echo "sudo -Hiu postgres psql -c 'DROP USER IF EXISTS cyberpgsql;'"
      echo "sudo -Hiu postgres psql -U postgres -c \"CREATE USER cyberpgsql WITH SUPERUSER PASSWORD '$pg_password'\""
      echo "sudo -Hiu postgres psql -c 'DROP DATABASE IF EXISTS connectors;'"
      echo "sudo -Hiu postgres createdb connectors 'cyberpgsql'"

      echo "> /var/lib/pgsql/$pg_version/data/pg_hba.conf"
      echo "echo -e \"local   all             postgres                                md5 \n\nlocal   all             cyberpgsql                              md5 \n\nhost    all             cyberpgsql             127.0.0.1/32     md5\n\nhost    all             cyberpgsql             ::1/128          md5\" > /var/lib/pgsql/$pg_version/data/pg_hba.conf"
      echo "chown postgres:postgres /var/lib/pgsql/$pg_version/data/pg_hba.conf"
      echo "chmod 600 /var/lib/pgsql/$pg_version/data/pg_hba.conf"
      echo "systemctl restart postgresql-$pg_version.service"
      echo "Once done please update the new agent id in /opt/cyops/configs/integrations/agent/agent_config.yml under cyops.instance.agentId"
      exit 1
    fi
  fi
}

pre_validate_script() {

  if ! curl -I --silent --connect-timeout 10 https://$product_yum_server -k >/dev/null; then
    echo "======================================================================================================"
    echo "Please ensure connectivity to $product_yum_server for agent installation"
    echo "======================================================================================================"
    if [ $is_upgrade -eq 1 ]; then
      /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": -1, "status": "Unable to reach FortiSOAR yum server"}'
    fi
    exit 1
  fi

  if ! getent hosts $secure_message_exchange_host; then
    echo "=================================================================================================="
    echo "The Secure Message Exchange server $secure_message_exchange_host is not resolvable from this host."
    echo "It is recommended to establish the connectivity to the router before running the agent installer."
    echo "=================================================================================================="
    echo "Exit the installer (y/n)?"
    read -n 1 -r
    if [[ $REPLY =~ ^[Yy]$ ]]; then
      echo -e '\n'
      exit 1
    fi
  fi

  if [ $is_upgrade -eq 1 ]; then
    /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 1, "status": "Verified connectivity with FortiSOAR server"}'
  fi
}

disable_existing_postgres_repo() {
  # For Fix of following error
  # All matches were filtered out by modular filtering for argument: postgresql14-server
  sudo dnf -qy module disable postgresql
  yum clean all
}

add_enable_repo() {
  app_repo="[fsr-app]
name=FortiSOAR Application Repository
baseurl=https://$product_yum_server/$product_version/fortisoar/x86_64/
gpgcheck=0
enabled=1
sslverify=false"

  postgres_repo="[fsr-pgdg]
name=FortiSOAR PostgreSQL repository
baseurl=https://$product_yum_server/$product_version/third-party/postgres/pgdg$pg_version/
enabled=1
gpgcheck=1
gpgkey=https://$product_yum_server/$product_version/third-party/postgres/RPM-GPG-KEY-PGDG-$pg_version
sslverify=false"

  echo "$app_repo" >/etc/yum.repos.d/fsr-app.repo
  echo "$postgres_repo" >/etc/yum.repos.d/fsr-third-party.repo

  sudo chmod 644 /etc/yum.repos.d/fsr-app.repo
  sudo chmod 644 /etc/yum.repos.d/fsr-third-party.repo

  yum clean all
}

install_configure_postgres() {
  if ! rpm -qa | grep -q postgresql$pg_version-server; then
    echo "======================================================================================================"
    echo "Installing Postgres Server"
    echo "======================================================================================================"
    yum install -y postgresql$pg_version-server --nogpgcheck
    if [ $? -ne 0 ]; then
      echo "Unable to install PostgreSQL, Exiting the installer"
      echo "Validate connectivity with $product_yum_server or check host entry"
      exit 1
    fi

    #pg setup
    /usr/pgsql-$pg_version/bin/postgresql-$pg_version-setup initdb
    systemctl enable postgresql-$pg_version

    if [ $is_upgrade -ne 1 ]; then
      systemctl start postgresql-$pg_version.service
      # create pg user
      sudo -Hiu postgres psql -c 'DROP USER IF EXISTS cyberpgsql;'
      sudo -Hiu postgres psql -U postgres -c "CREATE USER cyberpgsql WITH SUPERUSER PASSWORD '$pg_password'"
      sudo -Hiu postgres psql -c 'DROP DATABASE IF EXISTS connectors;'
      sudo -Hiu postgres createdb connectors 'cyberpgsql'
    fi

    pg_hba='@pg_hba@'
    >/var/lib/pgsql/$pg_version/data/pg_hba.conf
    echo "$pg_hba" >/var/lib/pgsql/$pg_version/data/pg_hba.conf
    chown postgres:postgres /var/lib/pgsql/$pg_version/data/pg_hba.conf
    chmod 600 /var/lib/pgsql/$pg_version/data/pg_hba.conf

    systemctl restart postgresql-$pg_version.service
  fi
}

install_upgrade_agent() {
  if rpm -qa | grep -q $agent_rpm_name; then
    echo "Current version:" $(rpm -q --qf '%{version}' $agent_rpm_name)
    if [ $is_upgrade -eq 1 ]; then
      /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 1, "status": "Pre-validation done, starting agent upgrade."}'
    fi
    yum update -y $agent_rpm_name --nogpgcheck
    if [ $? -ne 0 ]; then
      if [ $is_upgrade -eq 1 ]; then
        /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 4, "status": "Failed to upgrade the agent rpm package"}'
        /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"collect_log": true, "log_path": "/var/log/cyops"}'
      fi
      exit 1
    fi
    systemctl daemon-reload
  else
    yum install -y $agent_rpm_name --nogpgcheck
    if [ $? -ne 0 ]; then
      exit 1
    fi
    systemctl enable $agent_rpm_name
  fi
}

remove_repo() {
  rm -f /etc/yum.repos.d/fsr-app.repo
  rm -f /etc/yum.repos.d/fsr-third-party.repo

  if [ $is_upgrade -eq 1 ]; then
    echo "Notifying master: removing the added yum repos"
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 1, "status": "Removing added repos"}'
  fi
}

write_config_file() {
  mkdir -p /opt/cyops/configs/integrations/agent
  agent_config='@agent_config@'
  ca_cert='@ca_cert@'
  client_cert='@client_cert@'
  client_key='@client_key@'
  connectors=$(
    cat <<EOF
@connectors@
EOF
  )

  echo "$agent_config" >/opt/cyops/configs/integrations/agent/agent_config.yml
  echo "$ca_cert" >/opt/cyops/configs/integrations/agent/ca_cert.pem
  echo "$client_cert" >/opt/cyops/configs/integrations/agent/client-cert.pem
  echo "$client_key" >/opt/cyops/configs/integrations/agent/client-key.pem
  echo "$connectors" >/opt/cyops/configs/integrations/agent/connectors.json
}

write_tmp_dir(){
  mkdir -p /opt/cyops/configs/integrations/agent/tmp
}

del_tmp_dir(){
  rm -rf /opt/cyops/configs/integrations/agent/tmp
}

write_tmp_config_file(){
  write_tmp_dir
  agent_config='@agent_config@'
  ca_cert='@ca_cert@'
  client_cert='@client_cert@'
  client_key='@client_key@'
  connectors=$(
    cat <<EOF
@connectors@
EOF
  )
  echo "$agent_config" >/opt/cyops/configs/integrations/agent/tmp/agent_config.yml
  echo "$ca_cert" >/opt/cyops/configs/integrations/agent/tmp/ca_cert.pem
  echo "$client_cert" >/opt/cyops/configs/integrations/agent/tmp/client-cert.pem
  echo "$client_key" >/opt/cyops/configs/integrations/agent/tmp/client-key.pem
  sed -i "s|\(/opt/cyops/configs/integrations/agent/\)|\1tmp/|g" /opt/cyops/configs/integrations/agent/tmp/agent_config.yml
}

can_connect_sme(){
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py "{\"code\": 1, \"status\": \"Checking if Agent can connect to SME host $(secure_message_exchange_host)\"}"
    echo "Checking if agent can connect to SME"
    sme_health=$(/opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"sme_health_check": true}')
    if [ "X$sme_health" != "XTrue" ]; then
        echo "Cannot connect to SME. Exiting update"
        sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py "{\"code\": 7, \"status\": \"FortiSOAR Agent instance cannot connect to SME host $(secure_message_exchange_host)\"}"
        exit 1
    fi
    /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py "{\"code\": 5, \"status\": \"FortiSOAR Agent instance can connect to SME host $(secure_message_exchange_host)\"}"
    echo "Agent can connect to SME"
    del_tmp_dir
}

restart_service() {
  # restart service
  echo "restarting services"
  if [ ! "$1" = "skip_postgres_restart" ]; then
    echo "Skipping PostgreSQL restart on subsequent restart request from installer"
    systemctl restart postgresql-$pg_version
  fi
  systemctl restart $agent_rpm_name
  if [ $? -ne 0 ]; then
    if [ $is_upgrade -eq 1 ]; then
      sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 4, "status": "Agent service failed to start"}'
      sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"collect_log": true, "log_path": "/var/log/cyops"}'
    fi
  fi
}

databse_migrate() {
  if [ $is_upgrade -eq 1 ]; then
    echo "Notifying master: starting database migrations"
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 2, "status": "Starting database migrations"}'
  fi

  # run migration
  cd $prefix/integrations
  sudo -u $agent_user $prefix/.env/bin/python3 manage.py migrate

  if [ $is_upgrade -eq 1 ]; then
    echo "Notifying master: completed database migrations"
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 2, "status": "Completed database migration"}'
  fi
}

create_cache_table() {
  # create cache table
  cd $prefix/integrations
  sudo -u $agent_user $prefix/.env/bin/python3 manage.py createcachetable

}

install_gcc() {
  rpm -q gcc >/dev/null
  if [ $? -ne 0 ]; then
    echo "======================================================================================================"
    echo "attempting to install gcc"
    echo "======================================================================================================"
    yum install -y gcc >/dev/null 2>&1

    if [ 0 -ne $? ]; then
      echo "======================================================================================================"
      echo "Failed to install GCC rpm. Some of the FortiSOAR connectors may have a dependency on gcc."
      echo "It is recommended to install gcc by enabling the CentOS yum repositories on this instance and running 'yum install gcc -y'"
      echo "======================================================================================================"
    else
      echo "successfully installed gcc"
    fi
  fi
}

install_update_connectors() {
  cd /opt/cyops-integrations
  if [ $is_upgrade -eq 1 ]; then
    echo "Notifying master: updated utilities connector"
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 2, "status": "Updating utilities connector"}'
  fi
  echo "Installing specified connectors"
  sudo -u $agent_user /opt/cyops-integrations/.env/bin/python /opt/cyops-integrations/integrations/manage.py install_connector --connectors_path "/opt/cyops/configs/integrations/agent/connectors.json"
}

install_connector_dep() {
  echo "installing integration pip packages"
  for requirements in $(find /opt/cyops/configs/integrations/connectors/ -name requirements.txt); do sudo -u fortisoar /opt/cyops-integrations/.env/bin/pip install -r $requirements; done
}

collect_logs() {
  echo "Agent Deployed Successfully"
  if [ $is_upgrade -eq 1 ]; then
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 3, "status": "Agent upgraded successfully!"}'
    sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"collect_log": true, "log_path": "/var/log/cyops"}'
  fi
}

post_config_update() {
  echo "Agent configuration refreshed successfully"
  sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"code": 6, "status": "Agent configuration refreshed successfully!"}'
  sudo -u $agent_user /opt/cyops-integrations/.env/bin/python $prefix/agent_helper.py '{"collect_log": true, "log_path": "/var/log/cyops"}'
}

move_file_to_integration() {
  if [[ -d $prefix ]]; then
    mv $pwd/agent_helper.py $prefix
  fi
}

change_file_ownership() {
  # reset all file ownership to FortiSOAR user in case new files have been created
  chown -R $agent_user:$agent_group $prefix
  restorecon -R $prefix
  chown -R $agent_user:$agent_group /var/log/cyops/
  restorecon -R $prefix /var/log/cyops
  chown -R $agent_user:$agent_group /opt/cyops/configs/integrations/agent
  restorecon -R /opt/cyops/configs/integrations/agent
}

set_locale() {
  export LANG=en_US.UTF-8
  export LANGUAGE=en_US.UTF-8
  export LC_COLLATE=C
  export LC_CTYPE=en_US.UTF-8
}

create_csagent_symlink() {
  ln -sf /opt/cyops-integrations/integrations/cs_sdk.py /usr/bin/csagent
}

prepare_pip_conf_for_integrations() {
    # setting product_yum_server as default index url on agent for cyops-integration's pip.conf
    # This is required since the latest packages present on the pypi.org may cause conflict with existing connector installation.
    # If user wants to point to pypi.org as extra-index-url for additional packages, steps will be documented.
    f_cyops_integrations_pip_conf="/opt/cyops-integrations/.env/pip.conf"
    local s_extra_index_url="extra-index-url"
    local s_index_url="index-url"
    
    echo "Setting index-url in ${f_cyops_integrations_pip_conf}"
    sed -i "s/${s_extra_index_url}/${s_index_url}/g" $f_cyops_integrations_pip_conf

    if [ $? -ne 0 ]; then
        echo "WARNING: Failed to update the ${f_cyops_integrations_pip_conf} file."
    fi
}

version_compare() {
    local s_ver1=$1
    local s_ver2=$2
    local a_ver1=($(echo $s_ver1 | tr '.' "\n"))
    local a_ver2=($(echo $s_ver2 | tr '.' "\n"))
    local i_no_of_ele_ver1=${#a_ver1[@]}
    local i_no_of_ele_ver2=${#a_ver2[@]}
    local i_result=0
    if [ "X$i_no_of_ele_ver1" != "X$i_no_of_ele_ver2" ]; then
        echo -e "version_compare sub-routine needs identical number of elements"
        my_exit 1
    fi

    for ((i = 0; i < $i_no_of_ele_ver1; i++)); do
        if [ ${a_ver1[$i]} -eq ${a_ver2[$i]} ]; then
            continue
        fi
        if [ ${a_ver1[$i]} -gt ${a_ver2[$i]} ]; then
            #version 1 is greater
            i_result=1
        else
            #version 2 is greater
            i_result=2
        fi
        break
    done
    echo $i_result
}

check_kernel_version() {
    echo "------------------------------"
    echo "Operating System Support Check"
    echo "------------------------------"
    current_kernel_version=`uname -r |cut -d- -f1`
    supported_kernel_version='5.14.0'
    status=$(version_compare $supported_kernel_version $current_kernel_version)
    if [ $? -ne 0 ]; then
        echo "Failed to compare the version"
        exit 1
    fi
    if [ $status -eq 1 ]; then
        # Installed version is less than 7.0.0
        echo "Error: Found kernel version $current_kernel_version."
        echo "Version $product_version agent installer is only supported for Linux kernel versions $supported_kernel_version or higher. You must re-run the installer on a new VM with Redhat/Alma/Rocky Linux 9 as the base Operating System"
        exit 1
    else
        echo "Current Kernel version $current_kernel_version is supported for agent installation"
    fi
}

copy_agent_config() {
    # Skip if not an upgrade
    if [ "$is_upgrade" -eq 0 ]; then
        return
    fi

    # Check if the config file already exists
    if [ -f "$f_config" ]; then
        return
    fi

    # Create agent directory if not exists
    local dest_dir="/opt/cyops/configs/integrations/agent"
    if [ -d "$dest_dir" ]; then
        echo "Directory $dest_dir already exists. Skipping creation."
    else
        echo "Creating directory: $dest_dir"
        mkdir "$dest_dir"
        chown "fortisoar:fortisoar" "$dest_dir"
        chmod 755 "$dest_dir"
    fi

    # Copy agent_config.yml to new path
    old_config_path="/opt/cyops-integrations/integrations/configs/agent_config.yml"
    if [ -f "$old_config_path" ]; then
        echo "Copying agent_config.yml to the new configuration path..."
        cp -a "$old_config_path" "$f_config"
    else
        # This section should not be run as $old_config_path or $f_config must exists in case of upgrade
        echo "Error: $old_config_path and $f_config does not exist."
        exit 1
    fi
}

f_script_name="agent-$product_version"
time_stamp=$(date +"%F"-"%s")
mkdir -p /var/log/cyops
f_log="/var/log/cyops/${f_script_name}-${time_stamp}.log"
touch $f_log
f_file_descriptor_3="/dev/fd/3"
exec 3>&1 1>>${f_log} 2>&1

if [ "X$config_update" = "XTrue" ]; then
  {
    check_kernel_version
    set_locale
    install_basic_package
    identify_bytes_to_skip
    check_if_upgrade
    copy_agent_config
    extract_archive
    move_file_to_integration
    pre_validate_script
    set_pg_password
    write_tmp_config_file
    can_connect_sme
    write_config_file
    write_pg_password_in_config
    change_file_ownership
    post_config_update
    restart_service skip_postgres_restart
  } | awk '{ print strftime("[%Y-%m-%d %H:%M:%S]"), $0; fflush(); }' | tee $f_file_descriptor_3
  exit 0
fi

{
  prompt_shared_enc
  check_kernel_version
  set_locale
  install_basic_package
  identify_bytes_to_skip
  check_if_upgrade
  copy_agent_config
  extract_archive
  move_file_to_integration
  set_pg_password
  identify_existing_agent
  pre_validate_script
  disable_existing_postgres_repo
  add_enable_repo
  install_configure_postgres
  install_upgrade_agent
  create_csagent_symlink
  prepare_pip_conf_for_integrations
  remove_repo
  write_config_file
  write_product_details
  remove_old_constants_for_fortitip
  generate_enc_key
  import_shared_enc
  write_pg_password_in_config
  re_encrypt_mq_password
  change_file_ownership
  databse_migrate
  create_cache_table
  update_secret_key
  restart_service
  install_gcc
  install_update_connectors
  install_connector_dep
  collect_logs
  restart_service skip_postgres_restart
} | awk '{ print strftime("[%Y-%m-%d %H:%M:%S]"), $0; fflush(); }' | tee $f_file_descriptor_3

exit 0
__PAYLOAD_STARTS_AFTER_THIS__
