""" Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end """
import io
import json
import logging
import os
import requests
import time
import uuid

from utils.config_parser import all_config
from base64 import b64encode, b64decode
from django.conf import settings
from django.core.cache import cache
from django.core.serializers.json import DjangoJSONEncoder
from urllib.parse import urlparse

logger = logging.getLogger('connectors')


def _sync_call_to_remote(data, method, __async=False):
    from postman.core.publisher import Publisher
    if not settings.MASTER_ID: _populated_Related_info()
    pub_config = {
        'exchangeName': 'dexchange.cyops.postman.{0}'.format(settings.MASTER_ID),
        'routingKey': 'route.postman.chapi.remoterequest.{0}'.format(settings.MASTER_ID)
    }
    message_json = {
        'data': data,
        'sourceId': settings.SELF_ID,
        'destinationId': settings.MASTER_ID,
        'method': method
    }
    callback_queue = 'route.postman.chapi.remoteresponse.{0}'.format(settings.SELF_ID)
    correlation_id = str(uuid.uuid4())
    publisher = Publisher(pub_config)
    if isinstance(message_json, bytes):
        message_json = message_json.decode()
    try:
        message = json.dumps(message_json)
    except:
        message = json.dumps(message_json, cls=DjangoJSONEncoder)

    publisher.publish(message, settings.MASTER_ID, correlation_id=correlation_id,
                      callback_queue=callback_queue, from_replay=True)

    if __async:
        return {'result': {'task_id': 'sample_value'}}

    timeout = time.time() + 60
    while not cache.get(correlation_id) and timeout > time.time():
        # Adding sleep as in 7.4.3 we are changing the code from
        # Checking for response in memory to database based cache, so its safe to add a sleep
        # to avoid recursive call
        time.sleep(settings.TIME_SLEEP_BETWEEN_SYNC_MQ_CALL)
        pass

    response = cache.get(correlation_id, {})
    cache.delete(correlation_id)


    if response.get('status') == 'success':
        return response
    else:
        msg = response.get('message', 'Request timeout')
        logger.error(msg)
        raise requests.exceptions.HTTPError(msg)


def make_request(url, method, body=None, __async=False, *args, **kwargs):
    if not bool(urlparse(url).netloc):
        url = settings.CRUD_HUB_URL + str(url)
    else:
        url = url
    if not body and __async: body = {'__async': __async}
    data = {
        'url': url,
        'body': body,
        'method': method,
        '__async': __async,
        'action': 'make_request'
    }
    data.update(**kwargs)

    response = _sync_call_to_remote(data, method, __async)
    return response if kwargs.get('return_complete_response_dict', False) else response.get('result')


def trigger_ingest_playbook(records, playbook_uuid, playbook_input_param="ingestedData", parent_env={}, batch_size=1000,
                            dedup_field=None, pb_params={}, **kwargs):
    if dedup_field:
        seen = set()
        records = [x for x in records if
                   [x[dedup_field].replace(" ", "") not in seen, seen.add(x[dedup_field].replace(" ", ""))][0]]
    url = "/api/triggers/1/notrigger/" + playbook_uuid
    method = "POST"

    parent_wf = parent_env.get('wf_id')
    parent_step = parent_env.get('step_id')
    debug = parent_env.get('debug')
    try:
        for start_index in range(0, len(records), batch_size):
            env = {playbook_input_param: records[start_index: start_index + batch_size]}
            if pb_params:
                env.update(pb_params)
            payload = {
                "_eval_input_params_from_env": True,
                "env": env
            }
            if parent_wf:
                payload['parent_wf'] = parent_wf
            if parent_step:
                payload['step_id'] = parent_step
            if debug:
                payload['env']['debug'] = debug
            make_request(url, method, body=payload)
    except Exception as e:
        logger.error("Failed to insert a batch of feeds with error: " + str(e))


def _populated_Related_info():
    settings.MASTER_ID = all_config.get('cyops.instance.masterId')
    settings.SELF_ID = all_config.get('cyops.instance.agentId')


def maybe_json_or_raise(response):
    '''
    Helper function for processing request responses

    Returns any json found in the response. Otherwise, it will extract the
    response as text, or, failing that, as bytes.

    :return: the response from the request
    :rtype: dict or str or bytes
    :raises: :class:`requests.HTTPError` if status code was 4xx or 5xx
    '''
    if response.ok:
        try:
            logger.info('Processing request responses.')
            return response.json(strict=False)
        except Exception:
            return response.text or response.content
    else:
        msg = ''
        try:
            msg = response.json()
            logger.warn(msg)
        except Exception:
            pass
        if not msg:
            msg = response.text
            logger.warn(msg)

        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError as e:
            # add any response content to the error message
            msg = '{} :: {} :: Url: {}'.format(str(e), msg, response.url)
            logger.error(msg)
            raise requests.exceptions.HTTPError(msg, response=response)


def make_file_upload_request(file_name, file_content, file_type, collection='files', *args, **kwargs):
    if type(file_content) == io.BufferedReader:
        file_content = file_content.read()
    if isinstance(file_content, bytes):
        file_content = b64encode(file_content)
        file_content = file_content.decode()

    data = {
        'file_name': file_name,
        'file_content': file_content,
        'file_type': file_type,
        'collection': collection,
        'action': 'make_file_upload_request'
    }
    data.update(**kwargs)
    response = _sync_call_to_remote(data, 'POST')
    return response.get('result')


def download_file_from_cyops(iri, headers=None, *args, **kwargs):
    if not bool(urlparse(iri).netloc):
        iri = settings.CRUD_HUB_URL + str(iri)

    # get the @id and type (with or without a trailing slash)
    split = iri.split('/')
    obj_id = split[-1]
    if not obj_id:
        obj_id = split[-2]
        obj_type = split[-3]
    else:
        obj_type = split[-2]

    # retrieve the file info from the attachment
    if obj_type != 'files':
        collection = '/api/3/{obj_type}/{obj_id}'.format(obj_type=obj_type, obj_id=obj_id)
        res = make_request(collection, 'GET', None, *args, **kwargs)
        file_iri = res['file']
        if type(file_iri) is not str:
            file_iri = file_iri.get('@id', '')
        return download_file_from_cyops(file_iri, *args, **kwargs)

    response = make_request(iri, 'GET', None, return_complete_response_dict=True, *args, **kwargs)
    content = response.get('result')
    headers = response.get('headers')
    path = write_response_to_file(content)

    save_file_in_env(kwargs.get('env', {}), path)

    return {
        'cyops_file_path': path,
        'filename': headers.get('filename', '')
    }


def write_response_to_file(content):
    file_dir = settings.TMP_FILE_ROOT
    os.makedirs(file_dir, exist_ok=True)
    filename = uuid.uuid4().hex
    path = file_dir + filename
    with open(path, 'wb') as file:
        file.write(b64decode(content))
    return path


def save_file_in_env(env, filename, metadata=None, *args, **kwargs):
    file_dict = env.get('files', {})
    file_dict[filename] = metadata
    env['files'] = file_dict
