import arrow
import base64
import binascii
import hmac

from cshmac import exceptions
from cshmac.utils import HmacFingerprintBuilder


def validate_hmac_message(authorization_header: str, full_uri: str,
                          request_method: str, private_key: str,
                          payload: str=None):
    """
    Validates a message signed via HMAC. Returns True if all of the following
    are true, and raises a validation error otherwise:

    1. The authorization header starts with CS and is properly base64 encoded
    2. The timestamp in the request is within 30 seconds
    3. The modulus for each key matches
    4. The fingerprint provided can be rebuilt exactly
    """

    if not authorization_header.startswith('CS '):
        raise exceptions.MalformedAuthorizationHeader()

    try:
        decoded_header = base64.b64decode(
            authorization_header.replace('CS ', '', 1)
        )
        algorithm, given_time, public_key, hashed_fingerprint = \
            decoded_header.decode().split(';')
    except (UnicodeDecodeError, ValueError, binascii.Error):
        raise exceptions.MalformedAuthorizationHeader()

    seconds_difference = (arrow.utcnow() - arrow.get(given_time)).seconds
    if seconds_difference > 60:
        raise exceptions.HmacSignatureExpired()

    if not payload:
        payload = public_key

    builder = HmacFingerprintBuilder(private_key, given_time,
                                     full_uri, request_method, payload)

    if not hmac.compare_digest(builder.build(), hashed_fingerprint):
        raise exceptions.InvalidFingerprint()

    return True
