from __future__ import absolute_import

import base64
from datetime import datetime

from requests.auth import AuthBase

from cshmac.utils import (
    HmacFingerprintBuilder,
    TIME_FORMAT,
    DEFAULT_ALGORITHM,
)


class HmacAuth(AuthBase):
    """
    Auth that builds an HMAC Authorization header for making API requests to
    CrudHub.

    The Authorization header should be formatted as "CS {base64_header}" where
    base64_header is the following string base64 encoded:

    *sha256;{time};{public_key};{hashed_fingerprint}*

    See HmacFingerprintBuilder for details on how hashed_fingerprint is built
    and the expected time format.
    """

    def __init__(self, full_uri, request_method, public_key, private_key,
                 payload=''):
        self.public_key = public_key
        self.private_key = private_key
        self.full_uri = full_uri
        self.request_method = request_method
        self.payload = payload

    def _create_auth_header(self, public_key, private_key,
                            algorithm, verb, uri, payload):
        current_time = datetime.utcnow().strftime(TIME_FORMAT)

        hashed_fingerprint = HmacFingerprintBuilder(
            private_key, current_time, uri, verb, payload
        ).build()

        base64_header = base64.b64encode(
            '{0};{1};{2};{3}'.format(algorithm, current_time, public_key,
                                     hashed_fingerprint).encode())

        return 'CS {}'.format(base64_header.decode())

    def __call__(self, request):
        if not self.public_key or not self.private_key:
            raise Exception('Missing API Key information to form HMAC request')

        method = self.request_method.upper()
        if method == 'GET':
            self.payload = self.public_key

        request.headers["Authorization"] = self._create_auth_header(
            self.public_key,
            self.private_key,
            DEFAULT_ALGORITHM,
            method,
            self.full_uri,
            self.payload)

        return request
