/* Copyright (C) 2008 - 2024 Fortinet Inc.
All rights reserved.
FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE */


'use strict';

(function () {
    angular
        .module('cybersponse')
        .factory('aiAssistantService', aiAssistantService);

    aiAssistantService.$inject = ['$q', 'API', '$resource', '$interval', '$rootScope', 'playbookService', 'websocketService', 'connectorService', '$http', 'toaster', 'settingsService', 'ViewTemplateService', 'recommendationService', '$filter', '_', 'commonService'];

    function aiAssistantService($q, API, $resource, $interval, $rootScope, playbookService, websocketService, connectorService, $http, toaster, settingsService, ViewTemplateService, recommendationService, $filter, _, commonService) {

        var service = {
            getAllPlaybooks: getAllPlaybooks,
            checkPlaybookExecutionCompletionByPolling: checkPlaybookExecutionCompletionByPolling,
            checkPlaybookExecutionCompletion: checkPlaybookExecutionCompletion,
            copyWithoutBotGeneral: copyWithoutBotGeneral,
            connectorHealthCheck: connectorHealthCheck,
            constantMessages: constantMessages,
            getArtifacts: getArtifacts,
            getDataFromPlaybook: getDataFromPlaybook,
            restoreOriginalValues: restoreOriginalValues,
            getRecommendationSettings: getRecommendationSettings,
            getKeyStoreRecord: getKeyStoreRecord,
            objectToString: objectToString,
            replaceArtifactsWithMaskedData: replaceArtifactsWithMaskedData,
            getRecommendedPlaybooks: getRecommendedPlaybooks,
            unmaskResult: unmaskResult,
            getPlaybook: getPlaybook,
            getSimilarRecords: getSimilarRecords,
            replaceArtifactsInString: replaceArtifactsInString,
            epochToReadable: epochToReadable
        }
        return service;

        function getRecommendationSettings(module) {
            var recommendationSettings = {};
            return $q.all([settingsService.getSystem(), ViewTemplateService.get('modules-' + module + '-detail')])
                .then(function (results) {
                    recommendationSettings.globalSettings = results[0];
                    recommendationSettings.globalSettings.publicValues.recommendation = recommendationSettings.globalSettings.publicValues.recommendation || {};
                    recommendationSettings.globalSettings.publicValues.recommendation.strategy = angular.isUndefined(recommendationSettings.globalSettings.publicValues.recommendation.strategy) ? 'falconBased' : recommendationSettings.globalSettings.publicValues.recommendation.strategy;
                    var data = results[1];
                    if (data.config.workspace && recommendationSettings.strategy !== 'null' && recommendationSettings.strategy !== null) {
                        recommendationSettings.workspace = data.config.workspace;
                        recommendationSettings.workspace.strategy = recommendationSettings.globalSettings.publicValues.recommendation.strategy;
                        return recommendationSettings;
                    }
                    return null; // Return null if suggested playbooks are not enabled
                });
        }

        function getSimilarRecords(workspace, entity, dataFields) {
            var defer = $q.defer();

            if (workspace.similarRecords && workspace.similarRecords.enable) {
                if (workspace.strategy === 'falconBased') { // elastic based similar records
                    var _payload = _similarRequestPayload(workspace.similarRecords, entity, dataFields);
                    _payload.size = 5;
                    recommendationService.getRecommendations(_payload).then(function (response) {
                        // _setSimilarityRecords(response.data.result);
                        defer.resolve(response.data.result)
                    }, function (error) {
                        defer.reject(error);
                    })
                }
                else { // FSR ML base similar records
                    var mlengineParams = {
                        records: $filter('getEndPathName')(entity.originalData['@id']),
                        module: entity.module
                    };
                    _executeFSRMLAction('similar', mlengineParams, entity, dataFields).then(function (response) {
                        defer.resolve(response);
                    },
                        function (error) {
                            defer.reject(error);
                        }
                    );
                }
            }
            else {
                defer.reject();
            }
            return defer.promise;
        }

        function getRecommendedPlaybooks(recommendationSettings, entity) {
            var _payload = _similarRequestPayload(recommendationSettings, entity);
            var defer = $q.defer();
            var suggestedPlaybookList = [];
            recommendationService.getRecommendations(_payload).then(function (response) {
                _payload = _suggestedPlaybooksPayload(response.data.result, entity);  // Params: workspace.suggestedPlaybooks
                recommendationService.getSuggestedPlaybooks(_payload).then(function (response) {
                    var sequence = response.data.sequence || [];
                    var executedPlaybooks = response.data.executedPlaybooks || [];

                    // Get action playbooks to filter suggested playbooks, Playbooks should be listed as per conditions on manual trigger
                    playbookService.getActionPlaybooks(entity, true).then(function (playbooks) {
                        var activePlaybooks = playbooks.reduce(function (filtered, playbook) {
                            if (playbook.isActive && !playbook._hide) {
                                filtered.push(playbook.uuid);
                            }
                            return filtered;
                        }, []);
                        sequence = _.filter(sequence, function (playbookUuid) {
                            return _.contains(activePlaybooks, playbookUuid);
                        });
                        // Fetch playbook description and name details
                        if (sequence.length > 0) {
                            playbookService.getPlaybooksData(sequence, ['name', 'description', 'steps', 'triggerStep']).then(function (results) {
                                // Retrieve allowed playbooks list
                                var allowedPlaybooks = results['hydra:member'].map(playbook => playbook.uuid);
                                // Filter sequence list to remove deleted or private playbooks if any
                                if (sequence.length > allowedPlaybooks.length) {
                                    sequence = _.filter(sequence, function (playbookUuid) {
                                        return _.contains(allowedPlaybooks, playbookUuid);
                                    });
                                }
                                if (sequence.length > 0) {
                                    // Restrict records count to 5
                                    sequence = sequence.length > 5 ? sequence.splice(0, 5) : sequence;
                                    // Form suggested playbooks object for render
                                    sequence.forEach(function (record) {
                                        suggestedPlaybookList.push({ actionUuid: record });
                                    });
                                    results['hydra:member'].forEach(function (playbook) {
                                        suggestedPlaybookList.forEach(function (record) {
                                            if (record.actionUuid === playbook.uuid) {
                                                var triggerStep = playbookService.getTriggerStep(playbook);
                                                record.actionName = triggerStep.arguments.title || playbook.name;
                                                record.actionDesc = playbook.description;
                                                record.executed = _.contains(executedPlaybooks, playbook.uuid);
                                            }
                                        });
                                        defer.resolve(suggestedPlaybookList);
                                    });
                                }
                            }).finally(function () {
                            });
                        }
                        else{
                            defer.resolve([])
                        }
                    })
                }, function (error) {
                    defer.reject(error);
                })
            }, function (error) {
                defer.reject(error);
            });
            return defer.promise;
        }

        // function to excute FSR ML connector action 
        function _executeFSRMLAction(_type, params, entity, dataFields) {
            var defer = $q.defer();
            var connectorName = 'fortisoar-ml-engine';
            var operation = _type === 'prediction' ? 'predict' : 'similar';
            // get recommendation by FSR ML connector
            connectorService.executeConnectorAction(connectorName, null, operation, null, params).then(function (response) {
                // response.data contains array of uuid's of similar records
                getRecordDetails(response.data, entity, dataFields).then(function (result) {
                    // _setSimilarityRecords(result);
                    defer.resolve(result);
                });
            }, function (error) {
                defer.reject(error)
            });
            return defer.promise;

        }

        function getRecordDetails(records, entity, fields) {
            var defer = $q.defer();
            var _fields = fields;
            commonService.getRecordsDetail({
                module: entity.module,
                uuids: records,
                selectFields: _fields
            }).then(function (response) {
                defer.resolve(response);
            }, function (error) {
                $scope.params.similarError = error.data.message;
                $scope.params.similarErrorStatus = true;
                defer.reject(error);
            });
            return defer.promise;
        }


        function _suggestedPlaybooksPayload(uuidArray, entity) {  // Params: config
            var similarRecordsUUID = uuidArray.map((record) => {
                var recordObj = {};
                recordObj.uuid = record.uuid;
                recordObj.score = record.__score;
                return recordObj;
            });
            return {
                resource: entity.module,
                recordUuid: $filter('getEndPathName')(entity.originalData['@id']),
                similarRecordsUuid: similarRecordsUUID
            };
        }

        function _similarRequestPayload(config, entity, outputFields) {
            var fields = [];
            var dataFields = [];
            angular.forEach(config.fields, function (field) {
                var weight = config.enableWeightRange ? field.weight : 1;
                fields.push({
                    name: field.name,
                    weight: weight
                });
            });

            if (outputFields) {
                dataFields = outputFields;
            }
            else {
                // card view fields
                angular.forEach(config.mapping, function (fieldName) {
                    if (fieldName) {
                        dataFields.push(fieldName);
                    }
                });
            }
            dataFields.push('createDate');
            /* jshint camelcase: false */
            return {
                index: 'cyops_' + entity.module,
                module_type: entity.module,
                sort: '_score',
                offset: 0,
                size: 10,
                fields: fields,
                filter: config.filter,
                query_type: 'similar',
                fromDate: 1,
                id: $filter('getEndPathName')(entity.originalData['@id']), // alert id
                output_fields: dataFields
            };
        }




        //trigger respective playbook to either get playbook of conversation text
        function getDataFromPlaybook(playbookTags, parametersForPlaybook, scope) {
            var defer = $q.defer();
            var queryObjectPlaybook = {
                'sort': [
                    {
                        'field': 'createDate',
                        'direction': 'DESC',
                        '_fieldName': 'createDate'
                    }
                ],
                'limit': 1,
                'logic': 'AND',
                'filters': [
                    {
                        'field': 'recordTags',
                        'value': playbookTags,
                        'display': '',
                        'operator': 'in',
                        'type': 'array'
                    },
                    {
                        'field': 'isActive',
                        'value': true,
                        'operator': 'eq'
                    }
                ],
                '__selectFields': [
                    'id',
                    'name'
                ]
            };
            scope.processCompleted = 6;
            // add a tag to the playbook, get the playbook instead of hardcoding the playbook iri
            getAllPlaybooks(queryObjectPlaybook).then(function (playbookUUID) {
                var queryUrl = API.MANUAL_TRIGGER + playbookUUID + '?force_debug=true';
                //Parametersforplaybook -> 'Edit Parameters' in playbook
                $http.post(queryUrl, parametersForPlaybook).then(function (result) {
                    if (result && result.data && result.data.task_id) {
                        checkPlaybookExecutionCompletion([result.data.task_id], function (response) {
                            if (response && (response.status === 'finished' || response.status === 'failed' || response.status === 'terminated')) {
                                playbookService.getExecutedPlaybookLogData(response.instance_ids).then(function (res) {
                                    defer.resolve(res);
                                }, function (err) {
                                    defer.reject(err);
                                });
                            }
                        }, function (error) {
                            defer.reject(error);
                        }, scope);
                    }
                }, function (err) {
                    defer.reject(err);
                });
            }, function (err) {
                defer.reject(err);
            });
            return defer.promise;
        }

        function restoreOriginalValues(maskedText, maskedValues) {
            maskedText = Object.entries(maskedValues.emails).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            maskedText = Object.entries(maskedValues.ips).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            maskedText = Object.entries(maskedValues.domains).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            maskedText = Object.entries(maskedValues.hashes).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            return maskedText;
        }


        function getArtifacts(text) {
            const htmlTagRegex = /<[^>]*>/g;
            text.replace(htmlTagRegex, '');
            // var text = "Outbound cleartext password usage from non guest network from 192.168.77.39 to 49.163.227.34 on port 110";
            return $resource(API.INTEGRATIONS + 'connectors/?name=' + 'cyops_utilities')
                .get()
                .$promise
                .then(function (connectorMetaDataForVersion) {
                    return connectorService.executeConnectorAction('cyops_utilities', connectorMetaDataForVersion.data[0].version, 'extract_artifacts', null, { data: text });
                })
                .catch(function (error) {
                    console.error('Error:', error);
                    throw error; // Rethrow the error to be handled by the caller
                });
        }

        function objectToString(obj) {
            let result = '';
            for (const [key, value] of Object.entries(obj)) {
                // Check if the value starts with a newline character and remove \n 
                const formattedValue = value.replace(/\n/g, '');
                result += `${key}: ${formattedValue}\n\n`;
            }
            return result;
        }

        function replaceArtifactsInString(stringToParse, replacements){
            constantMessages().keysToParseInOrder.forEach(type => {
                if (replacements[type]) {
                    replacements[type].forEach((artifact, index) => {
                        // handling a case if a string ends with \ 
                        artifact = artifact.replace(/\\$/, "");
                        const regex = new RegExp(artifact, 'g');
                        stringToParse = stringToParse.replace(regex, `{${type}-${index + 1}}`);
                    });
                }
            });
            return stringToParse;
        }

        function replaceArtifactsWithMaskedData(obj, replacements) {
            for (const key in obj) {
                if (typeof obj[key] === 'string') {
                    constantMessages().keysToParseInOrder.forEach(type => {
                        if (replacements[type]) {
                            replacements[type].forEach((artifact, index) => {
                                // handling a case if a string ends with \ 
                                artifact = artifact.replace(/\\$/, "");
                                const regex = new RegExp(artifact, 'g');
                                obj[key] = obj[key].replace(regex, `{${type}-${index + 1}}`);
                            });
                        }
                    });
                }
            }
            return obj;
        }

        function unmaskResult(data, string) {
            constantMessages().keysToParseInOrder.forEach(key => {
                if (data.hasOwnProperty(key) && Array.isArray(data[key])) {
                    var regex = new RegExp(`{${key}-\\d+}`, 'g');
                    string = string.replace(regex, match => {
                        const index = parseInt(match.match(/\d+/)[0]) - 1;
                        return data[key][index] || match;
                    });
                    var regex = new RegExp(`${key}-\\d+`, 'g');
                    string = string.replace(regex, match => {
                        const index = parseInt(match.match(/\d+/)[0]) - 1;
                        return data[key][index] || match;
                    });
                }
            });
            return string;
        }

        function connectorHealthCheck(connectorName) {
            return $resource(API.INTEGRATIONS + 'connectors/?name=' + connectorName)
                .get()
                .$promise
                .then(function (connectorMetaDataForVersion) {
                    return connectorService.getConnector(connectorName, connectorMetaDataForVersion.data[0].version);
                })
                .then(function (connectorMetaData) {
                    if (connectorMetaData.configuration && connectorMetaData.configuration.length > 0) {
                        var promises = connectorMetaData.configuration.map(function (configs) {
                            return connectorService.getConnectorHealth(connectorMetaData, configs.config_id)
                                .then(function (configurationDetails) {
                                    return configurationDetails.status === 'Available';
                                });
                        });

                        // Use Promise.all to wait for all promises to resolve
                        return Promise.all(promises)
                            .then(function (results) {
                                // If any result is true, set connectorHealthCheckFlag to true
                                return results.some(Boolean);
                            });
                    } else {
                        return false;
                    }
                })
                .catch(function (error) {
                    console.error("Error in getConnector or getConnectorHealth:", error);
                    return false;
                });
        }

        function poll(_reqConfig, callback) {
            return $interval(function () {
                if (_reqConfig.hasValueReturned) { //check flag before start new call
                    callback(_reqConfig);
                }
                _reqConfig.thresholdValue = _reqConfig.thresholdValue - 1; //Decrease threshold value
                if (_reqConfig.thresholdValue === 0) {
                    stopPoll(_reqConfig); // Stop $interval if it reaches to threshold
                }
            }, _reqConfig.pollInterval);
        }
        function stopPoll(_reqConfig) {
            $interval.cancel(_reqConfig.pollPromise);
            _reqConfig.pollPromise = undefined;
            _reqConfig.thresholdValue = 0; //reset all flags.
            _reqConfig.hasValueReturned = true;
        }

        //If websocket is not active make api calls to check if the playbook execution is completed
        function checkPlaybookExecutionCompletionByPolling(params) {
            var _pollConfig = {
                hasValueReturned: true,
                thresholdValue: 150,
                pollInterval: params.pollInterval || 2000,
                pollPromise: undefined,
                defer: undefined
            };
            _pollConfig.defer = $q.defer();
            _pollConfig.pollPromise = poll(_pollConfig, function (callbackParam) {
                callbackParam.hasValueReturned = false;
                $resource(API.WORKFLOW + 'api/workflows/log_list/?format=json&parent__isnull=True&task_id=' + params.taskId).save({},
                    function (data) {
                        _pollConfig.hasValueReturned = true;
                        if (data['hydra:member'] && data['hydra:member'].length > 0 && (data['hydra:member'][0].status === 'finished' || data['hydra:member'][0].status === 'failed' || data['hydra:member'][0].status === 'terminated')) {
                            stopPoll(callbackParam);
                            _pollConfig.defer.resolve(data['hydra:member'][0]);
                        }
                    },
                    function (error) {
                        statusCodeService(error, true);
                        if (params.callback) {
                            params.callback(error);
                        }
                        stopPoll(callbackParam);
                        _pollConfig.defer.reject(error);
                    });
            });
            if (params.keepPollConfig) {
                playbookPollConfig.push(_pollConfig);
            }
            return _pollConfig.defer.promise;
        }

        //adds a listener to the triggered playbook
        function checkPlaybookExecutionCompletion(taskIds, callbackFunction, errorHandler, scope) {
            var defer = $q.defer();
            var processCompleted = 0;
            if ($rootScope.websocketConnected) {
                websocketService.subscribe('runningworkflow', function (result) {
                    if (scope.generatePlaybookFlag && (result.task_id && taskIds.includes(result.task_id) && result.status && (result.status === 'finished' || result.status === 'failed'))) {
                        processCompleted++;
                        updateProgressBar(scope, false, false, processCompleted);
                    }
                    if (result.task_id && taskIds.includes(result.task_id) && result.status && result.parent_wf === 'null' && (result.status === 'finished' || result.status === 'failed' || result.status === 'finished_with_error' || result.status === 'terminated')) {
                        taskIds.splice(taskIds.indexOf(result.task_id), 1);
                    }
                    if (taskIds.length === 0 || (result.status === 'failed' || result.status === 'finished_with_error' || result.status === 'terminated')) {
                        callbackFunction(result);
                        websocketService.unsubscribe(scope.playbookStatusSubscription);
                        updateProgressBar(scope, false, true);
                        scope.playbookStatusSubscription = undefined;
                    }
                }).then(function (sub) {
                    playbookService.detachPaybookStatusWebsocket(scope.playbookStatusSubscription);
                    if (scope) {
                        scope.playbookStatusSubscription = sub;
                    }
                    defer.resolve(sub);
                }, function () {
                    defer.reject();
                    if (errorHandler) {
                        errorHandler();
                    }
                });
            }
            else {
                var taskIdsCopy = angular.copy(taskIds);
                angular.forEach(taskIdsCopy, function (taskId) {
                    if (scope.generatePlaybookFlag) {
                        updateProgressBar(scope, true, false);
                    }
                    checkPlaybookExecutionCompletionByPolling({ taskId: taskId }).then(function (response) {
                        if (response && (response.status === 'finished' || response.status === 'failed' || response.status === 'terminated')) {
                            taskIds.splice(taskIds.indexOf(taskId), 1);
                            response.instance_ids = parseInt(response['@id'].split('/wf/api/workflows/')[1]);
                        }
                        if (taskIds.length === 0) {
                            callbackFunction(response);
                            updateProgressBar(scope, false, true);
                        }
                    }, function () {
                        defer.reject();
                        if (errorHandler) {
                            errorHandler();
                        }
                    });
                });
            }
            return defer.promise;
        }

        //Update the loading percentage
        function updateProgressBar(scope, moveWithInterval, executionComplete, processCompleted) {
            if (executionComplete) {
                scope.processCompleted = 100;
                executionComplete = false;
            }
            else if (moveWithInterval) {
                var stepsExecuted = 1;
                var intervalId = setInterval(function () {
                    if (stepsExecuted < scope.numberOfDivisions) {
                        var chunk = Math.floor(((100 / scope.numberOfDivisions) * stepsExecuted));
                        if (chunk < 97) {
                            scope.processCompleted = chunk;
                        }
                        //If the number of steps in description and playbook block created are different
                        stepsExecuted++;
                    }
                    else {
                        clearInterval(intervalId);
                    }
                }, 30000);
            }
            else {
                var chunk = Math.floor(((100 / scope.numberOfDivisions) * processCompleted));
                //Check if the percentage is not greater than 100
                if (chunk < 97) {
                    scope.processCompleted = chunk;
                }
            }
        }

        //Copy all text to the local storage, without botgeneral
        function copyWithoutBotGeneral(originalData) {
            // Create a deep copy of the original data to avoid modifying the original object
            var copiedData = JSON.parse(JSON.stringify(originalData));

            // Filter out objects with type "botGeneral" from playbookSuggestionMessages except the first one
            copiedData.playbookSuggestionMessages = copiedData.playbookSuggestionMessages.filter(function (message, index) {
                return index === 0 || message.type !== "botGeneral";
            });

            // Filter out objects with type "botGeneral" from conversationInDesigner except the first one
            copiedData.conversationInDesigner = copiedData.conversationInDesigner.filter(function (message, index) {
                return index === 0 || message.type !== "botGeneral";
            });

            copiedData.conversationThroughOutApplication = copiedData.conversationThroughOutApplication.filter(function (message, index) {
                return index === 0 || message.type !== "botGeneral";
            });

            return copiedData;
        }

        function getKeyStoreRecord(queryObject, module) {
            var defer = $q.defer();
            var url = API.QUERY + module;
            $resource(url).save(queryObject, function (response) {
                defer.resolve(response);
            }, function (err) {
                defer.reject(err);
            })
            return defer.promise;
        }

        function getPlaybook(playbookIRI, module) {
            var defer = $q.defer();
            if (playbookService.loadedPlaybookActions && playbookService.loadedPlaybookActions[module]) {
                var _pb = _.find(playbookService.loadedPlaybookActions[module].playbooks, function (pb) {
                    return pb['@id'] === playbookIRI;
                });
                if (_pb) {
                    defer.resolve({ data: _pb });
                    return defer.promise;
                }
            }
            var _playbookId = $filter('getEndPathName')(playbookIRI);
            $http.get(API.BASE + API.WORKFLOWS + _playbookId + '?$relationships=true').then(function (response) {
                defer.resolve(response);
            }, function (error) {
                defer.reject(error);
            });
            return defer.promise;
        }

        function epochToReadable(epochTime) {
            // Convert epoch time to milliseconds
            const milliseconds = epochTime;
        
            // Create a new Date object with the milliseconds
            const date = new Date(milliseconds);
        
            // Extract individual date components
            const year = date.getFullYear();
            const month = ('0' + (date.getMonth() + 1)).slice(-2); // Adding 1 because month is zero-based
            const day = ('0' + date.getDate()).slice(-2);
            const hours = ('0' + date.getHours()).slice(-2);
            const minutes = ('0' + date.getMinutes()).slice(-2);
            const seconds = ('0' + date.getSeconds()).slice(-2);
        
            // Construct the readable date format
            const readableFormat = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
        
            return readableFormat;
        }


        function getAllPlaybooks(queryObject) {
            var defer = $q.defer();
            var url = API.QUERY + 'workflows';
            $resource(url).save(queryObject, function (response) {
                if (response['hydra:member'] && (response['hydra:member'][0])) {
                    defer.resolve(response['hydra:member'][0]['uuid']);
                }
                else {
                    defer.reject("Playbook Not Found");
                    toaster.error({ body: "Playbook not found" });
                }
            }, function (error) {
                defer.reject(error);
            });
            return defer.promise;
        }

        function constantMessages() {
            return {
                queryForKeyStore: {
                    "sort": [
                        {
                            "field": "id",
                            "direction": "ASC",
                            "_fieldName": "id"
                        }
                    ],
                    "limit": 30,
                    "logic": "AND",
                    "filters": [
                        {
                            "field": "key",
                            "operator": "like",
                            "_operator": "like",
                            "value": "%FortiAI Static Questions%",
                            "type": "primitive"
                        },
                        {
                            "sort": [],
                            "limit": 30,
                            "logic": "AND",
                            "filters": []
                        }
                    ],
                    "__selectFields": [
                        "id",
                        "key",
                        "value",
                        "notes",
                        "@id",
                        "@type",
                        "jSONValue"
                    ]
                },
                keysToParseInOrder: ["Email", "URL", "Host", "IP", "MD5", "SHA1", "SHA256", "CVE", "Registry"],
                playbookTags: {
                    pBDesignerDescription: 'aibot-playbookPlanSuggestion',
                    pBDesignerSteps: 'aibot-playbookBlockSuggestion',
                    conversation: 'aibot-conversation'
                },
                configurationFailedMessage: "Unfortunately, there's not much that can be done at the moment. It seems that the OpenAI integration isn't properly configured or might be encountering issues. We recommend reaching out to your system administrator for assistance.",
                initialMessage: {
                    playbookSuggestionMessages: "Hi there! How can I help you today? I can help you generate playbook templates for your common use cases. For help on best practices on asking questions that give best results, use the 'help' icon above.",
                    conversationInDesigner: "Hi there! How can I help you today? Ask me questions like ‘how can I remove all HTML tags from a string’ OR ‘give some suggestions for responding to Ransomware alerts’ or ‘log samples for Fortigate firewall’.",
                    conversationThroughOutApplication: "Hi there! How can I help you today?"
                },
                connectorNotConfigured: 'Connector is not configured',
                connectorFetchApiFails: 'Error in fetching connector configuration',
                playbookFailed: "I'm sorry, it seems like there was an issue generating a response. Please try your request again, or refer to the advisor troubleshooting documentation for assistance.",
                playbookFailedToaster: 'Error in fetching playbook data',
                playbookSuggestionMessages: {
                    proceedingToGenerate: 'Proceeding to generate playbook template.',
                    generatedSuccessfully: 'Playbook template successfully generated! Adding the block now.',
                    buildingDescription: 'Sure, let me build and share a playbook outline to review.'
                }
            };
        }

    }
})();