/* Copyright (C) 2008 - 2025 Fortinet Inc.
All rights reserved.
FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE */


'use strict';

(function () {
    angular
        .module('cybersponse')
        .factory('aiAssistantService', aiAssistantService);

    aiAssistantService.$inject = ['$q', 'API', '$resource', '$interval', '$rootScope', 'playbookService', 'websocketService', 'connectorService', '$http', 'toaster', 'settingsService', 'ViewTemplateService', 'recommendationService', '$filter', '_', 'commonService','$location'];

    function aiAssistantService($q, API, $resource, $interval, $rootScope, playbookService, websocketService, connectorService, $http, toaster, settingsService, ViewTemplateService, recommendationService, $filter, _, commonService, $location) {

        var service = {
            checkPlaybookExecutionCompletion: checkPlaybookExecutionCompletion,
            checkPlaybookExecutionCompletionByPolling: checkPlaybookExecutionCompletionByPolling,
            constantMessages: constantMessages,
            connectorHealthCheck: connectorHealthCheck,
            copyWithoutBotGeneral: copyWithoutBotGeneral,
            epochToReadable: epochToReadable,
            executeAction: executeAction,
            getAllPlaybooks: getAllPlaybooks,
            getArtifacts: getArtifacts,
            getDataFromPlaybook: getDataFromPlaybook,
            getKeyStoreRecord: getKeyStoreRecord,
            getPlaybook: getPlaybook,
            getRecommendationSettings: getRecommendationSettings,
            getRecommendedPlaybooks: getRecommendedPlaybooks,
            getSimilarRecords: getSimilarRecords,
            objectToString: objectToString,
            replaceArtifactsInString: replaceArtifactsInString,
            replaceArtifactsWithMaskedData: replaceArtifactsWithMaskedData,
            restoreOriginalValues: restoreOriginalValues,
            unmaskString: unmaskString,
            getUserLoginId: getUserLoginId,
            getUserPreferences: getUserPreferences,
            setUserPreferences: setUserPreferences,
            getDetailData: getDetailData,
            fetchURL: fetchURL,
            getModuleIRI: getModuleIRI,
            getUUIDFromURL: getUUIDFromURL,
            getUpdatedURL: getUpdatedURL,
            checkIfRecordOpen: checkIfRecordOpen,
            getPastConversationAction: getPastConversationAction
        }
        return service;

        function getRecommendationSettings(module) {
            var recommendationSettings = {};
            return $q.all([settingsService.getSystem(), ViewTemplateService.get('modules-' + module + '-detail')])
                .then(function (results) {
                    recommendationSettings.globalSettings = results[0];
                    recommendationSettings.globalSettings.publicValues.recommendation = recommendationSettings.globalSettings.publicValues.recommendation || {};
                    recommendationSettings.globalSettings.publicValues.recommendation.strategy = angular.isUndefined(recommendationSettings.globalSettings.publicValues.recommendation.strategy) ? 'falconBased' : recommendationSettings.globalSettings.publicValues.recommendation.strategy;
                    var data = results[1];
                    if (data.config.workspace && recommendationSettings.strategy !== 'null' && recommendationSettings.strategy !== null) {
                        recommendationSettings.workspace = data.config.workspace;
                        recommendationSettings.workspace.strategy = recommendationSettings.globalSettings.publicValues.recommendation.strategy;
                        return recommendationSettings;
                    }
                    return null; // Return null if suggested playbooks are not enabled
                });
        }

        function getSimilarRecords(workspace, entity, dataFields) {
            var defer = $q.defer();

            if (workspace.similarRecords && workspace.similarRecords.enable) {
                if (workspace.strategy === 'falconBased') { // elastic based similar records
                    var _payload = _similarRequestPayload(workspace.similarRecords, entity, dataFields);
                    _payload.size = 5;
                    recommendationService.getRecommendations(_payload).then(function (response) {
                        // _setSimilarityRecords(response.data.result);
                        defer.resolve(response.data.result)
                    }, function (error) {
                        defer.reject(error);
                    })
                }
                else { // FSR ML base similar records
                    var mlengineParams = {
                        records: $filter('getEndPathName')(entity.originalData['@id']),
                        module: entity.module
                    };
                    _executeFSRMLAction('similar', mlengineParams, entity, dataFields).then(function (response) {
                        defer.resolve(response);
                    },
                        function (error) {
                            defer.reject(error);
                        }
                    );
                }
            }
            else {
                defer.reject();
            }
            return defer.promise;
        }

        function getRecommendedPlaybooks(recommendationSettings, entity) {
            var _payload = _similarRequestPayload(recommendationSettings, entity);
            var defer = $q.defer();
            var suggestedPlaybookList = [];
            recommendationService.getRecommendations(_payload).then(function (response) {
                _payload = _suggestedPlaybooksPayload(response.data.result, entity);  // Params: workspace.suggestedPlaybooks
                recommendationService.getSuggestedPlaybooks(_payload).then(function (response) {
                    var sequence = response.data.sequence || [];
                    var executedPlaybooks = response.data.executedPlaybooks || [];

                    // Get action playbooks to filter suggested playbooks, Playbooks should be listed as per conditions on manual trigger
                    playbookService.getActionPlaybooks(entity, true).then(function (playbooks) {
                        var activePlaybooks = playbooks.reduce(function (filtered, playbook) {
                            if (playbook.isActive && !playbook._hide) {
                                filtered.push(playbook.uuid);
                            }
                            return filtered;
                        }, []);
                        sequence = _.filter(sequence, function (playbookUuid) {
                            return _.contains(activePlaybooks, playbookUuid);
                        });
                        // Fetch playbook description and name details
                        if (sequence.length > 0) {
                            playbookService.getPlaybooksData(sequence, ['name', 'description', 'steps', 'triggerStep']).then(function (results) {
                                // Retrieve allowed playbooks list
                                var allowedPlaybooks = results['hydra:member'].map(playbook => playbook.uuid);
                                // Filter sequence list to remove deleted or private playbooks if any
                                if (sequence.length > allowedPlaybooks.length) {
                                    sequence = _.filter(sequence, function (playbookUuid) {
                                        return _.contains(allowedPlaybooks, playbookUuid);
                                    });
                                }
                                if (sequence.length > 0) {
                                    // Restrict records count to 5
                                    sequence = sequence.length > 5 ? sequence.splice(0, 5) : sequence;
                                    // Form suggested playbooks object for render
                                    sequence.forEach(function (record) {
                                        suggestedPlaybookList.push({ actionUuid: record });
                                    });
                                    results['hydra:member'].forEach(function (playbook) {
                                        suggestedPlaybookList.forEach(function (record) {
                                            if (record.actionUuid === playbook.uuid) {
                                                var triggerStep = playbookService.getTriggerStep(playbook);
                                                record.actionName = triggerStep.arguments.title || playbook.name;
                                                record.actionDesc = playbook.description;
                                                record.executed = _.contains(executedPlaybooks, playbook.uuid);
                                            }
                                        });
                                        defer.resolve(suggestedPlaybookList);
                                    });
                                }
                            }).finally(function () {
                            });
                        }
                        else{
                            defer.resolve([])
                        }
                    })
                }, function (error) {
                    defer.reject(error);
                })
            }, function (error) {
                defer.reject(error);
            });
            return defer.promise;
        }

        // function to excute FSR ML connector action 
        function _executeFSRMLAction(_type, params, entity, dataFields) {
            var defer = $q.defer();
            var connectorName = 'fortisoar-ml-engine';
            var operation = _type === 'prediction' ? 'predict' : 'similar';
            // get recommendation by FSR ML connector
            connectorService.executeConnectorAction(connectorName, null, operation, null, params).then(function (response) {
                // response.data contains array of uuid's of similar records
                getRecordDetails(response.data, entity, dataFields).then(function (result) {
                    // _setSimilarityRecords(result);
                    defer.resolve(result);
                });
            }, function (error) {
                defer.reject(error)
            });
            return defer.promise;

        }

        function getRecordDetails(records, entity, fields) {
            var defer = $q.defer();
            var _fields = fields;
            commonService.getRecordsDetail({
                module: entity.module,
                uuids: records,
                selectFields: _fields
            }).then(function (response) {
                defer.resolve(response);
            }, function (error) {
                $scope.params.similarError = error.data.message;
                $scope.params.similarErrorStatus = true;
                defer.reject(error);
            });
            return defer.promise;
        }


        function _suggestedPlaybooksPayload(uuidArray, entity) {  // Params: config
            var similarRecordsUUID = uuidArray.map((record) => {
                var recordObj = {};
                recordObj.uuid = record.uuid;
                recordObj.score = record.__score;
                return recordObj;
            });
            return {
                resource: entity.module,
                recordUuid: $filter('getEndPathName')(entity.originalData['@id']),
                similarRecordsUuid: similarRecordsUUID
            };
        }

        function _similarRequestPayload(config, entity, outputFields) {
            var fields = [];
            var dataFields = [];
            angular.forEach(config.fields, function (field) {
                var weight = config.enableWeightRange ? field.weight : 1;
                fields.push({
                    name: field.name,
                    weight: weight
                });
            });

            if (outputFields) {
                dataFields = outputFields;
            }
            else {
                // card view fields
                angular.forEach(config.mapping, function (fieldName) {
                    if (fieldName) {
                        dataFields.push(fieldName);
                    }
                });
            }
            dataFields.push('createDate');
            /* jshint camelcase: false */
            return {
                index: 'cyops_' + entity.module,
                module_type: entity.module,
                sort: '_score',
                offset: 0,
                size: 10,
                fields: fields,
                filter: config.filter,
                query_type: 'similar',
                fromDate: 1,
                id: $filter('getEndPathName')(entity.originalData['@id']), // alert id
                output_fields: dataFields
            };
        }




        //trigger respective playbook to either get playbook of conversation text
        function getDataFromPlaybook(playbookTags, parametersForPlaybook, scope) {
            var defer = $q.defer();
            var queryObjectPlaybook = {
                'sort': [
                    {
                        'field': 'createDate',
                        'direction': 'DESC',
                        '_fieldName': 'createDate'
                    }
                ],
                'limit': 1,
                'logic': 'AND',
                'filters': [
                    {
                        'field': 'recordTags',
                        'value': playbookTags,
                        'display': '',
                        'operator': 'in',
                        'type': 'array'
                    },
                    {
                        'field': 'isActive',
                        'value': true,
                        'operator': 'eq'
                    }
                ],
                '__selectFields': [
                    'id',
                    'name'
                ]
            };
            scope.processCompleted = 6;
            // add a tag to the playbook, get the playbook instead of hardcoding the playbook iri
            getAllPlaybooks(queryObjectPlaybook).then(function (playbookUUID) {
                var queryUrl = API.MANUAL_TRIGGER + playbookUUID + '?force_debug=true';
                //Parametersforplaybook -> 'Edit Parameters' in playbook
                $http.post(queryUrl, parametersForPlaybook).then(function (result) {
                    if (result && result.data && result.data.task_id) {
                        checkPlaybookExecutionCompletion([result.data.task_id], function (response) {
                            if (response && (response.status === 'finished' || response.status === 'failed' || response.status === 'terminated')) {
                                playbookService.getExecutedPlaybookLogData(response.instance_ids).then(function (res) {
                                    defer.resolve(res);
                                }, function (err) {
                                    defer.reject(err);
                                });
                            }
                        }, function (error) {
                            defer.reject(error);
                        }, scope);
                    }
                }, function (err) {
                    defer.reject(err);
                });
            }, function (err) {
                defer.reject(err);
            });
            return defer.promise;
        }

        function restoreOriginalValues(maskedText, maskedValues) {
            maskedText = Object.entries(maskedValues.emails).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            maskedText = Object.entries(maskedValues.ips).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            maskedText = Object.entries(maskedValues.domains).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            maskedText = Object.entries(maskedValues.hashes).reduce((acc, [key, value]) => acc.replaceAll(key, value), maskedText);
            return maskedText;
        }


        function getArtifacts(text) {
            const htmlTagRegex = /<[^>]*>/g;
            text.replace(htmlTagRegex, '');
            return executeAction('cyops_utilities', 'extract_artifacts', null, { data: text });
        }

        function executeAction(connector_name, connector_action, userLoginId, payload) {
            return $resource(API.INTEGRATIONS + 'connectors/?name=' + connector_name)
            .get()
                .$promise
                .then(function (connectorMetaDataForVersion) {
                    return connectorService.executeConnectorAction(connector_name, connectorMetaDataForVersion.data[0].version, connector_action, userLoginId, payload);
                })
                .catch(function (error) {
                    console.error('Error:', error);
                    throw error; // Rethrow the error to be handled by the caller
                });
        }

        function objectToString(obj) {
            let result = '';
            for (const [key, value] of Object.entries(obj)) {
                // Check if the value starts with a newline character and remove \n 
                const formattedValue = value.replace(/\n/g, '');
                result += `${key}: ${formattedValue}\n\n`;
            }
            return result;
        }

        //to replace artifacts in give string 
        function replaceArtifactsInString(stringToParse, replacements, maskedObjectFactory){
            constantMessages().keysToMaskArtifactsInOrder.forEach(type => {
                if (replacements[type]) {
                    replacements[type].forEach((artifact, index) => {
                        // handling a case if a string ends with \ 
                        artifact = artifact.replace(/\\$/, "");
                        const regex = new RegExp(artifact, 'g');
                        //get index of given type i.e email,ip,domain..
                        const ioc_type_mask_index = maskedObjectFactory.ioc_type.findIndex((ioc)=> ioc.type === type);
                        let typeCount = 0;
                        if(ioc_type_mask_index === -1) { //if typeIndex is not present it is first entry
                            typeCount = 1;
                            maskedObjectFactory.ioc_type.push({'type': type, 'count': typeCount});
                            stringToParse = stringToParse.replace(regex, `{${type}-${typeCount}}`);
                            maskedObjectFactory.masked_data[`{${type}-${typeCount}}`] =  artifact;
                        } else { //else check if artifact value is present in maskedObjectFactory
                            let valueExists = Object.values(maskedObjectFactory.masked_data).includes(artifact);
                            if(valueExists){ 
                                let objectKey = Object.entries(maskedObjectFactory.masked_data).find(([key, value]) => value === artifact)?.[0];
                                //if artifact value is present in maskedObjectFactory use that key to replace string
                                stringToParse = stringToParse.replace(regex, `${objectKey}`);
                            }else{ 
                                //else increase the count and add value against new key
                                typeCount = ++maskedObjectFactory.ioc_type[ioc_type_mask_index].count;
                                maskedObjectFactory.masked_data[`{${type}-${typeCount}}`] =  artifact;
                                stringToParse = stringToParse.replace(regex, `{${type}-${typeCount}}`);
                            }  
                        }      
                    });
                }
            });
            return stringToParse;
        }

        //entire record data object to be masked on detail view
        function replaceArtifactsWithMaskedData(obj, replacements, maskedObjectFactory) {
            for (const key in obj) {
                if (typeof obj[key] === 'string') {
                    obj[key] = replaceArtifactsInString(obj[key], replacements, maskedObjectFactory);                  
                }
            }
            return obj;
        }
        
        function unmaskString(masked_ObjectFactory,string){
            let maskedDataEntries = Object.entries(masked_ObjectFactory).reverse();
           //for (const key in object) {
            maskedDataEntries.forEach(([key, value]) => {
                // Added { and } at the end of key to check for {{IP-1}}
                let literalKey = `{${key}}`;
                if (string.includes(literalKey)) {
                    string = string.replace(new RegExp(literalKey, 'g'), value);
                }
                // To check for {IP-1}
                if (string.includes(key)) {
                    string = string.replace(new RegExp(key, 'g'), value);
                }
                // to check for IP-1
                if (string.includes(key.replace(/[{}]/g, ''))) {
                    string = string.replace(new RegExp(key.replace(/[{}]/g, ''), 'g'), value);
                }
            });
            return string;
        }

        function connectorHealthCheck(aiConfiguration, userLoginId) {
            var errorMessage = '';
            var connectorName = aiConfiguration['llmIntegrationName'];
            var isMultiUser = aiConfiguration['isMultiConfigAvailable'];
            var connectorTitle = aiConfiguration['llmIntegrationTitle'];
            return $resource(API.INTEGRATIONS + 'connectors/?name=' + connectorName)
                .get()
                .$promise
                .then(function (connectorMetaDataForVersion) {
                    return connectorService.getConnector(connectorName, connectorMetaDataForVersion.data[0].version);
                })
                .then(function (connectorMetaData) {
                    if (connectorMetaData.configuration && connectorMetaData.configuration.length > 0) {
                        var connector_config = false; 
                        if(isMultiUser){ //for multi user connector_definition will check user login id matches
                            connector_config = connectorMetaData.configuration.find(function (config) {
                                return config.name === userLoginId;
                            });
                            if(angular.isUndefined(connector_config)){
                                errorMessage = `The ${connectorTitle} connector configuration with the matching user login ID not found.`
                            }
                        }else{ //for single user connector_definition will check default property
                            connector_config = connectorMetaData.configuration.find(function (config) {
                                return config.default;
                            });
                            if(angular.isUndefined(connector_config)){
                                errorMessage = `The default configuration for the ${connectorTitle} connector not found.`
                            }
                        }
                        if(connector_config) {
                            return connectorService.getConnectorHealth(connectorMetaData, connector_config.config_id)
                            .then(function (configurationDetails) {
                                return {
                                    'availability_status': configurationDetails.status === 'Available',
                                    'connector_config': connector_config
                                };
                            });
                        } else {
                            return {'error_message': errorMessage};
                        }
                    } else {
                        return false;
                    }
                })
                .catch(function (error) {
                    console.error("Error in getConnector or getConnectorHealth:", error);
                    return error;
                });
        }

        function poll(_reqConfig, callback) {
            return $interval(function () {
                if (_reqConfig.hasValueReturned) { //check flag before start new call
                    callback(_reqConfig);
                }
                _reqConfig.thresholdValue = _reqConfig.thresholdValue - 1; //Decrease threshold value
                if (_reqConfig.thresholdValue === 0) {
                    stopPoll(_reqConfig); // Stop $interval if it reaches to threshold
                }
            }, _reqConfig.pollInterval);
        }
        function stopPoll(_reqConfig) {
            $interval.cancel(_reqConfig.pollPromise);
            _reqConfig.pollPromise = undefined;
            _reqConfig.thresholdValue = 0; //reset all flags.
            _reqConfig.hasValueReturned = true;
        }

        //If websocket is not active make api calls to check if the playbook execution is completed
        function checkPlaybookExecutionCompletionByPolling(params) {
            var _pollConfig = {
                hasValueReturned: true,
                thresholdValue: 150,
                pollInterval: params.pollInterval || 2000,
                pollPromise: undefined,
                defer: undefined
            };
            _pollConfig.defer = $q.defer();
            _pollConfig.pollPromise = poll(_pollConfig, function (callbackParam) {
                callbackParam.hasValueReturned = false;
                $resource(API.WORKFLOW + 'api/workflows/log_list/?format=json&parent__isnull=True&task_id=' + params.taskId).save({},
                    function (data) {
                        _pollConfig.hasValueReturned = true;
                        if (data['hydra:member'] && data['hydra:member'].length > 0 && (data['hydra:member'][0].status === 'finished' || data['hydra:member'][0].status === 'failed' || data['hydra:member'][0].status === 'terminated')) {
                            stopPoll(callbackParam);
                            _pollConfig.defer.resolve(data['hydra:member'][0]);
                        }
                    },
                    function (error) {
                        statusCodeService(error, true);
                        if (params.callback) {
                            params.callback(error);
                        }
                        stopPoll(callbackParam);
                        _pollConfig.defer.reject(error);
                    });
            });
            if (params.keepPollConfig) {
                playbookPollConfig.push(_pollConfig);
            }
            return _pollConfig.defer.promise;
        }

        //update progress bar using user preference table 
        function checkPlaybookExecutionCompletion(scope){
            var defer = $q.defer();
            getUserPreferences().then(function (response) {
                if (response?.preferences?.genai_metadata?.playbook_gen_progress) {
                    let _percentageCompleted = response.preferences.genai_metadata.playbook_gen_progress; //schema updated as mentioned in connector
                    if(_percentageCompleted === 100){
                        defer.resolve('completed');
                    }
                    else{
                        scope.processCompleted = _percentageCompleted;
                    }
                }   
              });
            return defer.promise;
        }

        //Copy all text to the local storage, without botgeneral
        function copyWithoutBotGeneral(originalData) {
            // Create a deep copy of the original data to avoid modifying the original object
            var copiedData = JSON.parse(JSON.stringify(originalData));
            // Filter out objects with type "botGeneral" from conversationThroughOutApplication except the first one
            copiedData.socAssitantConversation = copiedData.socAssitantConversation.filter(function (message, index) {
                return index === 0 || message.type !== "botGeneral";
            });
            return copiedData;
        }

        function getKeyStoreRecord(queryObject, module) {
            var defer = $q.defer();
            var url = API.QUERY + module;
            $resource(url).save(queryObject, function (response) {
                defer.resolve(response);
            }, function (err) {
                defer.reject(err);
            })
            return defer.promise;
        }

        function getPlaybook(playbookIRI, module) {
            var defer = $q.defer();
            if (playbookService.loadedPlaybookActions && playbookService.loadedPlaybookActions[module]) {
                var _pb = _.find(playbookService.loadedPlaybookActions[module].playbooks, function (pb) {
                    return pb['@id'] === playbookIRI;
                });
                if (_pb) {
                    defer.resolve({ data: _pb });
                    return defer.promise;
                }
            }
            var _playbookId = $filter('getEndPathName')(playbookIRI);
            $http.get(API.BASE + API.WORKFLOWS + _playbookId + '?$relationships=true').then(function (response) {
                defer.resolve(response);
            }, function (error) {
                defer.reject(error);
            });
            return defer.promise;
        }

        function epochToReadable(epochTime) {
            // Convert epoch time to milliseconds
            const milliseconds = epochTime;
        
            // Create a new Date object with the milliseconds
            const date = new Date(milliseconds);
        
            // Extract individual date components
            const year = date.getFullYear();
            const month = ('0' + (date.getMonth() + 1)).slice(-2); // Adding 1 because month is zero-based
            const day = ('0' + date.getDate()).slice(-2);
            const hours = ('0' + date.getHours()).slice(-2);
            const minutes = ('0' + date.getMinutes()).slice(-2);
            const seconds = ('0' + date.getSeconds()).slice(-2);
        
            // Construct the readable date format
            const readableFormat = `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
        
            return readableFormat;
        }


        function getAllPlaybooks(queryObject) {
            var defer = $q.defer();
            var url = API.QUERY + 'workflows';
            $resource(url).save(queryObject, function (response) {
                if (response['hydra:member'] && (response['hydra:member'][0])) {
                    defer.resolve(response['hydra:member'][0]['uuid']);
                }
                else {
                    defer.reject("Playbook Not Found");
                    toaster.error({ body: "Playbook not found" });
                }
            }, function (error) {
                defer.reject(error);
            });
            return defer.promise;
        }

        function constantMessages() {
            return {
                keyStoreProperties: {
                    fortiAIStaticQuestions: {
                        queryParameters: {
                            "sort": [
                                {
                                    "field": "id",
                                    "direction": "ASC",
                                    "_fieldName": "id"
                                }
                            ],
                            "limit": 30,
                            "logic": "AND",
                            "filters": [
                                {
                                    "field": "key",
                                    "operator": "like",
                                    "_operator": "like",
                                    "value": "%fortiai-static-questions%",
                                    "type": "primitive"
                                },
                                {
                                    "sort": [],
                                    "limit": 30,
                                    "logic": "AND",
                                    "filters": []
                                }
                            ],
                            "__selectFields": [
                                "id",
                                "key",
                                "value",
                                "notes",
                                "@id",
                                "@type",
                                "jSONValue"
                            ]
                        },
                        errorMessage: 'fortiai-static-questions key store record not found, refer documentation'
                    },
                    fortiAIConfiguration: {
                        queryParameters: {
                            "sort": [
                                {
                                    "field": "id",
                                    "direction": "ASC",
                                    "_fieldName": "id"
                                }
                            ],
                            "limit": 30,
                            "logic": "AND",
                            "filters": [
                                {
                                    "field": "key",
                                    "operator": "like",
                                    "_operator": "like",
                                    "value": "%fortiai-configurations%",
                                    "type": "primitive"
                                },
                                {
                                    "sort": [],
                                    "limit": 30,
                                    "logic": "AND",
                                    "filters": []
                                }
                            ],
                            "__selectFields": [
                                "id",
                                "key",
                                "value",
                                "notes",
                                "@id",
                                "@type",
                                "jSONValue"
                            ]
                        },
                        errorMessage: 'fortiai-configurations key store record not found, refer documentation'
                    }
                },
                keysToMaskArtifactsInOrder: ['URL', 'Email', 'Domain' , 'Host' , 'IP',  'MD5', 'SHA1', 'SHA256'],
                playbookTags: {
                    pBDesignerSteps: 'aibot-playbookBlockSuggestion'
                },
                botGeneralMessages:{
                    configurationFailedMessage: "Unfortunately, there's not much that can be done at the moment. It seems that the OpenAI integration isn't properly configured or might be encountering issues. We recommend reaching out to your system administrator for assistance.",
                    initialMessage: "Hi there! How can I help you today?",
                    connectorNotConfigured: 'Connector is not configured',
                    defaultConfigurationNotPresent: `Unfortunately, there's not much that can be done at the moment. It seems that the OpenAI integration configuration not set as default. We recommend reaching out to your system administrator for assistance.`,
                    connectorFetchApiFails: 'Error in fetching connector configuration',
                    playbookFailed: "I'm sorry, it seems like there was an issue generating a response. Please try your request again, or refer to the advisor troubleshooting documentation for assistance.",
                    playbookFailedToaster: 'Error in fetching playbook data',
                    proceedingToGenerate: 'Proceeding to generate playbook template.',
                    generatedSuccessfully: 'Playbook template successfully generated! Adding the block now.',
                    buildingDescription: 'Sure, let me build and share a playbook outline to review.',
                    fetchingConversation: 'Loading conversation history',
                    keystorePermissionError: 'The user does not have read permission for the keystore module.',
                    assistantNotFound: 'Assistant not found, run the FortiAI configuration wizard',
                    assistantFailed: 'Failed to fetch assistant',
                    connectorNotAvailable: 'Connector is not available',
                    playbookDesignerInitialMessage: 'Looking to build a playbook? I can help you build a playbook to trigger actions, send alerts, or streamline tasks. What do you want it to do?',
                    connectorInitialMessage: 'Hi there! I can guide you step-by-step to build a connector. Ready to get started?',  
                    similarRecordError:'Similar records could not be loaded, which may affect the GenAI response. Please review the Recommendation Engine settings',
                    pbStepGenerationCompleted:'Playbook Generation completed',
                    badGatewayErrorMessage: 'Due to a technical error, the request has timed out. Please clear the conversation and try again'
                },
                responseProtocol: {
                    recordNavigation: 'fsr_module_navigate',
                    recordCreation: 'fsr_hostname',
                    playbookOutline: 'is_playbook_outline',
                    playbookExecutionLog: 'fsr_new_tab',
                    fileDownload: 'download'
                }
            };
        }

        function getUserLoginId(_uuid){
            return $resource(API.AUTH + 'users?uuid=' + _uuid)
            .get()
            .$promise
            .then(function (userLoginDetails) {
                return userLoginDetails;
            })
            .catch(function (error) {
                console.error('Error:', error);
                throw error; // Rethrow the error to be handled by the caller
            });
        }

        /**
        * @ngdoc service
        * @name fortisoar.aiAssistantService
        * @description
        *
        * The `getUserPreferences` gets thread_id from preferences
        *
        **/
        function getUserPreferences() {
            var defer = $q.defer();
            $http.get(API.BASE + 'userpreference').then(function(response) {
                defer.resolve(response.data);
              }, function(err) {
                defer.reject(err);
            });
            return defer.promise;
        }

        /**
        * @ngdoc service
        * @name fortisoar.aiAssistantService
        * @description
        *
        * The `setUserPreferences` gets thread_id from preferences
        *
        * @param {object} entities openai thread_id
        **/
        function setUserPreferences(params){
            var defer = $q.defer();
            var queryUrl = API.BASE + 'userpreference';
            $http.post(queryUrl, params).then(function (result) {
                if(result){
                    defer.resolve(result);
                }
            },
            function(error){
                if(error){
                    defer.resolve(error);
                }
            });
            return defer.promise;
        }

        //get detail data of the module selected for masking
        function getDetailData(module,record_id,relatedModule,select_fields){
            var defer = $q.defer();
            var _url = API.BASE + module + '/' + record_id + '/' + relatedModule + '?__selectFields=' + select_fields + '&$relationships=true';
            $http.get(_url).then(function(response) {
                defer.resolve(response.data['hydra:member']);
              }, function(err) {
                defer.resolve([]);
            });
            return defer.promise;
        }

        //fetch URL from string to navigate
        function fetchURL(latestMessage){
            let str = latestMessage;
            const match = str.match(/\(https:[^\)]+\)/);
            if (match) {
                const matchString = match[0];
                const textInBrackets = matchString.replace(/[()]/g, '');
                str = textInBrackets;
            }
            return str;
        }

        //check if IRI present
        function getModuleIRI(url) {
            const parts = url.split('/');
            var module = parts[parts.length - 2];
            var id = parts[parts.length - 1];
            return {
                module: module,
                id: id
            };
        }

        //check if UUID is present in navigation url
        function getUUIDFromURL(url){
            const parts = url.split('/');
            const lastPart = parts[parts.length - 1];
            return lastPart;
        }

        //get updated URL for navigations to be replaced in latest message 
        function getUpdatedURL(url,protocol,latestMessage){
            var updatedURL = '';
            var replacementProtocol = 'https://';
            switch (protocol) {
                case 'recordNavigation':
                    replacementProtocol = replacementProtocol + constantMessages().responseProtocol.recordNavigation;
                    updatedURL = latestMessage.replaceAll(replacementProtocol, '');
                    break;
                case 'recordCreation':
                    replacementProtocol = replacementProtocol + constantMessages().responseProtocol.recordCreation;
                    updatedURL = latestMessage.replaceAll(replacementProtocol, '');
                    break;
                case 'playbookExecutionLog':
                    replacementProtocol = replacementProtocol + constantMessages().responseProtocol.playbookExecutionLog;
                    let hrefLink = '<a href="'+ url + '" target="_blank"> <span class="fa fa-external-link" style="color:#20a6af"></span></a>';
                    let messageWithoutProtocol = latestMessage.replace('('+ replacementProtocol + url + ')',hrefLink);
                    updatedURL = messageWithoutProtocol;
                    break;      
                default:
                    break;
            }
            return updatedURL;
        }

        //function returns boolean true if the current uuid is same as present in navigation URL.
        function checkIfRecordOpen(recordIRI,navigateURL){
            let _urlUUID = getUUIDFromURL(navigateURL);
            let _currentRecordUUID = getUUIDFromURL(recordIRI);
            return _currentRecordUUID === _urlUUID;
        }

        function getPastConversationAction(userLoginId, payload){
            var defer = $q.defer();
            executeAction('aiassistant-utils', 'get_past_conversation', userLoginId, payload).then(function (threadList) {
                defer.resolve(threadList);
            }, function(error){
                if(error.status === 400){
                    defer.resolve(error); //error to be handled separately for each response error
                }else{
                    defer.reject(error);
                }
            });
            return defer.promise;
        }
    }
})();