import base64
import unittest
from unittest.mock import patch, Mock

from Crypto.PublicKey import RSA
from freezegun import freeze_time

from cshmac import exceptions
from cshmac.validators import validate_hmac_message

CURRENT_TIME = '2015-12-10 11:59:01'


@freeze_time(CURRENT_TIME)
class TestValidateHmacMessage(unittest.TestCase):
    def setUp(self):
        self.given_time = CURRENT_TIME
        self.public_key = 'pubkey'
        self.private_key = 'privatekey'
        self.full_uri = 'https://dev.venom.cyber/wf/workflow/start/'
        self.request_method = 'POST'
        self.payload = 'I am playbook!'

    def build_auth_header(self,
                          fingerprint='fingerprint'):
        auth_header = '{0};{1};{2};{3}'.format('sha256',
                                               self.given_time,
                                               self.public_key,
                                               fingerprint)
        self.auth_header = 'CS ' + base64.b64encode(
            auth_header.encode()).decode()

    def assert_hmac_failed(self, exception_class):
        with self.assertRaises(exception_class):
            validate_hmac_message(self.auth_header,
                                  self.full_uri,
                                  self.request_method,
                                  self.private_key,
                                  self.payload)

    def setup_key_mocks(self, mock_method, key_1_modulus, key_2_modulus):
        key_1 = Mock()
        key_1.key = Mock()
        key_1.key.n = key_1_modulus

        key_2 = Mock()
        key_2.key = Mock()
        key_2.key.n = key_2_modulus
        mock_method.side_effect = [key_1, key_2]

    def test_failure_if_auth_header_malformed_no_CS(self):
        self.auth_header = 'Basic stuffz'
        self.assert_hmac_failed(exceptions.MalformedAuthorizationHeader)

    def test_failure_garbage_header(self):
        self.auth_header = 'CS what'
        self.assert_hmac_failed(exceptions.MalformedAuthorizationHeader)

    def test_failure_garbage_header_incorrect_padding(self):
        self.auth_header = 'CS huh'
        self.assert_hmac_failed(exceptions.MalformedAuthorizationHeader)

    def test_failure_garbage_header_not_enough_values_to_unpack(self):
        self.auth_header = 'CS ~'
        self.assert_hmac_failed(exceptions.MalformedAuthorizationHeader)

    def test_failure_invalid_time(self):
        self.given_time = '2015-12-10 11:57:01'
        self.build_auth_header()
        self.assert_hmac_failed(exceptions.HmacSignatureExpired)

    def test_failure_modulus_not_matching(self):
        with patch.object(RSA, 'importKey') as mock_method:
            self.build_auth_header()
            self.setup_key_mocks(mock_method, 1, 2)
            self.assert_hmac_failed(exceptions.KeyModulusNotMatching)

    def test_failure_invalid_fingerprint(self):
        with patch.object(RSA, 'importKey') as mock_method:
            self.build_auth_header()
            self.setup_key_mocks(mock_method, 1, 1)
            self.assert_hmac_failed(exceptions.InvalidFingerprint)

    @patch('cshmac.utils.HmacFingerprintBuilder.build')
    def test_success(self, mock_build_func):
        mock_build_func.return_value = 'fingerprint'
        with patch.object(RSA, 'importKey') as mock_method:
            self.build_auth_header()
            self.setup_key_mocks(mock_method, 1, 1)
            self.assertTrue(validate_hmac_message(
                self.auth_header, self.full_uri, self.request_method,
                self.private_key, self.payload))
