import hashlib
import hmac

TIME_FORMAT = '%Y-%m-%d %H:%M:%S'
DEFAULT_ALGORITHM = 'sha256'


class HmacFingerprintBuilder:
    """
    Builds an HMAC fingerprint by generating the following string and hashing
    it using HMAC with SHA256:

    *sha256.{request_method}.{time}.{absolute_uri}.{hashed_payload}*

    Time has the format of: YYYY-mm-dd HH(24-hour):MM:SS

    Used for forming HMAC requests as a client
    as well as (eventually) server side HMAC authentication.
    """

    def __init__(self, private_key, time, full_uri, request_method,
                 payload=''):
        self.private_key = private_key \
            if type(private_key) is bytes else private_key.encode()
        self.full_uri = full_uri
        self.request_method = request_method
        self.payload = payload if type(payload) is bytes else payload.encode()
        self.time = time

    def build(self):
        digest_method = hashlib.new(DEFAULT_ALGORITHM)
        digest_method.update(self.payload)
        hashed_payload = digest_method.hexdigest()

        raw_fingerprint = "{0}.{1}.{2}.{3}.{4}".format(DEFAULT_ALGORITHM,
                                                       self.request_method,
                                                       self.time,
                                                       self.full_uri,
                                                       hashed_payload)

        hashed = hmac.new(self.private_key, raw_fingerprint.encode(),
                          hashlib.sha256)
        return hashed.hexdigest()
