import { DatePipe } from '@angular/common';
import { Injectable, inject } from '@angular/core';
import * as _ from 'lodash-es';
import { first, isObservable, map, switchMap } from 'rxjs';
import { ALL_RECORDS_SIZE, ApiService, DatetimeFloatingFilterComponent, DatetimeGridFilterComponent, EntityService, FIXED_MODULE, LookupFloatingFilterComponent, LookupGridFilterComponent, Query, SystemSettingsService, Tag } from "shared";
import { TagsCellRendererComponent } from '../cell-renderers/tags-cell-renderer.component';
import { TextObservableCellRendererComponent } from '../cell-renderers/text-observable-cell-renderer.component';
import * as i0 from "@angular/core";
export class GridColumnService {
    constructor() {
        this.systemSettingsService = inject(SystemSettingsService);
        this.entityService = inject(EntityService);
        this.apiService = inject(ApiService);
        this.datePipe = inject(DatePipe);
    }
    create(fields, query = new Query(), columns = [], useUserPrefColumns = true, customColDefs = [], defaultSort = []) {
        let columnDefinitions = [];
        let columnList = {};
        let columnOrder = [];
        _.forEach(columns, (column) => {
            let field = fields[column.name];
            let customColDef = customColDefs.find((colDef) => colDef.field === column.name);
            if ((field && !field.isRelationship) || customColDef) {
                columnOrder.push(column.name);
                let _columnDef = customColDef || this.getColumnDefinition(field, column);
                columnList[column.name] = _columnDef;
                columnDefinitions.push(_columnDef);
            }
        });
        // _.forEach(fields, (field: Field) => {
        //   let tempField = cloneDeep(field);
        //   if (tempField && !columnList[tempField.name] && !tempField.isRelationship && ((_.isUndefined(tempField.visibility) || tempField.visibility))) {
        //     if (useUserPrefColumns && columnOrder.length > 0) {
        //       tempField.gridColumn = false;
        //       let columnVisible = _.find(columnOrder, (columnName: string) => { return columnName === tempField.name })
        //       if (columnVisible) {
        //         tempField.gridColumn = true;
        //       }
        //     }
        //     columnDefinitions.push(
        //       this.getColumnDefinition(tempField, new GridColumn({ name: tempField.name }))
        //     );
        //   }
        // });
        columnOrder = _.compact(columnOrder.length > 0 ? columnOrder : _.map(columnDefinitions.filter((definition) => {
            return !definition.hide;
        }), 'field'));
        columnDefinitions = _.sortBy(columnDefinitions, 'title');
        if (query.sort.length > 0) {
            this.setSort(columnDefinitions, query.sort);
        }
        else if (defaultSort) {
            this.setSort(columnDefinitions, defaultSort);
        }
        let orderedColumns = this.orderColumns(columnDefinitions, columnOrder);
        //if (!skipCheckboxes) {
        // orderedColumns.unshift({
        //   headerName: '',
        //   field: 'checkbox',
        //   sortable: false,
        //   resizable: false,
        //   pinned: true,
        //   cellRendererParams: {
        //     suppressCellClick: true
        //   },
        //   suppressHeaderFilterButton: true,
        //   suppressHeaderMenuButton: true,
        //   suppressSizeToFit: true,
        //   floatingFilter: false,
        //   width: 40,
        //   suppressMovable: true,
        //   lockPosition: true,
        //   lockVisible: true,
        //   headerCheckboxSelection: true,
        //   checkboxSelection: true,
        //   showDisabledCheckboxes: true,
        // });
        //}
        return orderedColumns;
    }
    orderColumns(columnsToOrder, orderedColumnNames = []) {
        let sortedColumns = [];
        _.forEach(orderedColumnNames, (columnName) => {
            columnsToOrder.forEach((column) => {
                if (column.field === columnName) {
                    sortedColumns.push(column);
                }
            });
        });
        let extraColumns = _.difference(columnsToOrder, sortedColumns);
        extraColumns = _.sortBy(extraColumns, 'field');
        return _.concat(sortedColumns, extraColumns);
    }
    getColumnDefinition(field, column) {
        let _columnVisible = this.getColumnVisibility(field, column);
        const formTypeMap = {
            datetime: 'dateColumn',
            text: null,
            textarea: null,
            richtext: null,
            number: 'numberColumn',
            lookup: 'lookupColumn',
            select: 'selectColumn',
            tags: 'tagsColumn',
            checkbox: null
        };
        const cellDataTypeMap = {
            datetime: 'date',
            text: null,
            textarea: null,
            richtext: null,
            select: 'select',
            // lookup: 'lookup',
            checkbox: 'boolean'
        };
        const nonSortableTypes = ['tags'];
        let colDef = _.extend({
            hide: !_columnVisible,
            headerName: field.title,
            field: field.name,
            type: formTypeMap[field.formType],
            cellDataType: cellDataTypeMap[field.formType],
            editable: _.isUndefined(column.editable) ? false : column.editable,
            resizable: _.isUndefined(column.resizable) ? true : column.resizable,
            sortable: _.isUndefined(column.sortable) ? nonSortableTypes.indexOf(field.formType) === -1 : column.sortable,
            tooltipValueGetter: _.isUndefined(column.tooltipValueGetter) ? null : column.tooltipValueGetter,
            filterParams: _.isUndefined(column.filterParams) ? {} : column.filterParams,
            mainMenuItems: this.getMainMenuItems(),
            suppressHeaderFilterButton: true,
            // refData: field.mapping,
            headerComponentParams: {
                field: field
            }
        }, field.defaultColDef || {});
        if (!_.isUndefined(column.cellRenderer)) {
            colDef.cellRenderer = column.cellRenderer;
        }
        if (!_.isUndefined(column.cellRendererParams)) {
            colDef.cellRendererParams = column.cellRendererParams;
        }
        if (field.formType === 'lookup') {
            colDef.cellRenderer = TextObservableCellRendererComponent;
        }
        if (field.formType === 'tags') {
            colDef.cellRenderer = TagsCellRendererComponent;
            colDef.wrapText = true;
            colDef.autoHeight = true;
        }
        if (colDef.sortable) {
            colDef.sort = _.isUndefined(column.sort) ? null : column.sort;
        }
        if (!_.isUndefined(column.width)) {
            colDef.width = column.width;
        }
        // if (_column.editable) {
        //   _column.cellEditRenderer = field.fieldComponent;
        // }
        // if (_column.enableFiltering) {
        //   let filterComponent = field.type === 'string' ? field.fieldComponent : InputFieldComponent;
        //   _column.filterCompoment = filterComponent;
        //   let _fieldOptions = new FieldOptions();
        //   _fieldOptions.showLabel = false;
        //   field.validation.required = false;
        //   _column.filterComponentParams = {
        //     field: field,
        //     fieldOptions: _fieldOptions,
        //     fieldClass: field.formType === 'picklist' ? 'form-select-xs grid-picklist' : 'form-control-xs'
        //   }
        // }
        return colDef;
    }
    setSort(columns, sort = []) {
        sort.forEach((sortItem) => {
            const _foundIndex = _.findIndex(columns, (column) => {
                return column.field === sortItem.field;
            });
            if (_foundIndex !== -1) {
                columns[_foundIndex].sort = sortItem.direction.toLowerCase();
            }
        });
    }
    getColumnVisibility(field, column) {
        if (_.isUndefined(column.visible)) {
            return field.gridColumn;
        }
        else {
            return column.visible;
        }
    }
    getMainMenuItems() {
        return ['sortAscending', 'sortDescending', 'sortUnSort', 'separator', 'columnChooser', 'separator', 'pinSubMenu', 'separator', 'autoSizeThis', 'autoSizeAll', 'separator', 'resetColumns'];
    }
    getColumns(columns, visibleColumns = []) {
        let gridColumns = [];
        columns.forEach((column) => {
            gridColumns.push(new GridColumn({
                name: column,
                visible: visibleColumns.length === 0 || visibleColumns.indexOf(column) > -1
            }));
        });
        return gridColumns;
    }
    getDefaultColumnTypes() {
        return {
            numberColumn: { width: 130, filter: 'agNumberColumnFilter' },
            dateColumn: {
                filter: DatetimeGridFilterComponent,
                filterParams: {},
                floatingFilterComponent: DatetimeFloatingFilterComponent,
                suppressFloatingFilterButton: true,
                suppressHeaderFilterButton: true
            },
            lookupColumn: {
                filter: LookupGridFilterComponent,
                filterParams: {},
                suppressFiltersToolPanel: true,
                floatingFilterComponent: LookupFloatingFilterComponent,
                suppressFloatingFilterButton: true,
                suppressHeaderFilterButton: true
            },
            selectColumn: {
                filter: 'agSetColumnFilter',
                filterParams: {
                    suppressMiniFilter: true,
                    keyCreator: (params) => params.value.value,
                    valueFormatter: (params) => params.value.label,
                    values: (params) => {
                        let options = params.colDef.headerComponentParams.field.dataSource.options || [];
                        params.success(options);
                    }
                }
            },
            tagsColumn: {
                filter: 'agSetColumnFilter',
                filterParams: {
                    values: (params) => {
                        let tagsBaseUrl = params.column.getColDef().headerComponentParams?.field.baseUrl;
                        this.entityService.build(FIXED_MODULE.TAGS).pipe(switchMap((tagsEntity) => {
                            tagsEntity.baseUrl = tagsBaseUrl;
                            return this.apiService.getList(tagsEntity, Tag, {
                                limit: ALL_RECORDS_SIZE
                            });
                        }), map((results) => {
                            let options = results.map((tag) => tag.name);
                            params.success(options);
                        })).subscribe();
                    }
                },
            }
        };
    }
    getDefaultDataTypeDefinitions() {
        let systemSettings = this.systemSettingsService.get();
        return {
            date: {
                baseDataType: 'date',
                extendsDataType: 'date',
                valueParser: params => {
                    if (!params.newValue) {
                        return null;
                    }
                    return new Date(params.newValue);
                },
                valueFormatter: (params) => {
                    if (!params.value) {
                        return '';
                    }
                    let date = new Date(params.value);
                    let dateString = this.datePipe.transform(date, systemSettings.preferences.dateTimeFormat);
                    return dateString || '';
                },
            },
            select: {
                baseDataType: 'text',
                extendsDataType: 'text',
                valueParser: params => {
                    return params.newValue;
                },
                valueFormatter: (params) => {
                    if (params.value && params.colDef.headerComponentParams.field) {
                        let foundValue = params.colDef.headerComponentParams.field.dataSource.options.find((option) => option.value === params.value);
                        return foundValue?.label || params.value;
                    }
                    return params.value;
                }
            }
        };
    }
    getDefaultColumnDef() {
        const loadingStateByEntry = new Map();
        const updateTableState = (params) => {
            loadingStateByEntry.set(params.value, false);
            setTimeout(() => {
                params.api.refreshCells({
                    suppressFlash: true,
                    rowNodes: [params.node],
                    columns: [params.column],
                });
            });
        };
        return {
            filter: 'agTextColumnFilter',
            menuTabs: ['generalMenuTab', 'columnsMenuTab'],
            filterParams: {
                maxNumConditions: 1
            },
            floatingFilter: true,
            cellRenderer: null,
            cellClassRules: {
                'grid-table-cell-loading': (params) => {
                    const mapEntry = loadingStateByEntry.get(params.value);
                    if (mapEntry !== undefined) {
                        return mapEntry;
                    }
                    if (isObservable(params.value)) {
                        loadingStateByEntry.set(params.value, true);
                        params.value.pipe(first()).subscribe({
                            next: () => updateTableState(params),
                            error: () => updateTableState(params),
                        });
                        return true;
                    }
                    return false;
                },
            },
        };
    }
    static { this.ɵfac = function GridColumnService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || GridColumnService)(); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: GridColumnService, factory: GridColumnService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(GridColumnService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], null, null); })();
export class GridColumn {
    constructor(params) {
        this.name = '';
        this.resizable = true;
        this.editable = false;
        _.extend(this, params);
    }
}
//# sourceMappingURL=data:application/json;base64,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