import { Injectable, inject } from '@angular/core';
import _ from 'lodash-es';
import { isBoolean, keys } from 'lodash-es';
import { concatMap, map, of } from 'rxjs';
import { DEFAULT_GRID_ROWS, EntityService, Filter, FilterGroup, OPERATOR_MAPPINGS, Query } from "shared";
import * as i0 from "@angular/core";
import * as i1 from "@angular/common/http";
import * as i2 from "shared";
export class GridDataSourceService {
    constructor(http, apiService, dateService, getEndPathNamePipe) {
        this.http = http;
        this.apiService = apiService;
        this.dateService = dateService;
        this.getEndPathNamePipe = getEndPathNamePipe;
        this.entityService = inject(EntityService);
    }
    getDataSource(entity, baseQuery, dataInterceptor, alwaysIncludeColList = [], treeFilterField) {
        let subscription;
        return {
            getRows: (params) => {
                params.api.hideOverlay();
                const availableFieldNames = entity.attributes.filter((field) => field.gridColumn).map((field) => field.name);
                const visibleColIds = params.api.getAllDisplayedColumns().map((col) => col.getColId());
                const visibleColumnFields = availableFieldNames.filter((fieldName) => visibleColIds.indexOf(fieldName) > -1 || alwaysIncludeColList.indexOf(fieldName) > -1);
                let defaultQuery = _.cloneDeep(baseQuery);
                if (treeFilterField) { // for treedata
                    if (!defaultQuery) {
                        defaultQuery = new Query();
                    }
                    let parentId = null;
                    if (params.request.groupKeys.length > 0) {
                        parentId = this.getEndPathNamePipe.transform(params.request.groupKeys.slice(-1)[0]);
                    }
                    defaultQuery?.filters.push(new Filter(treeFilterField, 'eq', parentId));
                }
                subscription = this.loadRows(entity, params.request, defaultQuery, visibleColumnFields).subscribe((response) => {
                    let modifiedData = dataInterceptor ? dataInterceptor(response['hydra:member']) : response['hydra:member'];
                    params.success({
                        rowData: this.loadLookupData(modifiedData, entity),
                        rowCount: response['hydra:totalItems']
                    });
                    if (params.api.paginationGetRowCount() === 0) {
                        params.api.showNoRowsOverlay();
                    }
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    loadLookupData(rows, entity) {
        let lookupFields = entity.attributes.filter((field) => field.formType === 'lookup');
        rows.forEach((row) => {
            lookupFields.forEach((field) => {
                // convert lookups to observables
                const uuid = row[field.name];
                if (!uuid) {
                    row[field.name] = of(field.blankDisplay || '');
                    return;
                }
                row[field.name] = this.entityService.build(field.dataSource.model).pipe(concatMap((subEntity) => this.entityService.getRecordDisplay(subEntity, uuid)));
            });
        });
        // Free up memory
        this.entityService.clearDisplayCache(true);
        return rows;
    }
    getDatasourceByParams(entity, queryParams) {
        let subscription;
        return {
            getRows: (params) => {
                params.api.hideOverlay();
                subscription = this.apiService.getList(entity, undefined, queryParams)
                    .subscribe((rowData) => {
                    params.success({
                        rowData: rowData,
                        // rowCount: 50
                    });
                    if (rowData.length === 0) {
                        params.api.showNoRowsOverlay();
                    }
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    getDatasourceByCustomRequest(request = { url: '', type: 'GET', body: {} }, queryParams, entity, dataInterceptor) {
        let subscription;
        return {
            getRows: (params) => {
                let request$ = request.type === 'GET' ? this.http.get(request.url, { params: queryParams }) : this.http.post(request.url, request.body, { params: queryParams });
                subscription = request$.subscribe((response) => {
                    let modifiedData = dataInterceptor ? dataInterceptor(response['hydra:member']) : response['hydra:member'];
                    params.success({
                        rowData: modifiedData,
                        rowCount: response['hydra:totalItems']
                    });
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    loadRows(entity, request, baseQuery, availableFieldNames = []) {
        let queryParams = {};
        queryParams.offset = request.startRow || 0;
        queryParams.limit = (request.endRow || DEFAULT_GRID_ROWS) - queryParams.offset;
        const query = this.convertAgGridQueryToQueryApi(request.filterModel, request.sortModel, request.groupKeys);
        let fullQuery;
        if (baseQuery) {
            fullQuery = {
                filters: [],
                logic: 'AND'
            };
            if (query.filters.length > 0) {
                fullQuery.filters.push(query.getQueryObject());
            }
            if (baseQuery.filters.length > 0) {
                fullQuery.filters.push(baseQuery.getQueryObject());
            }
            if (query.sort.length) {
                fullQuery.sort = query.sort;
            }
        }
        else {
            fullQuery = query.getQueryObject();
        }
        if (availableFieldNames.length > 0) {
            // UUID is required
            availableFieldNames.push('uuid');
            availableFieldNames = _.uniq(availableFieldNames);
            fullQuery.fields = availableFieldNames;
        }
        if (!fullQuery.sort?.length && entity.defaultSort?.length) {
            fullQuery.sort = entity.defaultSort;
        }
        if (!fullQuery.sort?.length) {
            console.warn(`${entity.title} does not have a default sort specified - this could cause infinite loops.`);
        }
        return this.apiService.query(entity, fullQuery, queryParams).pipe(map((response) => {
            return response;
        }));
    }
    convertAgGridQueryToQueryApi(filterModel, sortModel, groupKeys) {
        let query = new Query();
        keys(filterModel).forEach((fieldName) => {
            let fieldValue = filterModel[fieldName];
            if (fieldName === 'ag-Grid-AutoColumn') {
                if (groupKeys.length > 0) {
                    return; // remove tree column filter while expanding any node.
                }
                fieldName = 'name'; //TODO: this was for the Tree node column filter. mostly it will be name. Handle this gracefully.
            }
            let filter;
            if (fieldValue.type === 'blank') {
                filter = new Filter(fieldName, 'isnull', true);
            }
            else if (fieldValue.type === 'notBlank') {
                filter = new Filter(fieldName, 'isnull', false);
            }
            else if (fieldValue.filterType === 'boolean') {
                // Special boolean case
                if (fieldValue.filter === 'any') {
                    return;
                }
                filter = new Filter(fieldName, 'eq', fieldValue.filter);
            }
            else if (fieldValue.filterType === 'date' && fieldValue.filter) {
                filter = this.dateService.buildDateFilterGroup(fieldValue.filter, fieldName);
            }
            else if (fieldValue.filterType === 'set') {
                if (isBoolean(fieldValue.values[0])) {
                    filter = new Filter(fieldName, 'eq', fieldValue.values[0]);
                }
                else {
                    filter = new Filter(fieldName, 'in', fieldValue.values);
                }
            }
            else if (fieldValue.type === 'inRange') {
                filter = new FilterGroup([
                    new Filter(fieldName, 'lte', fieldValue.filterTo),
                    new Filter(fieldName, 'gte', fieldValue.filter)
                ], 'AND');
            }
            else if (fieldValue.filter) {
                let operator = OPERATOR_MAPPINGS[fieldValue.type] || 'eq';
                filter = new Filter(fieldName, operator, fieldValue.filter);
            }
            else {
                return;
            }
            query.filters.push(filter);
        });
        sortModel.forEach((sortModelItem) => {
            query.sort.push({
                field: sortModelItem.colId,
                direction: sortModelItem.sort === 'asc' ? 'ASC' : 'DESC'
            });
        });
        return query;
    }
    static { this.ɵfac = function GridDataSourceService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || GridDataSourceService)(i0.ɵɵinject(i1.HttpClient), i0.ɵɵinject(i2.ApiService), i0.ɵɵinject(i2.DateService), i0.ɵɵinject(i2.GetEndPathNamePipe)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: GridDataSourceService, factory: GridDataSourceService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(GridDataSourceService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.HttpClient }, { type: i2.ApiService }, { type: i2.DateService }, { type: i2.GetEndPathNamePipe }], null); })();
//# sourceMappingURL=data:application/json;base64,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