import { HttpContext } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { plainToInstance } from 'class-transformer';
import _ from 'lodash-es';
import { catchError, map, of, tap } from 'rxjs';
import { ALL_RECORDS_SIZE, API, FIXED_MODULE, IGNORE_ERRORS, UUID_REGEX } from '../constants';
import { Agent } from '../models/agent';
import { Connector } from '../models/connector';
import { ConnectorConfiguration } from '../models/connector-configuration';
import * as i0 from "@angular/core";
import * as i1 from "@angular/common/http";
import * as i2 from "./utility.service";
export class ConnectorService {
    constructor(http, utilityService) {
        this.http = http;
        this.utilityService = utilityService;
        this.cache = {};
    }
    // Connector Operations
    getConnector(name, version, agent, ignoreErrors = true) {
        let params = {};
        if (agent) {
            params.agent = agent;
        }
        const api = `${API.INTEGRATIONS}connectors/${name}/${version}/?format=json`;
        return this.http.post(api, {}, { params: params, context: new HttpContext().set(IGNORE_ERRORS, ignoreErrors) }).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    updateConnector(id, data = {}) {
        const api = `${API.INTEGRATIONS}connectors/${id}/?format=json`;
        return this.http.put(api, data).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    deleteConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/delete/files/?format=json`;
        return this.http.delete(api);
    }
    // Connector Config Operatons
    createConnectorConfig(data) {
        const api = `${API.INTEGRATIONS}configuration/?format=json`;
        return this.http.post(api, data).pipe(map((response) => {
            return plainToInstance(ConnectorConfiguration, response);
        }));
    }
    updateConnectorConfig(id, data) {
        const api = `${API.INTEGRATIONS}configuration/${id}/?format=json`;
        return this.http.put(api, data).pipe(map((response) => {
            return plainToInstance(ConnectorConfiguration, response);
        }));
    }
    deleteConnectorConfig(id) {
        const api = `${API.INTEGRATIONS}configuration/${id}/`;
        return this.http.delete(api);
    }
    // Development Connector
    getDevelopmentConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/?format=json`;
        return this.http.post(api, {}).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    createDevelopmentConnector(data = {}) {
        const api = `${API.INTEGRATIONS}connector/development/entity/?format=json`;
        return this.http.post(api, data).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    // Connector File Oeprations
    getConnectorFile(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/files/?format=json`;
        return this.http.post(api, {});
    }
    getConnectorFileContent(id, filepath) {
        const api = API.INTEGRATIONS + 'connector/development/entity/' + id + '/files' + '/?format=json';
        return this.http.post(api, { 'xpath': filepath });
    }
    createConnectorFile(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/files/?format=json`;
        return this.http.post(api, { xpath: data });
    }
    updateConnectorFile(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/rename/files/?format=json`;
        return this.http.post(api, data);
    }
    deleteConnectorFile(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/delete/files/?format=json`;
        return this.http.post(api, data);
    }
    // Connector folder operations
    createConnectorFolder(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/folders/?format=json`;
        return this.http.put(api, data).pipe(map((response) => {
            return response.data;
        }));
    }
    deleteConnectorFolder(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/delete/folders/?format=json`;
        return this.http.post(api, data);
    }
    getConnectorHealth(connector, config_id, agent) {
        let params = {
            config_id: config_id,
            connector_name: connector.name,
            connector_version: connector.version,
            agent: agent
        };
        const api = `${API.INTEGRATIONS}healthcheck/`;
        return this.http.post(api, params);
    }
    getConnectorOutputSchema(connectorName, connectorVersion, operation, config, params, agent) {
        const key = `${connectorName}/${connectorVersion}/${operation}`;
        if (this.cache[key] && !this.cache[key].error) {
            // TODO: Ensure this doesn't increase browser memory too much
            return of(this.cache[key]);
        }
        let queryParams = {};
        if (agent) {
            queryParams = { agent: agent };
        }
        const api = `${API.INTEGRATIONS}connector_output_schema/${connectorName}/${connectorVersion}/`;
        const data = {
            'operation': operation,
            'config': config,
            'params': params
        };
        return this.http.post(api, data, { params: queryParams, context: new HttpContext().set(IGNORE_ERRORS, true) }).pipe(catchError((error) => {
            console.warn(error);
            return of({
                data: {
                    output_schema: {}
                },
                error: true
            });
        }), tap((response) => {
            this.cache[key] = response;
        }));
        // Pending error handling
    }
    // Maybe common for dev and installed? or does not matter?
    executeConnectorAction(connector, operation, configuration, params, audit, auditInfo, agent) {
        const api = `${API.INTEGRATIONS}execute/?format=json`;
        let requestParams = {
            connector: connector.name,
            version: connector.version,
            config: configuration.config,
            operation: operation.operation,
            params: params,
            audit: audit,
            audit_info: auditInfo
        };
        if (agent) {
            requestParams.agent = agent;
        }
        return this.http.post(api, requestParams, {
            context: new HttpContext().set(IGNORE_ERRORS, true)
        });
    }
    // Maybe common for dev and installed? or does not matter?
    exportDevelopedConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/export/?format=json`;
        return this.http.get(api, { responseType: 'arraybuffer' });
        // Pending error handling
    }
    // Below is a list of pending functions - need to figure out if some can be consolidated together
    // publishConnector
    // installConnector
    // getAgents
    // getIngestionPlaybookCollectionUUID
    // getSamplePlaybooks
    // getDataSources
    // loadConnectors - this is dev or installed ?
    // loadByTags
    // getConfiguredConnectorList
    // getConnectorActionDetails
    // getConfiguredConnectors
    // getConnectorOutputSchema
    // getCategories
    // getAnnotations
    // getAnnotationConnectors
    // applyRolesToActions
    // downloadHelpPDF
    getDependencyStatus(connector) {
        const api = `${API.INTEGRATIONS}connectors/dependencies_check/${connector.name}/${connector.version}/?format=json`;
        return this.http.get(api).pipe(map((response) => {
            return response;
        }));
    }
    getFieldData(data) {
        const api = `${API.INTEGRATIONS}execute/?format=json`;
        return this.http.request('POST', api, data);
    }
    // installDependencies
    filterConnectorOperations(operation, installed) {
        installed = (!this.utilityService.isUndefined(installed) && _.isBoolean(installed)) ? installed : true;
        if (installed) {
            return (operation.enabled && (this.utilityService.isUndefined(operation.visible) || operation.visible === true));
        }
        return this.utilityService.isUndefined(operation.visible) || operation.visible === true;
    }
    getDevelopedConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/?format=json`;
        return this.http.post(api, {});
    }
    //Get Connector Template
    getConnectorTemplates(data) {
        const api = `${API.INTEGRATIONS}connector/development/templates/?format=json`;
        return this.http.post(api, data);
    }
    //Create New connector
    createConnector(connectorConfig) {
        const api = `${API.INTEGRATIONS}connector/development/entity/?format=json`;
        return this.http.post(api, connectorConfig);
    }
    loadConnectors(params = {}) {
        params.page_size = ALL_RECORDS_SIZE;
        return this.http.get(`${API.INTEGRATIONS}connectors/`, { params: params }).pipe(map((response) => {
            return plainToInstance(Connector, response['hydra:member']);
        }));
    }
    //get agents in connector configs
    getAgents(connector, installed) {
        let config = {
            'active': true
        };
        if (installed) {
            config.installed = true;
        }
        // versionService.getIntergrationVersionDetails().then(function (integrationVersionDetails) {
        return this.http.get(`${API.INTEGRATIONS}${FIXED_MODULE.CONNECTORS}/${FIXED_MODULE.AGENTS}/${connector.name}/`, { params: config }).pipe(map((installedAgents) => {
            if (installedAgents['hydra:member'].length > 0) {
                //TO DO: Agent compatibilty
                //installedAgents['hydra:member'].forEach((agent: any) => {
                //var minimumCompatibilityVersion = integrationVersionDetails.minimum_compatibility_version.split('-')[0];
                //agent.isIncompatible = parseFloat(agent.agent_version.replace(/\./g, '')) >= parseFloat(minimumCompatibilityVersion.replace(/\./g, '')) ? false : true;
                //});
            }
            return plainToInstance(Agent, installedAgents['hydra:member']);
        }));
    }
    getConfiguredAgents() {
        return this.http.get(`${API.INTEGRATIONS}${FIXED_MODULE.AGENTS}/`).pipe(map((response) => {
            return response['hydra:member'];
        }));
    }
    installConnectorOnAgents(name, version, queryParams = {}, payload) {
        queryParams.replace = true;
        queryParams.trust = true;
        return this.http.post(`${API.INTEGRATIONS}import-connector/${name}/${version}/`, payload, { params: queryParams });
    }
    getConnectorConfigurations(name, version, queryparams = {}) {
        return this.http.get(`${API.INTEGRATIONS}configuration/${name}/${version}/`, { params: queryparams });
    }
    getConfigurationDetail(uuid) {
        return this.http.get(`${API.INTEGRATIONS}configuration/${uuid}/`);
    }
    getConfigId(configurations, configId) {
        const defaultConfig = _.find(configurations, (config) => {
            return config.default;
        });
        let config_id = '';
        let isJinja = false;
        if (configId && configId.replace(/'/g, '').length > 0) {
            if (UUID_REGEX.test(configId) || /^[0-9a-f]{32}$/.test(configId)) {
                let foundConfig = _.find(configurations, (config) => {
                    return config.config_id === configId;
                });
                if (foundConfig) {
                    config_id = configId;
                }
                else if (defaultConfig) {
                    config_id = defaultConfig.config_id;
                }
            }
            else {
                isJinja = true;
                config_id = configId;
            }
        }
        else if (configurations.length > 0) {
            if (defaultConfig) {
                config_id = defaultConfig.config_id;
            }
            else if (configurations.length === 1) {
                configId = configurations[0].config_id;
            }
        }
        return { config_id, isJinja };
    }
    static { this.ɵfac = function ConnectorService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || ConnectorService)(i0.ɵɵinject(i1.HttpClient), i0.ɵɵinject(i2.UtilityService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: ConnectorService, factory: ConnectorService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(ConnectorService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.HttpClient }, { type: i2.UtilityService }], null); })();
//# sourceMappingURL=data:application/json;base64,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