import { Injectable } from '@angular/core';
import { difference, intersection, keys } from 'lodash-es';
import { BehaviorSubject, concatMap, filter, of, tap } from 'rxjs';
import * as i0 from "@angular/core";
import * as i1 from "./entity.service";
import * as i2 from "./websocket.service";
export class DataProviderService {
    constructor(entityService, websocketService) {
        this.entityService = entityService;
        this.websocketService = websocketService;
        this.topics$ = {};
        this.websocketTopics$ = {};
    }
    clear() {
        Object.keys(this.topics$).forEach((topic) => {
            if (this.topics$[topic]) {
                this.topics$[topic].complete();
                delete this.topics$[topic];
            }
        });
        Object.keys(this.websocketTopics$).forEach((topic) => {
            if (this.websocketTopics$[topic]) {
                this.websocketTopics$[topic].unsubscribe();
                delete this.websocketTopics$[topic];
            }
        });
    }
    observe(entity, uuid, relatedField, queryObject) {
        let topic = `${entity.type}${uuid ? '/' + uuid : ''}${relatedField ? '/' + relatedField : ''}`;
        if (!this.websocketTopics$[topic]) {
            this.websocketTopics$[topic] = this.websocketService.websocketSubscribe('/topic/' + topic).pipe(concatMap((message) => this.handleWebsocketUpdates(message, topic))).subscribe();
        }
        let selectedFields = queryObject?.selectedFields || ['uuid'];
        if (!this.topics$[topic]) {
            this.topics$[topic] = new BehaviorSubject({
                entity: entity,
                changedFields: selectedFields,
                sourceId: 'entityCache'
            });
        }
        entity = this.topics$[topic].value.entity;
        let newFields = difference(selectedFields ? selectedFields : [], keys(entity.values || {}));
        entity.uuid = uuid;
        return this.getData(topic, entity, newFields).pipe(concatMap(() => this.topics$[topic].pipe(filter((entityChange) => !queryObject?.selectedFields || intersection(entityChange.changedFields, queryObject?.selectedFields).length > 0))));
    }
    save(entityChange) {
        let topic = `${entityChange.entity.type}${entityChange.entity.uuid ? '/' + entityChange.entity.uuid : ''}`;
        return this.entityService.save(entityChange.entity, entityChange.changedFields).pipe(tap((entity) => {
            this.topics$[topic].next({
                entity: entity,
                changedFields: entityChange.changedFields,
                sourceId: 'entityCache'
            });
        }));
    }
    handleWebsocketUpdates(message, topic) {
        if (message.websocketSessionID !== this.websocketService.websocketSessionID && this.topics$[topic].value) {
            let entity = this.topics$[topic].value?.entity;
            let interestedFields = intersection(keys(this.topics$[topic].value?.entity.values || {}), message.data[0].changedata);
            if (interestedFields.length > 0) {
                return this.getData(topic, entity, interestedFields);
            }
        }
        return of(null);
    }
    getData(topic, entity, selectedFields = []) {
        if (!selectedFields.length) {
            return of(entity);
        }
        return this.entityService.hydrate(entity, selectedFields).pipe(tap((updatedEntity) => {
            this.topics$[topic].next({
                entity: updatedEntity,
                changedFields: selectedFields,
                sourceId: 'entityCache'
            });
        }));
    }
    static { this.ɵfac = function DataProviderService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || DataProviderService)(i0.ɵɵinject(i1.EntityService), i0.ɵɵinject(i2.WebsocketService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: DataProviderService, factory: DataProviderService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(DataProviderService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.EntityService }, { type: i2.WebsocketService }], null); })();
//# sourceMappingURL=data:application/json;base64,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