import { Injectable } from '@angular/core';
import _ from 'lodash-es';
import { concatMap, map, of } from 'rxjs';
import { ALL_RECORDS_SIZE, API } from '../constants';
import { Query } from '../../query-filters/models/query';
import { Filter } from '../../query-filters/models/filter';
import * as i0 from "@angular/core";
import * as i1 from "../../utility/pipes/get-end-path-name.pipe";
import * as i2 from "@angular/common/http";
import * as i3 from "./entity.service";
export class ExportService {
    constructor(getEndPathNamePipe, http, entityService) {
        this.getEndPathNamePipe = getEndPathNamePipe;
        this.http = http;
        this.entityService = entityService;
    }
    /**
      * @ngdoc method
      * @name xf-ui#exportService#loadRowsForExport
      * @methodOf exportService
      *
      * @description
      * Loads exported versions of the given rows from CH.
      *
      * @param {array} rows An array of entity objects to export. Each object must have '@id' key.
      * @param {string} entityType The name of the entity (workflow).
      * @returns {Promise} Promise that will resolve with the exported hydra response
      **/
    loadRowsForExport(rows, entityType, selectedFields, skipExportFlag = false, includeVersions, childEntityName) {
        if (rows.length === 0) {
            return of([]);
        }
        let ids = rows.map((row) => {
            return row['@id'] ? this.getEndPathNamePipe.transform(row['@id']) : row.uuid;
        });
        let operator = rows.length > 1 ? 'in' : 'eq';
        let queryBody = new Query({
            logic: 'AND',
            filters: [new Filter('uuid', operator, ids.join('|'))]
        });
        if (selectedFields) {
            queryBody.selectedFields = selectedFields;
        }
        if (entityType === 'workflows' || entityType === 'workflow_collections') {
            queryBody.ignoreFields = ['createDate', 'createUser', 'modifyDate', 'modifyUser'];
        }
        let queryParams = {
            $export: !skipExportFlag,
            limit: ALL_RECORDS_SIZE
        };
        if (includeVersions) {
            queryParams.$versions = true;
        }
        if (!childEntityName) {
            queryParams.$relationships = true;
        }
        return this.entityService.build(entityType).pipe(concatMap((entity) => {
            return this.http.post(`${API.QUERY}${entityType}`, queryBody.getQueryObject(), { params: queryParams }).pipe(concatMap((data) => {
                if (!childEntityName) {
                    return of(data['hydra:member']);
                }
                const collectionUuids = _.map(data['hydra:member'], 'uuid');
                let childQueryBody = new Query({
                    logic: 'AND',
                    filters: [new Filter('collection.uuid', 'in', collectionUuids.join('|'))]
                });
                let childQueryParams = {
                    $relationships: true,
                    $export: !skipExportFlag,
                    limit: ALL_RECORDS_SIZE
                };
                if (includeVersions) {
                    childQueryParams.$versions = true;
                }
                if (childEntityName && childEntityName === 'workflows') {
                    childQueryBody.ignoreFields = ['createDate', 'createUser', 'modifyDate', 'modifyUser'];
                }
                return this.http.post(`${API.QUERY}${childEntityName}`, childQueryBody.getQueryObject(), { params: childQueryParams }).pipe(map((childData) => {
                    data['hydra:member'].forEach((collection) => {
                        let childItems = _.filter(childData['hydra:member'], (childItem) => {
                            return this.getEndPathNamePipe.transform(childItem.collection) === collection.uuid;
                        });
                        collection[childEntityName] = childItems;
                    });
                    return data['hydra:member'];
                }));
            }));
        }));
    }
    static { this.ɵfac = function ExportService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || ExportService)(i0.ɵɵinject(i1.GetEndPathNamePipe), i0.ɵɵinject(i2.HttpClient), i0.ɵɵinject(i3.EntityService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: ExportService, factory: ExportService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(ExportService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.GetEndPathNamePipe }, { type: i2.HttpClient }, { type: i3.EntityService }], null); })();
//# sourceMappingURL=data:application/json;base64,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