/*
MIT License

Copyright (c) 2022 Ephox Corporation DBA Tiny Technologies, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
 */
import { fromEvent, takeUntil } from 'rxjs';
import { validEvents } from '../html-editor-module/Events';
// Caretaker note: `fromEvent` supports passing JQuery-style event targets, the editor has `on` and `off` methods which
// will be invoked upon subscription and teardown.
const listenTinyMCEEvent = (editor, eventName, destroy$) => fromEvent(editor, eventName).pipe(takeUntil(destroy$));
const bindHandlers = (ctx, editor, destroy$) => {
    const allowedEvents = getValidEvents(ctx);
    allowedEvents.forEach((eventName) => {
        const eventEmitter = ctx[eventName];
        listenTinyMCEEvent(editor, eventName.substring(2), destroy$).subscribe((event) => {
            // Caretaker note: `ngZone.run()` runs change detection since it notifies the forked Angular zone that it's
            // being re-entered. We don't want to run `ApplicationRef.tick()` if anyone listens to the specific event
            // within the template. E.g. if the `onSelectionChange` is not listened within the template like:
            // `<editor (onSelectionChange)="..."></editor>`
            // then it won't be "observed", and we won't run "dead" change detection.
            if (isObserved(eventEmitter)) {
                ctx.ngZone.run(() => eventEmitter.emit({ event, editor }));
            }
        });
    });
};
const getValidEvents = (ctx) => {
    const ignoredEvents = parseStringProperty(ctx.ignoreEvents, []);
    const allowedEvents = parseStringProperty(ctx.allowedEvents, validEvents).filter((event) => validEvents.includes(event) && !ignoredEvents.includes(event));
    return allowedEvents;
};
const parseStringProperty = (property, defaultValue) => {
    if (typeof property === 'string') {
        return property.split(',').map((value) => value.trim());
    }
    if (Array.isArray(property)) {
        return property;
    }
    return defaultValue;
};
let unique = 0;
const uuid = (prefix) => {
    const date = new Date();
    const time = date.getTime();
    const random = Math.floor(Math.random() * 1000000000);
    unique++;
    return prefix + '_' + random + unique + String(time);
};
const isTextarea = (element) => typeof element !== 'undefined' && element.tagName.toLowerCase() === 'textarea';
const normalizePluginArray = (plugins) => {
    if (typeof plugins === 'undefined' || plugins === '') {
        return [];
    }
    return Array.isArray(plugins) ? plugins : plugins.split(' ');
};
const mergePlugins = (initPlugins, inputPlugins) => normalizePluginArray(initPlugins).concat(normalizePluginArray(inputPlugins));
// eslint-disable-next-line @typescript-eslint/no-empty-function
const noop = () => { };
const isNullOrUndefined = (value) => value === null || value === undefined;
const isObserved = (o) => 
// RXJS is making the `observers` property internal in v8. So this is intended as a backwards compatible way of
// checking if a subject has observers.
o.observed || o.observers?.length > 0;
export { listenTinyMCEEvent, bindHandlers, uuid, isTextarea, normalizePluginArray, mergePlugins, noop, isNullOrUndefined };
//# sourceMappingURL=data:application/json;base64,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