import { DatePipe } from '@angular/common';
import { Injectable, inject } from '@angular/core';
import * as _ from 'lodash-es';
import { first, isObservable, map, switchMap } from 'rxjs';
import { ALL_RECORDS_SIZE, ApiService, DatetimeFloatingFilterComponent, DatetimeGridFilterComponent, EntityService, FIXED_MODULE, LookupFloatingFilterComponent, LookupGridFilterComponent, Query, SystemSettingsService, Tag } from "shared";
import { TagsCellRendererComponent } from '../cell-renderers/tags-cell-renderer.component';
import { TextObservableCellRendererComponent } from '../cell-renderers/text-observable-cell-renderer.component';
import * as i0 from "@angular/core";
export class GridColumnService {
    constructor() {
        this.systemSettingsService = inject(SystemSettingsService);
        this.entityService = inject(EntityService);
        this.apiService = inject(ApiService);
        this.datePipe = inject(DatePipe);
    }
    create(fields, query = new Query(), columns = [], useUserPrefColumns = true, customColDefs = [], defaultSort = []) {
        let columnDefinitions = [];
        let columnList = {};
        let columnOrder = [];
        _.forEach(columns, (column) => {
            let field = fields[column.name];
            let customColDef = customColDefs.find((colDef) => colDef.field === column.name);
            if ((field && !field.isRelationship) || customColDef) {
                columnOrder.push(column.name);
                let _columnDef = customColDef || this.getColumnDefinition(field, column);
                columnList[column.name] = _columnDef;
                columnDefinitions.push(_columnDef);
            }
        });
        // _.forEach(fields, (field: Field) => {
        //   let tempField = cloneDeep(field);
        //   if (tempField && !columnList[tempField.name] && !tempField.isRelationship && ((_.isUndefined(tempField.visibility) || tempField.visibility))) {
        //     if (useUserPrefColumns && columnOrder.length > 0) {
        //       tempField.gridColumn = false;
        //       let columnVisible = _.find(columnOrder, (columnName: string) => { return columnName === tempField.name })
        //       if (columnVisible) {
        //         tempField.gridColumn = true;
        //       }
        //     }
        //     columnDefinitions.push(
        //       this.getColumnDefinition(tempField, new GridColumn({ name: tempField.name }))
        //     );
        //   }
        // });
        columnOrder = _.compact(columnOrder.length > 0 ? columnOrder : _.map(columnDefinitions.filter((definition) => {
            return !definition.hide;
        }), 'field'));
        columnDefinitions = _.sortBy(columnDefinitions, 'title');
        if (query.sort.length > 0) {
            this.setSort(columnDefinitions, query.sort);
        }
        else if (defaultSort) {
            this.setSort(columnDefinitions, defaultSort);
        }
        let orderedColumns = this.orderColumns(columnDefinitions, columnOrder);
        //if (!skipCheckboxes) {
        // orderedColumns.unshift({
        //   headerName: '',
        //   field: 'checkbox',
        //   sortable: false,
        //   resizable: false,
        //   pinned: true,
        //   cellRendererParams: {
        //     suppressCellClick: true
        //   },
        //   suppressHeaderFilterButton: true,
        //   suppressHeaderMenuButton: true,
        //   suppressSizeToFit: true,
        //   floatingFilter: false,
        //   width: 40,
        //   suppressMovable: true,
        //   lockPosition: true,
        //   lockVisible: true,
        //   headerCheckboxSelection: true,
        //   checkboxSelection: true,
        //   showDisabledCheckboxes: true,
        // });
        //}
        return orderedColumns;
    }
    orderColumns(columnsToOrder, orderedColumnNames = []) {
        let sortedColumns = [];
        _.forEach(orderedColumnNames, (columnName) => {
            columnsToOrder.forEach((column) => {
                if (column.field === columnName) {
                    sortedColumns.push(column);
                }
            });
        });
        let extraColumns = _.difference(columnsToOrder, sortedColumns);
        extraColumns = _.sortBy(extraColumns, 'field');
        return _.concat(sortedColumns, extraColumns);
    }
    getColumnDefinition(field, column) {
        let _columnVisible = this.getColumnVisibility(field, column);
        const formTypeMap = {
            datetime: 'dateColumn',
            text: null,
            textarea: null,
            richtext: null,
            number: 'numberColumn',
            lookup: 'lookupColumn',
            select: 'selectColumn',
            tags: 'tagsColumn',
            checkbox: null
        };
        const cellDataTypeMap = {
            datetime: 'date',
            text: null,
            textarea: null,
            richtext: null,
            select: 'select',
            // lookup: 'lookup',
            checkbox: 'boolean'
        };
        const nonSortableTypes = ['tags'];
        let colDef = _.extend({
            hide: !_columnVisible,
            headerName: field.title,
            field: field.name,
            type: formTypeMap[field.formType],
            cellDataType: cellDataTypeMap[field.formType],
            editable: _.isUndefined(column.editable) ? false : column.editable,
            resizable: _.isUndefined(column.resizable) ? true : column.resizable,
            sortable: _.isUndefined(column.sortable) ? nonSortableTypes.indexOf(field.formType) === -1 : column.sortable,
            tooltipValueGetter: _.isUndefined(column.tooltipValueGetter) ? null : column.tooltipValueGetter,
            filterParams: _.isUndefined(column.filterParams) ? {} : column.filterParams,
            mainMenuItems: this.getMainMenuItems(column.sortable),
            suppressHeaderFilterButton: true,
            // refData: field.mapping,
            headerComponentParams: {
                field: field
            }
        }, field.defaultColDef || {});
        if (!_.isUndefined(column.cellRenderer)) {
            colDef.cellRenderer = column.cellRenderer;
        }
        if (!_.isUndefined(column.cellRendererParams)) {
            colDef.cellRendererParams = column.cellRendererParams;
        }
        if (field.formType === 'lookup') {
            colDef.cellRenderer = TextObservableCellRendererComponent;
        }
        if (field.formType === 'tags') {
            colDef.cellRenderer = TagsCellRendererComponent;
            colDef.wrapText = true;
            colDef.autoHeight = true;
        }
        if (colDef.sortable) {
            colDef.sort = _.isUndefined(column.sort) ? null : column.sort;
        }
        if (!_.isUndefined(column.width)) {
            colDef.width = column.width;
        }
        // if (_column.editable) {
        //   _column.cellEditRenderer = field.fieldComponent;
        // }
        // if (_column.enableFiltering) {
        //   let filterComponent = field.type === 'string' ? field.fieldComponent : InputFieldComponent;
        //   _column.filterCompoment = filterComponent;
        //   let _fieldOptions = new FieldOptions();
        //   _fieldOptions.showLabel = false;
        //   field.validation.required = false;
        //   _column.filterComponentParams = {
        //     field: field,
        //     fieldOptions: _fieldOptions,
        //     fieldClass: field.formType === 'picklist' ? 'form-select-xs grid-picklist' : 'form-control-xs'
        //   }
        // }
        return colDef;
    }
    setSort(columns, sort = []) {
        sort.forEach((sortItem) => {
            const _foundIndex = _.findIndex(columns, (column) => {
                return column.field === sortItem.field;
            });
            if (_foundIndex !== -1) {
                columns[_foundIndex].sort = sortItem.direction.toLowerCase();
            }
        });
    }
    getColumnVisibility(field, column) {
        if (_.isUndefined(column.visible)) {
            return field.gridColumn;
        }
        else {
            return column.visible;
        }
    }
    getMainMenuItems(sortable = true) {
        const sortMenuItems = sortable ? ['sortAscending', 'sortDescending', 'sortUnSort', 'separator'] : [];
        return [...sortMenuItems, 'columnChooser', 'separator', 'pinSubMenu', 'separator', 'autoSizeThis', 'autoSizeAll', 'separator', 'resetColumns'];
    }
    getColumns(columns, visibleColumns = []) {
        let gridColumns = [];
        columns.forEach((column) => {
            gridColumns.push(new GridColumn({
                name: column,
                visible: visibleColumns.length === 0 || visibleColumns.indexOf(column) > -1
            }));
        });
        return gridColumns;
    }
    getDefaultColumnTypes() {
        return {
            numberColumn: { width: 130, filter: 'agNumberColumnFilter' },
            dateColumn: {
                filter: DatetimeGridFilterComponent,
                filterParams: {},
                floatingFilterComponent: DatetimeFloatingFilterComponent,
                suppressFloatingFilterButton: true,
                suppressHeaderFilterButton: true
            },
            lookupColumn: {
                filter: LookupGridFilterComponent,
                filterParams: {},
                suppressFiltersToolPanel: true,
                floatingFilterComponent: LookupFloatingFilterComponent,
                suppressFloatingFilterButton: true,
                suppressHeaderFilterButton: true
            },
            selectColumn: {
                filter: 'agSetColumnFilter',
                filterParams: {
                    suppressMiniFilter: true,
                    keyCreator: (params) => params.value.value,
                    valueFormatter: (params) => params.value.label,
                    values: (params) => {
                        let options = params.colDef.headerComponentParams.field.dataSource.options || [];
                        params.success(options);
                    }
                }
            },
            tagsColumn: {
                filter: 'agSetColumnFilter',
                filterParams: {
                    values: (params) => {
                        let tagsBaseUrl = params.column.getColDef().headerComponentParams?.field.baseUrl;
                        this.entityService.build(FIXED_MODULE.TAGS).pipe(switchMap((tagsEntity) => {
                            tagsEntity.baseUrl = tagsBaseUrl;
                            return this.apiService.getList(tagsEntity, Tag, {
                                limit: ALL_RECORDS_SIZE
                            });
                        }), map((results) => {
                            let options = results.map((tag) => tag.name);
                            params.success(options);
                        })).subscribe();
                    }
                },
            }
        };
    }
    getDefaultDataTypeDefinitions() {
        let systemSettings = this.systemSettingsService.get();
        return {
            date: {
                baseDataType: 'date',
                extendsDataType: 'date',
                valueParser: params => {
                    if (!params.newValue) {
                        return null;
                    }
                    return new Date(params.newValue);
                },
                valueFormatter: (params) => {
                    if (!params.value) {
                        return '';
                    }
                    let date = new Date(params.value);
                    let dateString = this.datePipe.transform(date, systemSettings.preferences.dateTimeFormat);
                    return dateString || '';
                },
            },
            select: {
                baseDataType: 'text',
                extendsDataType: 'text',
                valueParser: params => {
                    return params.newValue;
                },
                valueFormatter: (params) => {
                    if (params.value && params.colDef.headerComponentParams.field) {
                        let foundValue = params.colDef.headerComponentParams.field.dataSource.options.find((option) => option.value === params.value);
                        return foundValue?.label || params.value;
                    }
                    return params.value;
                }
            }
        };
    }
    getDefaultColumnDef() {
        const loadingStateByEntry = new Map();
        const updateTableState = (params) => {
            loadingStateByEntry.set(params.value, false);
            setTimeout(() => {
                params.api.refreshCells({
                    suppressFlash: true,
                    rowNodes: [params.node],
                    columns: [params.column],
                });
            });
        };
        return {
            filter: 'agTextColumnFilter',
            menuTabs: ['generalMenuTab', 'columnsMenuTab'],
            filterParams: {
                maxNumConditions: 1
            },
            floatingFilter: true,
            cellRenderer: null,
            cellClassRules: {
                'grid-table-cell-loading': (params) => {
                    const mapEntry = loadingStateByEntry.get(params.value);
                    if (mapEntry !== undefined) {
                        return mapEntry;
                    }
                    if (isObservable(params.value)) {
                        loadingStateByEntry.set(params.value, true);
                        params.value.pipe(first()).subscribe({
                            next: () => updateTableState(params),
                            error: () => updateTableState(params),
                        });
                        return true;
                    }
                    return false;
                },
            },
        };
    }
    static { this.ɵfac = function GridColumnService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || GridColumnService)(); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: GridColumnService, factory: GridColumnService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(GridColumnService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], null, null); })();
export class GridColumn {
    constructor(params) {
        this.name = '';
        this.resizable = true;
        this.editable = false;
        _.extend(this, params);
    }
}
//# sourceMappingURL=data:application/json;base64,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