import { Injectable, inject } from '@angular/core';
import _ from 'lodash-es';
import { isBoolean, keys } from 'lodash-es';
import { concatMap, map, of } from 'rxjs';
import { DEFAULT_GRID_ROWS, EntityService, Filter, FilterGroup, OPERATOR_MAPPINGS, Query } from "shared";
import * as i0 from "@angular/core";
import * as i1 from "@angular/common/http";
import * as i2 from "shared";
export class GridDataSourceService {
    constructor(http, apiService, dateService, getEndPathNamePipe) {
        this.http = http;
        this.apiService = apiService;
        this.dateService = dateService;
        this.getEndPathNamePipe = getEndPathNamePipe;
        this.entityService = inject(EntityService);
    }
    getDataSource(entity, baseQuery, dataInterceptor, alwaysIncludeColList = [], treeFilterField) {
        let subscription;
        return {
            getRows: (params) => {
                params.api.hideOverlay();
                const availableFieldNames = entity.attributes.filter((field) => field.gridColumn).map((field) => field.name);
                const visibleColIds = params.api.getAllDisplayedColumns().map((col) => col.getColId());
                const visibleColumnFields = availableFieldNames.filter((fieldName) => visibleColIds.indexOf(fieldName) > -1 || alwaysIncludeColList.indexOf(fieldName) > -1);
                let defaultQuery = _.cloneDeep(baseQuery);
                if (treeFilterField) { // for treedata
                    if (!defaultQuery) {
                        defaultQuery = new Query();
                    }
                    let parentId = null;
                    if (params.request.groupKeys.length > 0) {
                        parentId = this.getEndPathNamePipe.transform(params.request.groupKeys.slice(-1)[0]);
                    }
                    defaultQuery?.filters.push(new Filter(treeFilterField, 'eq', parentId));
                }
                subscription = this.loadRows(entity, params.request, defaultQuery, visibleColumnFields).subscribe((response) => {
                    let modifiedData = dataInterceptor ? dataInterceptor(response['hydra:member']) : response['hydra:member'];
                    params.success({
                        rowData: this.loadLookupData(modifiedData, entity),
                        rowCount: response['hydra:totalItems']
                    });
                    if (params.api.paginationGetRowCount() === 0) {
                        params.api.showNoRowsOverlay();
                    }
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    loadLookupData(rows, entity) {
        let lookupFields = entity.attributes.filter((field) => field.formType === 'lookup');
        rows.forEach((row) => {
            lookupFields.forEach((field) => {
                // convert lookups to observables
                const uuid = row[field.name];
                if (!uuid) {
                    row[field.name] = of(field.blankDisplay || '');
                    return;
                }
                row[field.name] = this.entityService.build(field.dataSource.model).pipe(concatMap((subEntity) => this.entityService.getRecordDisplay(subEntity, uuid)));
            });
        });
        // Free up memory
        this.entityService.clearDisplayCache(true);
        return rows;
    }
    getDatasourceByParams(entity, queryParams) {
        let subscription;
        return {
            getRows: (params) => {
                params.api.hideOverlay();
                subscription = this.apiService.getList(entity, undefined, queryParams)
                    .subscribe((rowData) => {
                    params.success({
                        rowData: rowData,
                        // rowCount: 50
                    });
                    if (rowData.length === 0) {
                        params.api.showNoRowsOverlay();
                    }
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    getDatasourceByCustomRequest(request = { url: '', type: 'GET', body: {} }, queryParams, entity, dataInterceptor) {
        let subscription;
        return {
            getRows: (params) => {
                let request$ = request.type === 'GET' ? this.http.get(request.url, { params: queryParams }) : this.http.post(request.url, request.body, { params: queryParams });
                subscription = request$.subscribe((response) => {
                    let modifiedData = dataInterceptor ? dataInterceptor(response['hydra:member']) : response['hydra:member'];
                    params.success({
                        rowData: modifiedData,
                        rowCount: response['hydra:totalItems']
                    });
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    loadRows(entity, request, baseQuery, availableFieldNames = []) {
        let queryParams = {};
        queryParams.offset = request.startRow || 0;
        queryParams.limit = (request.endRow || DEFAULT_GRID_ROWS) - queryParams.offset;
        const query = this.convertAgGridQueryToQueryApi(request.filterModel, request.sortModel, request.groupKeys);
        let fullQuery;
        if (baseQuery) {
            fullQuery = {
                filters: [],
                logic: 'AND'
            };
            if (query.filters.length > 0) {
                fullQuery.filters.push(query.getQueryObject());
            }
            if (baseQuery.filters.length > 0) {
                fullQuery.filters.push(baseQuery.getQueryObject());
            }
            if (query.sort.length) {
                fullQuery.sort = query.sort;
            }
        }
        else {
            fullQuery = query.getQueryObject();
        }
        if (availableFieldNames.length > 0) {
            // UUID is required
            availableFieldNames.push('uuid');
            availableFieldNames = _.uniq(availableFieldNames);
            fullQuery.fields = availableFieldNames;
        }
        if (!fullQuery.sort?.length && entity.defaultSort?.length) {
            fullQuery.sort = entity.defaultSort;
        }
        if (!fullQuery.sort?.length) {
            console.warn(`${entity.title} does not have a default sort specified - this could cause infinite loops.`);
        }
        return this.apiService.query(entity, fullQuery, queryParams).pipe(map((response) => {
            return response;
        }));
    }
    convertAgGridQueryToQueryApi(filterModel, sortModel, groupKeys) {
        let query = new Query();
        keys(filterModel).forEach((fieldName) => {
            let fieldValue = filterModel[fieldName];
            if (fieldName === 'ag-Grid-AutoColumn') {
                if (groupKeys.length > 0) {
                    return; // remove tree column filter while expanding any node.
                }
                fieldName = 'name'; //TODO: this was for the Tree node column filter. mostly it will be name. Handle this gracefully.
            }
            let filter;
            if (fieldValue.type === 'blank') {
                filter = new Filter(fieldName, 'isnull', true);
            }
            else if (fieldValue.type === 'notBlank') {
                filter = new Filter(fieldName, 'isnull', false);
            }
            else if (fieldValue.filterType === 'boolean') {
                // Special boolean case
                if (fieldValue.filter === 'any') {
                    return;
                }
                filter = new Filter(fieldName, 'eq', fieldValue.filter);
            }
            else if (fieldValue.filterType === 'date' && fieldValue.filter) {
                filter = this.dateService.buildDateFilterGroup(fieldValue.filter, fieldName);
            }
            else if (fieldValue.filterType === 'set') {
                if (isBoolean(fieldValue.values[0])) {
                    filter = new Filter(fieldName, 'eq', fieldValue.values[0]);
                }
                else {
                    filter = new Filter(fieldName, 'in', fieldValue.values);
                }
            }
            else if (fieldValue.type === 'inRange') {
                filter = new FilterGroup([
                    new Filter(fieldName, 'lte', fieldValue.filterTo),
                    new Filter(fieldName, 'gte', fieldValue.filter)
                ], 'AND');
            }
            else if (fieldValue.filter) {
                let operator = OPERATOR_MAPPINGS[fieldValue.type] || 'eq';
                filter = new Filter(fieldName, operator, fieldValue.filter);
            }
            else {
                return;
            }
            query.filters.push(filter);
        });
        sortModel.forEach((sortModelItem) => {
            let fieldName = sortModelItem.colId;
            if (fieldName === 'ag-Grid-AutoColumn') {
                if (groupKeys.length > 0) {
                    return; // remove tree column filter while expanding any node.
                }
                fieldName = 'name'; //TODO: this was for the Tree node column filter. mostly it will be name. Handle this gracefully.
            }
            query.sort.push({
                field: fieldName,
                direction: sortModelItem.sort === 'asc' ? 'ASC' : 'DESC'
            });
        });
        return query;
    }
    static { this.ɵfac = function GridDataSourceService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || GridDataSourceService)(i0.ɵɵinject(i1.HttpClient), i0.ɵɵinject(i2.ApiService), i0.ɵɵinject(i2.DateService), i0.ɵɵinject(i2.GetEndPathNamePipe)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: GridDataSourceService, factory: GridDataSourceService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(GridDataSourceService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.HttpClient }, { type: i2.ApiService }, { type: i2.DateService }, { type: i2.GetEndPathNamePipe }], null); })();
//# sourceMappingURL=data:application/json;base64,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