import { Component, EventEmitter, Input, Output, forwardRef, inject } from '@angular/core';
import { NG_VALUE_ACCESSOR } from '@angular/forms';
import { autocompletion, completionKeymap } from '@codemirror/autocomplete';
import { HighlightStyle, StreamLanguage, syntaxHighlighting } from "@codemirror/language";
import { jinja2 } from "@codemirror/legacy-modes/mode/jinja2";
import { EditorState, Prec } from '@codemirror/state';
import { keymap, placeholder } from '@codemirror/view';
import { EditorView, basicSetup } from 'codemirror';
import { ThemeService } from '../../core/services/theme.service';
import { CommonModule } from '@angular/common';
import { linter } from '@codemirror/lint';
import { isBoolean } from 'lodash-es';
import { tags } from '@lezer/highlight';
import * as i0 from "@angular/core";
export class CodemirrorComponent {
    constructor(elementRef) {
        this.elementRef = elementRef;
        this.editorView = null;
        this.onChange = null;
        this.onTouched = null;
        this.code = '';
        this.themeService = inject(ThemeService);
        this.currentTheme = this.themeService.getTheme();
        this.focus = new EventEmitter();
        this.blur = new EventEmitter();
        this.extraKeyBindings = [];
        this.placeholder = '';
    }
    ngAfterViewInit() {
        const editorElement = this.elementRef.nativeElement.querySelector('div');
        let jinjaLanguage = StreamLanguage.define(jinja2);
        let customHighlights = HighlightStyle.define(customHighlightSpecs);
        const asyncCompletionSource = (context) => {
            if (!this.autocompleteGetter) {
                return null;
            }
            return this.autocompleteGetter(context);
        };
        const autocompleteExtension = autocompletion({
            override: [asyncCompletionSource], // Use async completion source
            activateOnTyping: true, // Enable suggestions on typing
        });
        let extensions = [
            basicSetup,
            jinjaLanguage,
            autocompleteExtension,
            keymap.of(completionKeymap),
            Prec.highest(keymap.of(this.extraKeyBindings)),
            EditorView.lineWrapping,
            this.handleUpdates(),
            this.handleFocusChanges(),
            placeholder(this.placeholder),
            syntaxHighlighting(customHighlights)
        ];
        if (this.linter) {
            let customLinter = this.linter;
            extensions.push(linter(view => customLinter(view)));
        }
        if (isBoolean(this.code)) {
            this.code = this.code ? '{{ True }}' : '{{ False }}';
        }
        const state = EditorState.create({
            doc: this.code ? this.code.toString() : '',
            extensions: extensions,
        });
        this.editorView = new EditorView({
            state,
            parent: editorElement
        });
        if (this.currentTheme === 'dark') {
            editorElement.classList.add('cm-dark-theme');
        }
    }
    handleUpdates() {
        return EditorView.updateListener.of((viewUpdate) => {
            if (viewUpdate.docChanged && this.onChange) {
                this.onChange(viewUpdate.state.doc.toString());
            }
        });
    }
    handleFocusChanges() {
        return EditorView.focusChangeEffect.of((state, focusing) => {
            if (focusing) {
                this.focus.emit();
            }
            else {
                this.blur.emit();
            }
            return null;
        });
    }
    replaceSelection(text, skipBrackets = false) {
        if (!this.editorView) {
            return;
        }
        const cursorIndex = this.editorView.state.selection.main.head;
        if (!skipBrackets && !this.isInCurlyBraces(this.editorView.state.doc.toString(), cursorIndex)) {
            text = `{{ ${text} }}`;
        }
        const tr = this.editorView.state.replaceSelection(text);
        this.editorView.dispatch(tr);
        setTimeout(() => {
            this.setFocus();
        });
    }
    isInCurlyBraces(s, index) {
        if (index < 0 || index >= s.length) {
            return false;
        }
        let i = 0;
        while (i < s.length) {
            if (s.substring(i, i + 2) === '{{') {
                const endIndex = s.indexOf('}}', i + 2);
                if (endIndex === -1) {
                    break;
                }
                if (i + 2 < index && index <= endIndex + 1) {
                    return true;
                }
                i = endIndex + 2;
            }
            else {
                i++;
            }
        }
        return false;
    }
    ngOnDestroy() {
        if (this.editorView) {
            this.editorView.destroy();
        }
    }
    // ControlValueAccessor methods
    writeValue(value) {
        this.code = value;
        if (this.editorView) {
            this.editorView.dispatch({
                changes: {
                    from: 0,
                    to: this.editorView.state.doc.length,
                    insert: value,
                },
            });
        }
    }
    registerOnChange(fn) {
        this.onChange = fn;
    }
    registerOnTouched(fn) {
        this.onTouched = fn;
    }
    setFocus() {
        this.editorView?.focus();
        const location = this.code.length;
        this.editorView?.dispatch({ selection: { anchor: location, head: location } });
    }
    static { this.ɵfac = function CodemirrorComponent_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || CodemirrorComponent)(i0.ɵɵdirectiveInject(i0.ElementRef)); }; }
    static { this.ɵcmp = /*@__PURE__*/ i0.ɵɵdefineComponent({ type: CodemirrorComponent, selectors: [["xf-code-mirror"]], inputs: { autocompleteGetter: "autocompleteGetter", linter: "linter", extraKeyBindings: "extraKeyBindings", placeholder: "placeholder" }, outputs: { focus: "focus", blur: "blur" }, standalone: true, features: [i0.ɵɵProvidersFeature([
                {
                    provide: NG_VALUE_ACCESSOR,
                    useExisting: forwardRef(() => CodemirrorComponent),
                    multi: true,
                },
            ]), i0.ɵɵStandaloneFeature], decls: 1, vars: 0, template: function CodemirrorComponent_Template(rf, ctx) { if (rf & 1) {
            i0.ɵɵelement(0, "div");
        } }, dependencies: [CommonModule], encapsulation: 2 }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(CodemirrorComponent, [{
        type: Component,
        args: [{
                standalone: true,
                imports: [CommonModule],
                selector: 'xf-code-mirror',
                template: '<div></div>',
                providers: [
                    {
                        provide: NG_VALUE_ACCESSOR,
                        useExisting: forwardRef(() => CodemirrorComponent),
                        multi: true,
                    },
                ]
            }]
    }], () => [{ type: i0.ElementRef }], { focus: [{
            type: Output
        }], blur: [{
            type: Output
        }], autocompleteGetter: [{
            type: Input
        }], linter: [{
            type: Input
        }], extraKeyBindings: [{
            type: Input
        }], placeholder: [{
            type: Input
        }] }); })();
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassDebugInfo(CodemirrorComponent, { className: "CodemirrorComponent", filePath: "lib/components/codemirror/codemirror.component.ts", lineNumber: 40 }); })();
const customHighlightSpecs = [
    {
        tag: [tags.atom, tags.bool, tags.url, tags.contentSeparator, tags.labelName],
        class: 'cm-atom', style: ''
    },
    {
        tag: tags.keyword,
        class: 'cm-keyword',
        color: ''
    },
    {
        tag: tags.meta,
        color: "#404740"
    },
    {
        tag: tags.link,
        textDecoration: "underline"
    },
    {
        tag: tags.heading,
        textDecoration: "underline",
        fontWeight: "bold"
    },
    {
        tag: tags.emphasis,
        fontStyle: "italic"
    },
    {
        tag: tags.strong,
        fontWeight: "bold"
    },
    {
        tag: tags.strikethrough,
        textDecoration: "line-through"
    },
    {
        tag: [tags.literal, tags.inserted],
        color: "#164"
    },
    {
        tag: [tags.string, tags.deleted],
        color: "#a11"
    },
    {
        tag: [tags.regexp, tags.escape, tags.special(tags.string)],
        color: "#e40"
    },
    {
        tag: tags.definition(tags.variableName),
        color: "#00f"
    },
    {
        tag: tags.local(tags.variableName),
        color: "#30a"
    },
    {
        tag: [tags.typeName, tags.namespace],
        color: "#085"
    },
    {
        tag: tags.className,
        color: "#167"
    },
    {
        tag: [tags.special(tags.variableName), tags.macroName],
        color: "#256"
    },
    {
        tag: tags.definition(tags.propertyName),
        color: "#00c"
    },
    {
        tag: tags.comment,
        color: "#940"
    },
    {
        tag: tags.invalid,
        color: "#f00"
    }
];
//# sourceMappingURL=data:application/json;base64,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