import { HttpContext } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { plainToInstance } from 'class-transformer';
import * as _ from 'lodash-es';
import { catchError, map, of, tap } from 'rxjs';
import { ALL_RECORDS_SIZE, API, FIXED_MODULE, IGNORE_ERRORS, UUID_REGEX } from '../constants';
import { Agent } from '../models/agent';
import { Connector } from '../models/connector';
import { ConnectorConfiguration } from '../models/connector-configuration';
import * as i0 from "@angular/core";
import * as i1 from "@angular/common/http";
import * as i2 from "./utility.service";
export class ConnectorService {
    constructor(http, utilityService) {
        this.http = http;
        this.utilityService = utilityService;
        this.cache = {};
    }
    // Connector Operations
    getConnector(name, version, agent, ignoreErrors = true) {
        let params = {};
        if (agent) {
            params.agent = agent;
        }
        const api = `${API.INTEGRATIONS}connectors/${name}/${version}/?format=json`;
        return this.http.post(api, {}, { params: params, context: new HttpContext().set(IGNORE_ERRORS, ignoreErrors) }).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    updateConnector(id, data = {}) {
        const api = `${API.INTEGRATIONS}connectors/${id}/?format=json`;
        return this.http.put(api, data).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    deleteConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/delete/files/?format=json`;
        return this.http.delete(api);
    }
    // Connector Config Operatons
    createConnectorConfig(data) {
        const api = `${API.INTEGRATIONS}configuration/?format=json`;
        return this.http.post(api, data).pipe(map((response) => {
            return plainToInstance(ConnectorConfiguration, response);
        }));
    }
    updateConnectorConfig(id, data) {
        const api = `${API.INTEGRATIONS}configuration/${id}/?format=json`;
        return this.http.put(api, data).pipe(map((response) => {
            return plainToInstance(ConnectorConfiguration, response);
        }));
    }
    deleteConnectorConfig(id) {
        const api = `${API.INTEGRATIONS}configuration/${id}/`;
        return this.http.delete(api);
    }
    // Development Connector
    getDevelopmentConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/?format=json`;
        return this.http.post(api, {}).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    createDevelopmentConnector(data = {}) {
        const api = `${API.INTEGRATIONS}connector/development/entity/?format=json`;
        return this.http.post(api, data).pipe(map((response) => {
            return plainToInstance(Connector, response);
        }));
    }
    // Connector File Oeprations
    getConnectorFile(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/files/?format=json`;
        return this.http.post(api, {});
    }
    getConnectorFileContent(id, filepath) {
        const api = API.INTEGRATIONS + 'connector/development/entity/' + id + '/files' + '/?format=json';
        return this.http.post(api, { 'xpath': filepath });
    }
    createConnectorFile(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/files/?format=json`;
        return this.http.post(api, { xpath: data });
    }
    updateConnectorFile(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/rename/files/?format=json`;
        return this.http.post(api, data);
    }
    deleteConnectorFile(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/delete/files/?format=json`;
        return this.http.post(api, data);
    }
    // Connector folder operations
    createConnectorFolder(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/folders/?format=json`;
        return this.http.put(api, data).pipe(map((response) => {
            return response.data;
        }));
    }
    deleteConnectorFolder(id, data) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/delete/folders/?format=json`;
        return this.http.post(api, data);
    }
    getConnectorHealth(connector, config_id, agent) {
        let params = {
            config_id: config_id,
            connector_name: connector.name,
            connector_version: connector.version,
            agent: agent
        };
        const api = `${API.INTEGRATIONS}healthcheck/`;
        return this.http.post(api, params);
    }
    getConnectorOutputSchema(connectorName, connectorVersion, operation, config, params, agent) {
        const key = `${connectorName}/${connectorVersion}/${operation}`;
        if (this.cache[key] && !this.cache[key].error) {
            // TODO: Ensure this doesn't increase browser memory too much
            return of(this.cache[key]);
        }
        let queryParams = {};
        if (agent) {
            queryParams = { agent: agent };
        }
        const api = `${API.INTEGRATIONS}connector_output_schema/${connectorName}/${connectorVersion}/`;
        const data = {
            'operation': operation,
            'config': config,
            'params': params
        };
        return this.http.post(api, data, { params: queryParams, context: new HttpContext().set(IGNORE_ERRORS, true) }).pipe(catchError((error) => {
            console.warn(error);
            return of({
                data: {
                    output_schema: {}
                },
                error: true
            });
        }), tap((response) => {
            this.cache[key] = response;
        }));
        // Pending error handling
    }
    // Maybe common for dev and installed? or does not matter?
    executeConnectorAction(connector, operation, configuration, params, audit, auditInfo, agent) {
        const api = `${API.INTEGRATIONS}execute/?format=json`;
        let requestParams = {
            connector: connector.name,
            version: connector.version,
            config: configuration.config,
            operation: operation.operation,
            params: params,
            audit: audit,
            audit_info: auditInfo
        };
        if (agent) {
            requestParams.agent = agent;
        }
        return this.http.post(api, requestParams, {
            context: new HttpContext().set(IGNORE_ERRORS, true)
        });
    }
    // Maybe common for dev and installed? or does not matter?
    exportDevelopedConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/export/?format=json`;
        return this.http.get(api, { responseType: 'arraybuffer' });
        // Pending error handling
    }
    // Below is a list of pending functions - need to figure out if some can be consolidated together
    // publishConnector
    // installConnector
    // getAgents
    // getIngestionPlaybookCollectionUUID
    // getSamplePlaybooks
    // getDataSources
    // loadConnectors - this is dev or installed ?
    // loadByTags
    // getConfiguredConnectorList
    // getConnectorActionDetails
    // getConfiguredConnectors
    // getConnectorOutputSchema
    // getCategories
    // getAnnotations
    // getAnnotationConnectors
    // applyRolesToActions
    // downloadHelpPDF
    getDependencyStatus(connector) {
        const api = `${API.INTEGRATIONS}connectors/dependencies_check/${connector.name}/${connector.version}/?format=json`;
        return this.http.get(api).pipe(map((response) => {
            return response;
        }));
    }
    getFieldData(data) {
        const api = `${API.INTEGRATIONS}execute/?format=json`;
        return this.http.post(api, data, {
            context: new HttpContext().set(IGNORE_ERRORS, true)
        });
    }
    // installDependencies
    filterConnectorOperations(operation, installed) {
        installed = (!this.utilityService.isUndefined(installed) && _.isBoolean(installed)) ? installed : true;
        if (installed) {
            return (operation.enabled && (this.utilityService.isUndefined(operation.visible) || operation.visible === true));
        }
        return this.utilityService.isUndefined(operation.visible) || operation.visible === true;
    }
    getDevelopedConnector(id) {
        const api = `${API.INTEGRATIONS}connector/development/entity/${id}/?format=json`;
        return this.http.post(api, {});
    }
    //Get Connector Template
    getConnectorTemplates(data) {
        const api = `${API.INTEGRATIONS}connector/development/templates/?format=json`;
        return this.http.post(api, data);
    }
    //Create New connector
    createConnector(connectorConfig) {
        const api = `${API.INTEGRATIONS}connector/development/entity/?format=json`;
        return this.http.post(api, connectorConfig);
    }
    loadConnectors(params = {}) {
        params.page_size = ALL_RECORDS_SIZE;
        return this.http.get(`${API.INTEGRATIONS}connectors/`, { params: params }).pipe(map((response) => {
            return plainToInstance(Connector, response['hydra:member']);
        }));
    }
    //get agents in connector configs
    getAgents(connector, installed) {
        let config = {
            'active': true
        };
        if (installed) {
            config.installed = true;
        }
        // versionService.getIntergrationVersionDetails().then(function (integrationVersionDetails) {
        return this.http.get(`${API.INTEGRATIONS}${FIXED_MODULE.CONNECTORS}/${FIXED_MODULE.AGENTS}/${connector.name}/`, { params: config }).pipe(map((installedAgents) => {
            if (installedAgents['hydra:member'].length > 0) {
                //TO DO: Agent compatibilty
                //installedAgents['hydra:member'].forEach((agent: any) => {
                //var minimumCompatibilityVersion = integrationVersionDetails.minimum_compatibility_version.split('-')[0];
                //agent.isIncompatible = parseFloat(agent.agent_version.replace(/\./g, '')) >= parseFloat(minimumCompatibilityVersion.replace(/\./g, '')) ? false : true;
                //});
            }
            return plainToInstance(Agent, installedAgents['hydra:member']);
        }));
    }
    getConfiguredAgents() {
        return this.http.get(`${API.INTEGRATIONS}${FIXED_MODULE.AGENTS}/`).pipe(map((response) => {
            return response['hydra:member'];
        }));
    }
    installConnectorOnAgents(name, version, queryParams = {}, payload) {
        queryParams.replace = true;
        queryParams.trust = true;
        return this.http.post(`${API.INTEGRATIONS}import-connector/${name}/${version}/`, payload, { params: queryParams });
    }
    getConnectorConfigurations(name, version, queryparams = {}) {
        return this.http.get(`${API.INTEGRATIONS}configuration/${name}/${version}/`, { params: queryparams });
    }
    getConfigurationDetail(uuid) {
        return this.http.get(`${API.INTEGRATIONS}configuration/${uuid}/`);
    }
    getConfigId(configurations, configId) {
        const defaultConfig = _.find(configurations, (config) => {
            return config.default;
        });
        let config_id = '';
        let isJinja = false;
        if (configId && configId.replace(/'/g, '').length > 0) {
            if (UUID_REGEX.test(configId) || /^[0-9a-f]{32}$/.test(configId)) {
                let foundConfig = _.find(configurations, (config) => {
                    return config.config_id === configId;
                });
                if (foundConfig) {
                    config_id = configId;
                }
                else if (defaultConfig) {
                    config_id = defaultConfig.config_id;
                }
            }
            else {
                isJinja = true;
                config_id = configId;
            }
        }
        else if (configurations.length > 0) {
            if (defaultConfig) {
                config_id = defaultConfig.config_id;
            }
            else if (configurations.length === 1) {
                configId = configurations[0].config_id;
            }
        }
        return { config_id, isJinja };
    }
    static { this.ɵfac = function ConnectorService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || ConnectorService)(i0.ɵɵinject(i1.HttpClient), i0.ɵɵinject(i2.UtilityService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: ConnectorService, factory: ConnectorService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(ConnectorService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.HttpClient }, { type: i2.UtilityService }], null); })();
//# sourceMappingURL=data:application/json;base64,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