import { Injectable } from '@angular/core';
import _ from 'lodash-es';
import { forEach, isUndefined } from 'lodash-es';
import { ReplaySubject, Subject, map, of } from 'rxjs';
import { v4 as uuid } from 'uuid';
import { API } from '../constants';
import { Field } from '../models/field';
import * as i0 from "@angular/core";
import * as i1 from "@angular/router";
import * as i2 from "xng-breadcrumb";
export class UtilityService {
    constructor(router, route, breadcrumbService) {
        this.router = router;
        this.route = route;
        this.breadcrumbService = breadcrumbService;
        // Individual components can subscribe to this to receive document click events
        this.documentClickedTarget = new Subject();
        this.navigationChanged = new ReplaySubject(1);
        this.permissionsChanged = new Subject();
    }
    generateUUID() {
        return uuid();
    }
    setTitleAlias(alias, value, newPath) {
        let breadcrumb = {
            label: value
        };
        if (newPath) {
            breadcrumb.routeInterceptor = () => newPath;
        }
        return this.breadcrumbService.set('@' + alias, breadcrumb);
    }
    isUndefined(value) {
        return value === undefined || value === null;
    }
    convertHexToRgbA(value) {
        value = (value === undefined || value === null) ? '#000000' : value;
        let hexcode;
        if (/^#([A-Fa-f0-9]{3}){1,2}$/.test(value)) {
            hexcode = value.substring(1).split('');
            if (hexcode.length === 3) {
                hexcode = [hexcode[0], hexcode[0], hexcode[1], hexcode[1], hexcode[2], hexcode[2]];
            }
            hexcode = '0x' + hexcode.join('');
            return 'rgba(' + [(hexcode >> 16) & 255, (hexcode >> 8) & 255, hexcode & 255].join(',') + ',0.8)';
        }
        else {
            return 'rgba(0, 0, 0, 0.1)';
        }
    }
    //set parameters in URL
    setParametersInUrl(queryParams) {
        forEach(queryParams, (paramValue, paramKey) => {
            if (paramValue === '' || paramValue === null) {
                delete queryParams[paramKey];
            }
        });
        this.router.navigate([], {
            relativeTo: this.route,
            queryParams: queryParams
        });
    }
    //append new parameter in URL
    appendNewParameterInUrl(newQueryParams) {
        this.router.navigate([], {
            relativeTo: this.route,
            queryParams: newQueryParams,
            queryParamsHandling: 'merge'
        });
    }
    filterFields(fields, search, fieldsAlreadyAdded) {
        return of(fields).pipe(map((fields) => {
            return fields.map((field) => field.asSearchableItem());
        }), map((items) => {
            return items.filter((item) => {
                return fieldsAlreadyAdded.indexOf(item.uuid) === -1;
            });
        }), map((items) => {
            return items.filter((item) => {
                return _.isString(item.display) && item.display.toLowerCase().indexOf(search.toLowerCase()) > -1;
            });
        }));
    }
    downloadFile(fileData, fileName, fileType) {
        let file;
        let downloadLink;
        // file
        file = new Blob([fileData], { type: fileType });
        const navigator = window.navigator;
        if (navigator && navigator.msSaveOrOpenBlob) {
            navigator.msSaveOrOpenBlob(file, fileName);
        }
        else {
            // Download link
            downloadLink = document.createElement('a');
            // File name
            downloadLink.download = fileName;
            // Create a link to the file
            downloadLink.href = window.URL.createObjectURL(file);
            // Hide download link
            downloadLink.style.display = 'none';
            // Add the link to DOM
            document.body.appendChild(downloadLink);
            // Click download link
            downloadLink.click();
        }
    }
    csvToJSON(CSVcontents) {
        let contents = CSVcontents.split('\r\n');
        let columnNames = ((contents[0]).replace(/\n|\r/g, '')).split(',');
        let data = [];
        for (let i = 1; i < contents.length; i++) {
            if (contents[i] !== '' && contents[i] !== null && contents[i] !== undefined) {
                let rowData = (contents[i].replace(/\n|\r/g, '')).split(',');
                let dataObj = {};
                for (let j = 0; j < rowData.length; j++) {
                    dataObj[columnNames[j]] = rowData[j];
                }
                data.push(dataObj);
            }
        }
        return { columns: columnNames, data: data };
    }
    newAttribute() {
        return {
            // uuid: $window.UUID.generate(),
            isNew: true,
            ownsRelationship: false,
            visibility: true,
            writeable: true,
            editable: true,
            gridColumn: false,
            defaultValue: '',
            value: null,
            validation: {
                required: false,
                minlength: 0,
                // maxlength: MAX_TEXT_LENGTH
            },
            name: '(Untitled)',
            title: '(Untitled)',
            descriptions: {
                singular: '(Untitled)'
            },
            formType: 'text',
            type: 'string'
        };
    }
    ;
    convertToCamelCase(string) {
        if (!isUndefined(string)) {
            return string.replace(/(?:^\w|[A-Z]|\b\w)/g, function (letter, index) {
                return index === 0 ? letter.toLowerCase() : letter.toUpperCase();
            }).replace(/\s+/g, '');
        }
        return string;
    }
    getBaseContextPath() {
        var baseHrefElement = document.querySelector('base');
        if (baseHrefElement) {
            return baseHrefElement.getAttribute('href') || '';
        }
        return '';
    }
    getIriApiPath(IRI) {
        if (IRI.charAt(0) === '/') {
            return IRI.substring(1);
        }
        // else if (IRI.indexOf('currentUserIri') > -1){
        //   return $interpolate(IRI)({currentUserIri: currentUserIriService});
        // }
        return IRI;
    }
    getOrigin() {
        return location.protocol + '://' + location.host + ':' + location.port + this.getBaseContextPath();
    }
    prependIri(value) {
        return `${API.API_3_BASE}${value}`;
    }
    formatBytes(a, b = 2) {
        if (!+a)
            return "0 Bytes";
        const c = 0 > b ? 0 : b;
        const d = Math.floor(Math.log(a) / Math.log(1024));
        return `${parseFloat((a / Math.pow(1024, d)).toFixed(c))} ${["Bytes", "KiB", "MiB", "GiB", "TiB", "PiB", "EiB", "ZiB", "YiB"][d]}`;
    }
    getTimeInterval(toDateStr, fromDateStr, ms_time) {
        let toDate, fromDate, ms;
        if (!ms_time) {
            toDate = new Date(toDateStr);
            fromDate = new Date(fromDateStr);
            ms = toDate - fromDate;
        }
        else {
            ms = ms_time;
        }
        let seconds = ms >= 1000 ? (ms / 1000) : parseInt((ms / 1000).toString(), 10);
        ms = ms % 1000;
        let minutes = parseInt((seconds / 60).toString(), 10);
        seconds = parseInt((seconds % 60).toString(), 10);
        let hours = parseInt((minutes / 60).toString(), 10);
        minutes = minutes % 60;
        let returnVal = '';
        if (hours > 0) {
            returnVal += Math.round(hours) + ' hours ';
        }
        if (minutes > 0) {
            returnVal += Math.round(minutes) + ' minutes ';
        }
        if (seconds > 0) {
            returnVal += Math.round(seconds) + ' seconds ';
        }
        if (ms > 0) {
            returnVal += Math.round(ms) + ' ms';
        }
        return returnVal;
    }
    addJinjaBraces(value) {
        return `{{${value}}}`;
    }
    removeJinjaBraces(value) {
        return ((value).replace('{{', '').replace('}}', '')).trim();
    }
    getTimeAgo(fromDateStr, toDateStr, ms_time) {
        let toDate, fromDate, ms;
        if (!ms_time) {
            toDate = toDateStr ? new Date(toDateStr) : new Date();
            fromDate = new Date(fromDateStr);
            ms = toDate - fromDate;
        }
        else {
            ms = ms_time;
        }
        let seconds = ms >= 1000 ? (ms / 1000) : parseInt((ms / 1000).toString(), 10);
        ms = ms % 1000;
        let minutes = parseInt((seconds / 60).toString(), 10);
        seconds = parseInt((seconds % 60).toString(), 10);
        let hours = parseInt((minutes / 60).toString(), 10);
        minutes = minutes % 60;
        let days = parseInt((hours / 24).toString(), 10);
        let months = parseInt((days / 30).toString(), 10);
        let years = parseInt((months / 12).toString(), 10);
        let returnVal = '';
        if (ms > 0) {
            returnVal = Math.round(ms) + ' ms';
        }
        if (seconds > 0) {
            returnVal = Math.round(seconds) + ' second(s) ';
        }
        if (minutes > 0) {
            returnVal = Math.round(minutes) + ' minute(s) ';
        }
        if (hours > 0) {
            returnVal = Math.round(hours) + ' hour(s) ';
        }
        if (days > 0) {
            returnVal = Math.round(days) + ' day(s) ';
        }
        if (months > 0) {
            returnVal = Math.round(months) + ' month(s) ';
        }
        if (years > 0) {
            returnVal = Math.round(years) + ' year(s) ';
        }
        return returnVal;
    }
    typeOf(value) {
        if (_.isArray(value)) {
            return 'list';
        }
        else if (_.isNumber(value)) {
            return 'number';
        }
        else if (_.isObject(value)) {
            return 'dict';
        }
        else if (_.isString(value)) {
            return 'string';
        }
        else if (_.isBoolean(value)) {
            return 'boolean';
        }
        else {
            return 'any';
        }
    }
    getFieldTypeByValue(value) {
        let dataType = Field.BuiltInType.TEXT;
        try {
            let val = JSON.parse(value);
            if (_.isNumber(val)) {
                dataType = Field.BuiltInType.DECIMAL;
            }
            else if (_.isBoolean(val)) {
                dataType = Field.BuiltInType.CHECKBOX;
            }
        }
        catch (error) {
            if (_.isArray(value)) {
                dataType = Field.BuiltInType.ARRAY;
            }
            else if (_.isObject(value)) {
                dataType = Field.BuiltInType.JSON;
            }
            else if ((new Date(value)).toString() !== 'Invalid Date') {
                dataType = Field.BuiltInType.DATETIME;
            }
        }
        return dataType;
    }
    getHigherVersionNumber(version1, version2) {
        const v1split = version1.split('.').map(Number);
        const v2split = version2.split('.').map(Number);
        let index = 0;
        while (index < v1split.length) {
            if (index === v2split.length) {
                return version1;
            }
            const v1Value = v1split[index];
            const v2Value = v2split[index];
            if (v1Value === v2Value) {
                index++;
                continue;
            }
            else if (v1Value > v2Value) {
                return version1;
            }
            else if (v2Value > v1Value) {
                return version2;
            }
        }
        return version2;
    }
    static { this.ɵfac = function UtilityService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || UtilityService)(i0.ɵɵinject(i1.Router), i0.ɵɵinject(i1.ActivatedRoute), i0.ɵɵinject(i2.BreadcrumbService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: UtilityService, factory: UtilityService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(UtilityService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.Router }, { type: i1.ActivatedRoute }, { type: i2.BreadcrumbService }], null); })();
//# sourceMappingURL=data:application/json;base64,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