import { Injectable } from '@angular/core';
import { BehaviorSubject, ReplaySubject, Subject, of, timer } from 'rxjs';
import { distinctUntilChanged, filter, finalize, map, retry, skip, switchAll, take, tap } from 'rxjs/operators';
import { webSocket } from 'rxjs/webSocket';
import { forEach } from 'lodash-es';
import { v4 as uuid } from 'uuid';
import { TOKEN } from '../constants';
import * as i0 from "@angular/core";
import * as i1 from "./storage.service";
export const WS_ENDPOINT = 'wss://' + location.host + '/api/ws/';
export const RECONNECT_INTERVAL = 5000;
export class WebsocketService {
    constructor(storageService) {
        this.storageService = storageService;
        this.websocketSessionID = null;
        this.observablesTopics = {};
        this.status$ = new BehaviorSubject(false);
        this.messages$ = new Subject();
    }
    connect() {
        this.create();
        this.connectionStatus$.pipe(skip(1), filter(status => !status), tap(() => this.create())).subscribe();
    }
    set(websocketSessionID) {
        this.websocketSessionID = websocketSessionID;
    }
    create() {
        this.close();
        this.websocketSessionID = this.websocketSessionID ?? uuid();
        let queryParams = `?websocketSessionID=${this.websocketSessionID}&token=${this.storageService.getItem(TOKEN.ACCESS_KEY)}`;
        const openObserver = new Subject();
        this.openObserverSubscription = openObserver.subscribe(() => {
            this.status$.next(true);
            forEach(this.observablesTopics, (value, key) => {
                this.websocketSubscribe(key, true);
            });
        });
        const closeObserver = new Subject();
        this.closeObserverSubscription = closeObserver.pipe(map((_) => false)).subscribe(this.status$);
        this.ws = webSocket({
            url: WS_ENDPOINT + queryParams,
            openObserver,
            closeObserver,
        });
        this.ws.pipe(retry({
            delay: (errs) => {
                this.status$.next(false);
                console.log(`Websocket connection down, will attempt reconnection in ${RECONNECT_INTERVAL}ms`);
                return timer(RECONNECT_INTERVAL);
            }
        })).subscribe(this.messages$);
    }
    get connectionStatus$() {
        return this.status$.pipe(distinctUntilChanged());
    }
    close() {
        if (this.ws) {
            this.openObserverSubscription.unsubscribe();
            this.closeObserverSubscription.unsubscribe();
            this.ws.unsubscribe();
        }
    }
    message(message) {
        this.connectionStatus$.pipe(filter(status => status), tap(() => this.ws.next(message)), take(1)).subscribe();
    }
    websocketSubscribe(topic, reconnect = false) {
        if (this.observablesTopics[topic] && !reconnect) {
            return this.observablesTopics[topic];
        }
        else {
            let messagesSubject$ = this.observablesTopics[topic] || new ReplaySubject(1);
            if (this.status$.value) {
                messagesSubject$.next(this.ws.multiplex(() => ({ subscribe: [topic] }), () => ({ unsubscribe: [topic] }), (message) => {
                    return message.topics.includes(topic);
                }).pipe(map((message) => message.message), finalize(() => {
                    if (!this.observablesTopics[topic].observed) {
                        this.observablesTopics[topic].complete();
                        delete this.observablesTopics[topic];
                    }
                })));
            }
            else {
                messagesSubject$.next(of());
            }
            this.observablesTopics[topic] = messagesSubject$;
            return this.observablesTopics[topic].pipe(switchAll());
        }
    }
    static { this.ɵfac = function WebsocketService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || WebsocketService)(i0.ɵɵinject(i1.StorageService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: WebsocketService, factory: WebsocketService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(WebsocketService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.StorageService }], null); })();
//# sourceMappingURL=data:application/json;base64,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