/*
MIT License

Copyright (c) 2022 Ephox Corporation DBA Tiny Technologies, Inc.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/
import { isPlatformBrowser, CommonModule } from '@angular/common';
import { Component, forwardRef, Inject, Input, PLATFORM_ID, InjectionToken, Optional, ChangeDetectionStrategy } from '@angular/core';
import { FormsModule, NG_VALUE_ACCESSOR } from '@angular/forms';
import { Subject } from 'rxjs';
import { getTinymce } from '../utils/TinyMCE';
import { listenTinyMCEEvent, bindHandlers, isTextarea, mergePlugins, uuid, noop, isNullOrUndefined } from '../utils/Utils';
import { Events } from './Events';
import * as i0 from "@angular/core";
import * as i1 from "@angular/common";
export const TINYMCE_SCRIPT_SRC = new InjectionToken('TINYMCE_SCRIPT_SRC');
const EDITOR_COMPONENT_VALUE_ACCESSOR = {
    provide: NG_VALUE_ACCESSOR,
    useExisting: forwardRef(() => HtmlEditorComponent),
    multi: true
};
export class HtmlEditorComponent extends Events {
    set disabled(val) {
        this._disabled = val;
        if (this._editor && this._editor.initialized) {
            if (typeof this._editor.mode?.set === 'function') {
                this._editor.mode.set(val ? 'readonly' : 'design');
            }
            else {
                this._editor.mode.set = () => val ? 'readonly' : 'design';
            }
        }
    }
    get disabled() {
        return this._disabled;
    }
    get editor() {
        return this._editor;
    }
    constructor(elementRef, ngZone, cdRef, platformId, platformLocation, tinymceScriptSrc) {
        super();
        this.cdRef = cdRef;
        this.platformId = platformId;
        this.platformLocation = platformLocation;
        this.tinymceScriptSrc = tinymceScriptSrc;
        this.cloudChannel = '7';
        this.apiKey = 'no-api-key';
        this.id = '';
        this.modelEvents = 'change input undo redo';
        this.onTouchedCallback = noop;
        this.destroy$ = new Subject();
        this.initialise = () => {
            const finalInit = {
                selector: undefined,
                target: this._element,
                inline: this.inline,
                readonly: this.disabled,
                debounce: true,
                toolbar_mode: 'wrap',
                plugins: mergePlugins((this.init && this.init.plugins), this.plugins),
                toolbar: this.toolbar || (this.init && this.init.toolbar),
                height: 350,
                max_height: 600,
                menubar: false,
                paste_data_images: true,
                extended_valid_elements: 'svg[*],defs[*],pattern[*],desc[*],metadata[*],g[*],mask[*],path[*],line[*],marker[*],rect[*],circle[*],ellipse[*],polygon[*],polyline[*],linearGradient[*],radialGradient[*],stop[*],image[*],view[*],text[*],textPath[*],title[*],tspan[*],glyph[*],symbol[*],switch[*],use[*]',
                autocompleteEnterKeyEvent: false,
                skin: "oxide-dark",
                color_default_foreground: 'white',
                content_style: 'body { color:#FFFFFF; }',
                base_url: `${this.platformLocation.getBaseHrefFromDOM()}tinymce/`,
                ...this.init,
                setup: (editor) => {
                    this._editor = editor;
                    listenTinyMCEEvent(editor, 'init', this.destroy$).subscribe(() => {
                        this.initEditor(editor);
                    });
                    bindHandlers(this, editor, this.destroy$);
                    if (this.init && typeof this.init.setup === 'function') {
                        this.init.setup(editor);
                    }
                }
            };
            if (isTextarea(this._element)) {
                this._element.style.visibility = '';
            }
            this.ngZone.runOutsideAngular(() => {
                getTinymce().init(finalInit);
            });
        };
        this._elementRef = elementRef;
        this.ngZone = ngZone;
    }
    writeValue(value) {
        if (this._editor && this._editor.initialized) {
            this._editor.setContent(isNullOrUndefined(value) ? '' : value);
        }
        else {
            this.initialValue = value === null ? undefined : value;
        }
    }
    registerOnChange(fn) {
        this.onChangeCallback = fn;
    }
    registerOnTouched(fn) {
        this.onTouchedCallback = fn;
    }
    setDisabledState(isDisabled) {
        this.disabled = isDisabled;
    }
    ngAfterViewInit() {
        if (isPlatformBrowser(this.platformId)) {
            this.id = this.id || uuid('tiny-angular');
            this.inline = this.inline !== undefined ? this.inline !== false : !!(this.init?.inline);
            this.createElement();
            if (getTinymce() !== null) {
                this.initialise();
            }
        }
    }
    ngOnDestroy() {
        this.destroy$.next();
        if (getTinymce() !== null) {
            getTinymce().remove(this._editor);
        }
    }
    createElement() {
        const tagName = typeof this.tagName === 'string' ? this.tagName : 'div';
        this._element = document.createElement(this.inline ? tagName : 'textarea');
        if (this._element) {
            if (document.getElementById(this.id)) {
                /* eslint no-console: ["error", { allow: ["warn"] }] */
                console.warn(`TinyMCE-Angular: an element with id [${this.id}] already exists. Editors with duplicate Id will not be able to mount`);
            }
            this._element.id = this.id;
            if (isTextarea(this._element)) {
                this._element.style.visibility = 'hidden';
            }
            this._elementRef.nativeElement.appendChild(this._element);
        }
    }
    initEditor(editor) {
        listenTinyMCEEvent(editor, 'blur', this.destroy$).subscribe(() => {
            this.cdRef.markForCheck();
            this.ngZone.run(() => this.onTouchedCallback());
        });
        listenTinyMCEEvent(editor, this.modelEvents, this.destroy$).subscribe(() => {
            this.cdRef.markForCheck();
            this.ngZone.run(() => this.emitOnChange(editor));
        });
        if (typeof this.initialValue === 'string') {
            this.ngZone.run(() => {
                editor.setContent(this.initialValue);
                if (editor.getContent() !== this.initialValue) {
                    this.emitOnChange(editor);
                }
                if (this.onInitNgModel !== undefined) {
                    this.onInitNgModel.emit(editor);
                }
            });
        }
    }
    emitOnChange(editor) {
        if (this.onChangeCallback) {
            this.onChangeCallback(editor.getContent({ format: this.outputFormat }));
        }
    }
    static { this.ɵfac = function HtmlEditorComponent_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || HtmlEditorComponent)(i0.ɵɵdirectiveInject(i0.ElementRef), i0.ɵɵdirectiveInject(i0.NgZone), i0.ɵɵdirectiveInject(i0.ChangeDetectorRef), i0.ɵɵdirectiveInject(PLATFORM_ID), i0.ɵɵdirectiveInject(i1.PlatformLocation), i0.ɵɵdirectiveInject(TINYMCE_SCRIPT_SRC, 8)); }; }
    static { this.ɵcmp = /*@__PURE__*/ i0.ɵɵdefineComponent({ type: HtmlEditorComponent, selectors: [["editor"]], inputs: { cloudChannel: "cloudChannel", apiKey: "apiKey", licenseKey: "licenseKey", init: "init", id: "id", initialValue: "initialValue", outputFormat: "outputFormat", inline: "inline", tagName: "tagName", plugins: "plugins", toolbar: "toolbar", modelEvents: "modelEvents", allowedEvents: "allowedEvents", ignoreEvents: "ignoreEvents", disabled: "disabled" }, standalone: true, features: [i0.ɵɵProvidersFeature([EDITOR_COMPONENT_VALUE_ACCESSOR]), i0.ɵɵInheritDefinitionFeature, i0.ɵɵStandaloneFeature], decls: 0, vars: 0, template: function HtmlEditorComponent_Template(rf, ctx) { }, dependencies: [CommonModule, FormsModule], styles: ["[_nghost-%COMP%]{display:block}"], changeDetection: 0 }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(HtmlEditorComponent, [{
        type: Component,
        args: [{ selector: 'editor', template: '', providers: [EDITOR_COMPONENT_VALUE_ACCESSOR], standalone: true, imports: [CommonModule, FormsModule], changeDetection: ChangeDetectionStrategy.OnPush, styles: [":host{display:block}\n"] }]
    }], () => [{ type: i0.ElementRef }, { type: i0.NgZone }, { type: i0.ChangeDetectorRef }, { type: Object, decorators: [{
                type: Inject,
                args: [PLATFORM_ID]
            }] }, { type: i1.PlatformLocation }, { type: undefined, decorators: [{
                type: Optional
            }, {
                type: Inject,
                args: [TINYMCE_SCRIPT_SRC]
            }] }], { cloudChannel: [{
            type: Input
        }], apiKey: [{
            type: Input
        }], licenseKey: [{
            type: Input
        }], init: [{
            type: Input
        }], id: [{
            type: Input
        }], initialValue: [{
            type: Input
        }], outputFormat: [{
            type: Input
        }], inline: [{
            type: Input
        }], tagName: [{
            type: Input
        }], plugins: [{
            type: Input
        }], toolbar: [{
            type: Input
        }], modelEvents: [{
            type: Input
        }], allowedEvents: [{
            type: Input
        }], ignoreEvents: [{
            type: Input
        }], disabled: [{
            type: Input
        }] }); })();
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassDebugInfo(HtmlEditorComponent, { className: "HtmlEditorComponent", filePath: "lib/form-fields/field-renderers/built-ins/richtext-html/html-editor-module/html-editor.component.ts", lineNumber: 56 }); })();
//# sourceMappingURL=data:application/json;base64,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