/* Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end */
import { Component, EventEmitter, Input, Output, forwardRef, inject } from '@angular/core';
import { NG_VALUE_ACCESSOR } from '@angular/forms';
import { autocompletion, completionKeymap } from '@codemirror/autocomplete';
import { HighlightStyle, StreamLanguage, syntaxHighlighting } from "@codemirror/language";
import { jinja2 } from "@codemirror/legacy-modes/mode/jinja2";
import { EditorState, Prec } from '@codemirror/state';
import { keymap, placeholder } from '@codemirror/view';
import { EditorView, basicSetup } from 'codemirror';
import { ThemeService } from '../../core/services/theme.service';
import { CommonModule } from '@angular/common';
import { linter } from '@codemirror/lint';
import { isBoolean } from 'lodash-es';
import { tags } from '@lezer/highlight';
import * as i0 from "@angular/core";
export class CodemirrorComponent {
    constructor(elementRef) {
        this.elementRef = elementRef;
        this.editorView = null;
        this.onChange = null;
        this.onTouched = null;
        this.code = '';
        this.themeService = inject(ThemeService);
        this.currentTheme = this.themeService.getTheme();
        this.focus = new EventEmitter();
        this.blur = new EventEmitter();
        this.extraKeyBindings = [];
        this.placeholder = '';
    }
    ngAfterViewInit() {
        const editorElement = this.elementRef.nativeElement.querySelector('div');
        let jinjaLanguage = StreamLanguage.define(jinja2);
        let customHighlights = HighlightStyle.define(customHighlightSpecs);
        const asyncCompletionSource = (context) => {
            if (!this.autocompleteGetter) {
                return null;
            }
            return this.autocompleteGetter(context);
        };
        const autocompleteExtension = autocompletion({
            override: [asyncCompletionSource], // Use async completion source
            activateOnTyping: true, // Enable suggestions on typing
        });
        let extensions = [
            basicSetup,
            jinjaLanguage,
            autocompleteExtension,
            keymap.of(completionKeymap),
            Prec.highest(keymap.of(this.extraKeyBindings)),
            EditorView.lineWrapping,
            this.handleUpdates(),
            this.handleFocusChanges(),
            placeholder(this.placeholder),
            syntaxHighlighting(customHighlights)
        ];
        if (this.linter) {
            let customLinter = this.linter;
            extensions.push(linter(view => customLinter(view)));
        }
        if (isBoolean(this.code)) {
            this.code = this.code ? '{{ True }}' : '{{ False }}';
        }
        const state = EditorState.create({
            doc: this.code ? this.code.toString() : '',
            extensions: extensions,
        });
        this.editorView = new EditorView({
            state,
            parent: editorElement
        });
        if (this.currentTheme === 'dark') {
            editorElement.classList.add('cm-dark-theme');
        }
    }
    handleUpdates() {
        return EditorView.updateListener.of((viewUpdate) => {
            if (viewUpdate.docChanged && this.onChange) {
                this.onChange(viewUpdate.state.doc.toString());
            }
        });
    }
    handleFocusChanges() {
        return EditorView.focusChangeEffect.of((state, focusing) => {
            if (focusing) {
                this.focus.emit();
            }
            else {
                this.blur.emit();
            }
            return null;
        });
    }
    replaceSelection(text, skipBrackets = false) {
        if (!this.editorView) {
            return;
        }
        const cursorIndex = this.editorView.state.selection.main.head;
        if (!skipBrackets && !this.isInCurlyBraces(this.editorView.state.doc.toString(), cursorIndex)) {
            text = `{{ ${text} }}`;
        }
        const tr = this.editorView.state.replaceSelection(text);
        this.editorView.dispatch(tr);
        setTimeout(() => {
            this.setFocus();
        });
    }
    isInCurlyBraces(s, index) {
        if (index < 0 || index >= s.length) {
            return false;
        }
        let i = 0;
        while (i < s.length) {
            if (s.substring(i, i + 2) === '{{') {
                const endIndex = s.indexOf('}}', i + 2);
                if (endIndex === -1) {
                    break;
                }
                if (i + 2 < index && index <= endIndex + 1) {
                    return true;
                }
                i = endIndex + 2;
            }
            else {
                i++;
            }
        }
        return false;
    }
    ngOnDestroy() {
        if (this.editorView) {
            this.editorView.destroy();
        }
    }
    // ControlValueAccessor methods
    writeValue(value) {
        this.code = value;
        if (this.editorView) {
            this.editorView.dispatch({
                changes: {
                    from: 0,
                    to: this.editorView.state.doc.length,
                    insert: value,
                },
            });
        }
    }
    registerOnChange(fn) {
        this.onChange = fn;
    }
    registerOnTouched(fn) {
        this.onTouched = fn;
    }
    setFocus() {
        this.editorView?.focus();
        const location = this.code.length;
        this.editorView?.dispatch({ selection: { anchor: location, head: location } });
    }
    static { this.ɵfac = function CodemirrorComponent_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || CodemirrorComponent)(i0.ɵɵdirectiveInject(i0.ElementRef)); }; }
    static { this.ɵcmp = /*@__PURE__*/ i0.ɵɵdefineComponent({ type: CodemirrorComponent, selectors: [["xf-code-mirror"]], inputs: { autocompleteGetter: "autocompleteGetter", linter: "linter", extraKeyBindings: "extraKeyBindings", placeholder: "placeholder" }, outputs: { focus: "focus", blur: "blur" }, standalone: true, features: [i0.ɵɵProvidersFeature([
                {
                    provide: NG_VALUE_ACCESSOR,
                    useExisting: forwardRef(() => CodemirrorComponent),
                    multi: true,
                },
            ]), i0.ɵɵStandaloneFeature], decls: 1, vars: 0, template: function CodemirrorComponent_Template(rf, ctx) { if (rf & 1) {
            i0.ɵɵelement(0, "div");
        } }, dependencies: [CommonModule], encapsulation: 2 }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(CodemirrorComponent, [{
        type: Component,
        args: [{
                standalone: true,
                imports: [CommonModule],
                selector: 'xf-code-mirror',
                template: '<div></div>',
                providers: [
                    {
                        provide: NG_VALUE_ACCESSOR,
                        useExisting: forwardRef(() => CodemirrorComponent),
                        multi: true,
                    },
                ]
            }]
    }], () => [{ type: i0.ElementRef }], { focus: [{
            type: Output
        }], blur: [{
            type: Output
        }], autocompleteGetter: [{
            type: Input
        }], linter: [{
            type: Input
        }], extraKeyBindings: [{
            type: Input
        }], placeholder: [{
            type: Input
        }] }); })();
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassDebugInfo(CodemirrorComponent, { className: "CodemirrorComponent", filePath: "lib/components/codemirror/codemirror.component.ts", lineNumber: 45 }); })();
const customHighlightSpecs = [
    {
        tag: [tags.atom, tags.bool, tags.url, tags.contentSeparator, tags.labelName],
        class: 'cm-atom', style: ''
    },
    {
        tag: tags.keyword,
        class: 'cm-keyword',
        color: ''
    },
    {
        tag: tags.meta,
        color: "#404740"
    },
    {
        tag: tags.link,
        textDecoration: "underline"
    },
    {
        tag: tags.heading,
        textDecoration: "underline",
        fontWeight: "bold"
    },
    {
        tag: tags.emphasis,
        fontStyle: "italic"
    },
    {
        tag: tags.strong,
        fontWeight: "bold"
    },
    {
        tag: tags.strikethrough,
        textDecoration: "line-through"
    },
    {
        tag: [tags.literal, tags.inserted],
        color: "#164"
    },
    {
        tag: [tags.string, tags.deleted],
        color: "#a11"
    },
    {
        tag: [tags.regexp, tags.escape, tags.special(tags.string)],
        color: "#e40"
    },
    {
        tag: tags.definition(tags.variableName),
        color: "#00f"
    },
    {
        tag: tags.local(tags.variableName),
        color: "#30a"
    },
    {
        tag: [tags.typeName, tags.namespace],
        color: "#085"
    },
    {
        tag: tags.className,
        color: "#167"
    },
    {
        tag: [tags.special(tags.variableName), tags.macroName],
        color: "#256"
    },
    {
        tag: tags.definition(tags.propertyName),
        color: "#00c"
    },
    {
        tag: tags.comment,
        color: "#940"
    },
    {
        tag: tags.invalid,
        color: "#f00"
    }
];
//# sourceMappingURL=data:application/json;base64,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