/* Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end */
import { HttpParams } from '@angular/common/http';
import { Injectable, inject } from '@angular/core';
import { plainToInstance } from 'class-transformer';
import { cloneDeep, forEach, isArray, isBoolean, isString, isUndefined, keyBy, merge, sortBy, values } from 'lodash-es';
import { Subject, combineLatest, of } from 'rxjs';
import { catchError, concatMap, map, tap } from 'rxjs/operators';
import { Entity } from '../models/entity';
import { Field } from '../models/field';
import { ApiService } from './api.service';
import { IntegerField } from "../../form-fields/field-types/integer";
import { ManyToManyField } from "../../form-fields/field-types/manytomany";
import { LookupField } from "../../form-fields/field-types/lookup";
import { TagsField } from "../../form-fields/field-types/tags";
import { DatetimeField } from "../../form-fields/field-types/datetime";
import { CheckboxField } from "../../form-fields/field-types/checkbox";
import { Query } from '../../query-filters/models/query';
import { FilterGroup } from '../../query-filters/models/filter-group';
import { Filter } from '../../query-filters/models/filter';
import { ModuleAttributeMap } from '../models/attribute-map';
import * as i0 from "@angular/core";
import * as i1 from "./model-metadata.service";
import * as i2 from "./permission.service";
import * as i3 from "./utility.service";
import * as i4 from "../../utility/pipes/get-end-path-name.pipe";
import * as i5 from "../../utility/pipes/interpolate.pipe";
import * as i6 from "../../utility/pipes/unix-to-date.pipe";
export class EntityService {
    constructor(modelMetadataService, permissionService, utilityService, getEndPathNamePipe, interpolate, unixToDatePipe) {
        this.modelMetadataService = modelMetadataService;
        this.permissionService = permissionService;
        this.utilityService = utilityService;
        this.getEndPathNamePipe = getEndPathNamePipe;
        this.interpolate = interpolate;
        this.unixToDatePipe = unixToDatePipe;
        this.apiService = inject(ApiService);
        this.loading$ = {};
        this.displayNameMapping = {};
        this.traitFields = {
            indexable: [
                plainToInstance(IntegerField, {
                    name: 'id',
                    descriptions: {
                        singular: 'ID'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'integer',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: true,
                    title: 'ID',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                })
            ],
            ownable: [
                plainToInstance(ManyToManyField, {
                    name: 'owners',
                    bulkAction: [],
                    dataSource: {
                        model: 'teams',
                        query: []
                    },
                    descriptions: {
                        singular: 'Owners'
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'manyToMany',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Owners',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: true
                })
            ],
            queueable: [
                plainToInstance(LookupField, {
                    name: 'xfQueue',
                    descriptions: {
                        singular: 'Queue'
                    },
                    bulkAction: {
                        allow: true,
                        buttonClass: 'btn btn-secondary btn-sm',
                        buttonIcon: 'fa fa-users fa-light',
                        buttonText: 'Change Queue'
                    },
                    dataSource: {
                        model: 'xf_queues',
                        query: {
                            filters: [],
                            logic: 'AND'
                        }
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'lookup',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Queue',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: true
                })
            ],
            taggable: [
                plainToInstance(TagsField, {
                    name: 'recordTags',
                    descriptions: {
                        singular: 'Tags'
                    },
                    bulkAction: [],
                    dataSource: {
                        model: 'tags',
                        query: []
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    baseUrl: '',
                    formType: 'tags',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Tags',
                    tooltip: 'Special characters # ? , \' \'\' and / are not allowed in tag.',
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: true
                })
            ],
            trackable: [
                plainToInstance(LookupField, {
                    name: 'createUser',
                    descriptions: {
                        singular: 'Created By'
                    },
                    bulkAction: [],
                    dataSource: {
                        model: 'users',
                        query: {
                            filters: [{
                                    field: 'type',
                                    operator: 'eq',
                                    value: 'sso'
                                }],
                            logic: "AND"
                        }
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'lookup',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Created By',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                }),
                plainToInstance(DatetimeField, {
                    name: 'createDate',
                    descriptions: {
                        singular: 'Created On'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'datetime',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Created On',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                }),
                plainToInstance(LookupField, {
                    name: 'modifyUser',
                    descriptions: {
                        singular: 'Modified By'
                    },
                    bulkAction: [],
                    dataSource: {
                        model: 'users',
                        query: {
                            filters: [{
                                    field: 'type',
                                    operator: 'eq',
                                    value: 'sso'
                                }],
                            logic: "AND"
                        }
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'lookup',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Modified By',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                }),
                plainToInstance(DatetimeField, {
                    name: 'modifyDate',
                    descriptions: {
                        singular: 'Modified On'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'datetime',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Modified On',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                })
            ],
            peerReplicable: [
                plainToInstance(LookupField, {
                    name: 'tenant',
                    descriptions: {
                        singular: 'Tenant'
                    },
                    bulkAction: [],
                    dataSource: {
                        model: 'tenants',
                        query: {
                            logic: 'AND',
                            filters: null,
                            sort: [
                                {
                                    field: 'name',
                                    direction: 'ASC'
                                }
                            ]
                        }
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'lookup',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: true,
                    title: 'Tenant',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                }),
                plainToInstance(CheckboxField, {
                    name: 'conflict',
                    descriptions: {
                        singular: 'Conflict'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: false,
                    encrypted: false,
                    exportable: false,
                    formType: 'checkbox',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Conflict',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: false,
                    writeable: false
                }),
                plainToInstance(IntegerField, {
                    name: 'tenantRecordId',
                    descriptions: {
                        singular: 'Tenant Record ID'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: false,
                    encrypted: false,
                    exportable: false,
                    formType: 'integer',
                    gridColumn: true,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: true,
                    title: 'Tenant Record ID',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                })
            ],
            conditionalOwnable: [
                plainToInstance(ManyToManyField, {
                    name: 'owners',
                    descriptions: {
                        singular: 'Owners'
                    },
                    bulkAction: [],
                    dataSource: {
                        model: 'teams',
                        query: []
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'manyToMany',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Owners',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: true
                }),
                plainToInstance(CheckboxField, {
                    name: 'isPrivate',
                    descriptions: {
                        singular: 'Is Private'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'checkbox',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Is Private',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: true
                })
            ],
            softDelete: [
                plainToInstance(CheckboxField, {
                    name: 'is_deleted',
                    descriptions: {
                        singular: 'Is Deleted'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: false,
                    encrypted: false,
                    exportable: false,
                    formType: 'checkbox',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Is Deleted',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: false,
                    writeable: false
                }),
                plainToInstance(DatetimeField, {
                    name: 'deleted_on',
                    descriptions: {
                        singular: 'Deleted On'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'datetime',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Deleted On',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: false
                })
            ],
            userOwnable: [
                plainToInstance(Field, {
                    name: 'assignee',
                    descriptions: {
                        singular: 'Assignee'
                    },
                    bulkAction: [],
                    dataSource: null,
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: null,
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'Assignee',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: true
                }),
                plainToInstance(ManyToManyField, {
                    name: 'userOwners',
                    descriptions: {
                        singular: 'User Owners'
                    },
                    bulkAction: [],
                    dataSource: {
                        model: 'people',
                        query: []
                    },
                    dataSourceFilters: null,
                    defaultValue: null,
                    encrypted: false,
                    exportable: false,
                    formType: 'manyToMany',
                    gridColumn: false,
                    htmlEscape: false,
                    peerReplicable: true,
                    searchable: false,
                    title: 'User Owners',
                    tooltip: null,
                    unique: false,
                    validation: [],
                    visibility: true,
                    writeable: true
                })
            ]
        };
    }
    getEntities() {
        return this.modelMetadataService.getModuleList().pipe(concatMap((modules) => {
            return this.castEntities(modules).pipe(map((entityList) => {
                return keyBy(entityList, 'type');
            }));
        }));
    }
    getEntityList() {
        return this.getEntities().pipe(map((entities) => values(entities)));
    }
    castEntities(modules) {
        let entities$ = [];
        modules.forEach((module) => {
            let entity = plainToInstance(Entity, cloneDeep(module));
            entity.attributes = sortBy(entity.attributes, 'orderIndex');
            let fieldTypes$ = [];
            entity.attributes.forEach((attribute, attributeIndex) => {
                // if (attribute instanceof Field && attribute.formType) {
                //   const attributeObject = module.attributes[attributeIndex];
                //   let fieldType$ = this.fieldService.getFieldRenderer(attributeObject.formType).pipe(
                //     map((fieldType: Type<Field> | null) => {
                //       if (fieldType) {
                //         entity.attributes[attributeIndex] = plainToInstance(fieldType, attributeObject as object);
                //       } else {
                //         entity.attributes[attributeIndex] = plainToInstance(Field, attributeObject as object);
                //       }
                //     })
                //   );
                //   fieldTypes$.push(fieldType$);
                // }
            });
            if (fieldTypes$.length) {
                entities$.push(combineLatest(fieldTypes$).pipe(map(() => entity)));
            }
            else {
                entities$.push(of(entity));
            }
        });
        return combineLatest(entities$);
    }
    build(type, mode = 'add') {
        let entity$;
        let metadata = this.modelMetadataService.getMetadataByModuleType(type);
        if (!metadata) {
            throw new Error('Metadata for "' + type + '" not found');
        }
        entity$ = this.castEntities([metadata]).pipe(map((entities) => {
            if (!entities.length) {
                return {};
            }
            else {
                return entities[0];
            }
        }));
        return entity$.pipe(concatMap((entity) => {
            let updatedEntity$;
            if (entity.parentType) { // handles only one level of inheritance as in our product we have only one level as of now
                updatedEntity$ = this.castEntities([this.modelMetadataService.getMetadataByModuleType(entity.parentType)]).pipe(map((parentEntities) => {
                    if (parentEntities.length) {
                        entity.attributes = entity.attributes.concat(parentEntities[0].attributes);
                    }
                    return entity;
                }));
            }
            else {
                updatedEntity$ = of(entity);
            }
            return updatedEntity$.pipe(map((entity) => {
                this.buildFields(entity, mode);
                let canUpdate = this.permissionService.availablePermission(entity.type, 'update');
                entity.attributes = entity.attributes.map((attribute) => {
                    const fieldPermissionWrite = this.permissionService.availableFieldPermission(self.name, attribute.name, 'write');
                    attribute.readOnly = !canUpdate || !fieldPermissionWrite;
                    attribute.editable = attribute.editable && !attribute.readOnly;
                    return attribute;
                });
                entity.evaluateAllFields();
                return entity;
            }));
        }));
    }
    hydrate(entity, fieldNames = true, URLparams = new HttpParams()) {
        if (!entity || !entity.uuid) {
            return of();
        }
        let queryBody = {
            filters: [{
                    field: 'uuid',
                    operator: 'eq',
                    value: entity.uuid
                }],
            logic: 'AND'
        };
        if (isArray(fieldNames)) {
            queryBody.fields = fieldNames;
        }
        return this.apiService.query(entity, queryBody).pipe(map((response) => {
            return response['hydra:member'][0];
        }), map((values) => {
            entity.values = merge(entity.values, values);
            entity.display = this.getDisplayValue(entity, entity.values);
            return entity;
        }));
    }
    getRecordDisplay(entity, uuid) {
        if (this.displayNameMapping[uuid]) {
            return of(this.displayNameMapping[uuid]);
        }
        else if (this.loading$[uuid]) {
            return this.loading$[uuid];
        }
        this.loading$[uuid] = new Subject();
        return this.apiService.getRecord(entity, uuid, undefined, true).pipe(catchError(() => {
            this.displayNameMapping[uuid] = 'System';
            this.loading$[uuid].next(this.displayNameMapping[uuid]);
            this.loading$[uuid].complete();
            delete this.loading$[uuid];
            return of();
        }), map((item) => {
            let display = this.getDisplayValue(entity, item);
            this.displayNameMapping[uuid] = display || 'No display';
            this.loading$[uuid].next(display);
            this.loading$[uuid].complete();
            delete this.loading$[uuid];
            return display;
        }));
    }
    clearDisplayCache(uuid) {
        if (isBoolean(uuid) && uuid) {
            this.displayNameMapping = {};
            return;
        }
        else if (isString(uuid)) {
            delete this.displayNameMapping[uuid];
        }
    }
    save(entity, fieldNames) {
        let body = {};
        fieldNames.forEach((fieldName) => {
            // if entity.values[fieldName], fails for boolean fields with value false
            if (fieldName in entity.values) {
                body[fieldName] = entity.values[fieldName];
            }
        });
        //Update existing entity
        if (entity.uuid) {
            const uuid = entity.uuid;
            return this.apiService.updateRecord(entity, uuid, body).pipe(tap(() => this.clearDisplayCache(uuid)), map((values) => {
                entity.values = merge(entity.values, values);
                entity.display = this.getDisplayValue(entity, entity.values);
                return entity;
            }));
        }
        //Create new entity
        else {
            return this.apiService.createRecord(entity, body).pipe(map((values) => {
                entity.values = merge(entity.values, values);
                entity.display = this.getDisplayValue(entity, entity.values);
                return entity;
            }));
        }
    }
    buildFields(entity, mode) {
        this.addTraitFields(entity);
        entity.attributes.sort((attribute) => attribute.orderIndex || 99999);
        entity.attributes.forEach((property) => {
            if (property) {
                property.setMode(mode);
                if (property.model) {
                    this.setDisplayName(property, entity);
                }
                // if (property.formType === 'image') {
                //   let imagePromise = loadImage(field);// we are not using this function. only in PB collection we use image. So if we expose this property then this is required.
                // } else {
            }
        });
    }
    //TODO: for case management update this.
    addTraitFields(entity) {
        entity.attributes = entity.attributes.concat(entity.indexable ? this.traitFields.indexable : []);
        entity.attributes = entity.attributes.concat(entity.ownable ? this.traitFields.ownable : []);
        entity.attributes = entity.attributes.concat(entity.peerReplicable ? this.traitFields.peerReplicable : []);
        // entity.attributes = entity.attributes.concat(entity.taggable ? this.traitFields.taggable as any[]: []);
        entity.attributes = entity.attributes.concat(entity.trackable ? this.traitFields.trackable : []);
        entity.attributes = entity.attributes.concat(entity.queueable ? this.traitFields.queueable : []);
        entity.attributes = entity.attributes.concat(entity.conditionalOwnable ? this.traitFields.conditionalOwnable : []);
        entity.attributes = entity.attributes.concat(entity.userOwnable ? this.traitFields.userOwnable : []);
    }
    setDisplayName(field, entity) {
        if (entity && entity.displayName) {
            field.displayTemplate = entity.displayName || '';
            field.defaultSort = field.defaultSort && entity.defaultSort ? entity.defaultSort : [];
            field.trackable = entity.trackable || false;
        }
        else {
            if (!field.displayTemplate) {
                field.displayTemplate = '';
            }
        }
        let value = entity.values[field.name];
        if (value && typeof (value) === 'object') {
            if (Array.isArray(value) && value.length > 0) {
                if (field.formType === Field.BuiltInType.TAGS) {
                    field.display = '';
                    value.forEach((item) => {
                        if (item.display === '') {
                            item.display = this.getEndPathNamePipe.transform(item);
                        }
                        else {
                            item.display = item.display + ', ' + this.getEndPathNamePipe.transform(item);
                        }
                    });
                }
                // else {//added for multislect picklist
                //   field.display = '';
                //   value.forEach((item) => {
                //     if (field.display === '') {
                //       value.display = this.interpolate.transform(field.displayTemplate, item);
                //     } else {
                //       value.display = value.display + ', ' + this.interpolate.transform(field.displayTemplate, item);
                //     }
                //   });
                // }
            }
            else {
                if (field.displayTemplate) {
                    value.display = this.interpolate.transform(field.displayTemplate, value);
                }
            }
        }
        return field;
    }
    getDisplayValue(entity, data) {
        return this.interpolate.transform(entity.displayName, data);
    }
    getDisplayFields(entity) {
        let displayNameRegex = /\{\{\s*(?<fieldName>\w+)\.?\w*\s*\}\}/g;
        let result, fields = [];
        while ((result = displayNameRegex.exec(entity.displayName)) !== null) {
            if (result?.groups?.fieldName) {
                fields.push(result.groups.fieldName);
            }
        }
        return fields;
    }
    applyValues(entity, fields, data) {
        forEach(data, (value, key) => {
            if (key.indexOf('@') !== 0 && !fields[key] && isArray(value)) {
                fields[key] = new Field(key, Field.BuiltInType.TEXT);
            }
        });
        forEach(fields, (field) => {
            if (data && !isUndefined(data[field.name]) && data[field.name] !== null) {
                var value = data[field.name];
                switch (field.formType) {
                    case Field.BuiltInType.INTEGER:
                    case Field.BuiltInType.DECIMAL:
                        value = parseFloat(value);
                        break;
                    case Field.BuiltInType.DATETIME:
                        value = this.unixToDatePipe.transform(value);
                        break;
                        // case 'date':
                        //  value = $filter('unixToDate')(value);
                        break;
                    // case 'object':
                    //   //https://stackoverflow.com/questions/5491605/empty-arrays-seem-to-equal-true-and-false-at-the-same-time
                    //   if(value == ""){// jshint ignore:line
                    //     value = {};
                    //   }
                    //   break;
                    default:
                        break;
                }
                //field.value = value;
                entity.values[field.name] = value;
            }
        });
        forEach(fields, (field) => {
            this.setDisplayName(field, entity);
        });
        return fields;
    }
    buildSearchQuery(fields, search) {
        let searchWords = search.match(/\w+/g);
        let andFilters = [];
        searchWords?.forEach((searchWord) => {
            let orFilters = [];
            fields.forEach((field) => {
                orFilters.push(new Filter(field, 'icontains', searchWord));
            });
            andFilters.push(new FilterGroup(orFilters, 'OR'));
        });
        let queryObject = {
            filters: andFilters,
            logic: 'AND'
        };
        if (andFilters.length === 1) {
            queryObject = andFilters[0];
        }
        let query = new Query(queryObject);
        return query;
    }
    buildSearchQueryFromDisplayName(entity, search) {
        let fields = [];
        const displayFieldMatches = entity.displayName.matchAll(/{{(?<type>[\s\S]+?)}}/g);
        for (const match of displayFieldMatches) {
            const requiredKey = match?.groups?.type.trim();
            if (requiredKey) {
                fields.push(requiredKey);
            }
        }
        return this.buildSearchQuery(fields, search);
    }
    createEntityFromData(inputData = {}, name = 'filterEntity') {
        const data = cloneDeep(inputData);
        let entity = new Entity(name);
        entity.attributes = [];
        let attr = {};
        forEach(data, (value, key) => {
            let dataType = this.utilityService.getFieldTypeByValue(value);
            let field = new ModuleAttributeMap[dataType].field(key);
            field.title = key;
            if (field.formType === 'json') {
                field.data = value;
            }
            attr[key] = field;
        });
        entity.fields = attr;
        entity.values = data;
        return entity;
    }
    static { this.ɵfac = function EntityService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || EntityService)(i0.ɵɵinject(i1.ModelMetadataService), i0.ɵɵinject(i2.PermissionService), i0.ɵɵinject(i3.UtilityService), i0.ɵɵinject(i4.GetEndPathNamePipe), i0.ɵɵinject(i5.InterpolatePipe), i0.ɵɵinject(i6.UnixToDatePipe)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: EntityService, factory: EntityService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(EntityService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.ModelMetadataService }, { type: i2.PermissionService }, { type: i3.UtilityService }, { type: i4.GetEndPathNamePipe }, { type: i5.InterpolatePipe }, { type: i6.UnixToDatePipe }], null); })();
//# sourceMappingURL=data:application/json;base64,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