/* Copyright start
  Copyright (C) 2008 - 2025 Fortinet Inc.
  All rights reserved.
  FORTINET CONFIDENTIAL & FORTINET PROPRIETARY SOURCE CODE
  Copyright end */
import { Injectable } from '@angular/core';
import _ from 'lodash-es';
import { forEach, isUndefined } from 'lodash-es';
import { ReplaySubject, Subject, map, of } from 'rxjs';
import { v4 as uuid } from 'uuid';
import { API } from '../constants';
import { Field } from '../models/field';
import * as i0 from "@angular/core";
import * as i1 from "@angular/router";
import * as i2 from "xng-breadcrumb";
export class UtilityService {
    constructor(router, route, breadcrumbService) {
        this.router = router;
        this.route = route;
        this.breadcrumbService = breadcrumbService;
        // Individual components can subscribe to this to receive document click events
        this.documentClickedTarget = new Subject();
        this.navigationChanged = new ReplaySubject(1);
        this.permissionsChanged = new Subject();
    }
    generateUUID() {
        return uuid();
    }
    setTitleAlias(alias, value, newPath) {
        let breadcrumb = {
            label: value,
        };
        if (newPath) {
            breadcrumb.routeInterceptor = () => newPath;
        }
        return this.breadcrumbService.set('@' + alias, breadcrumb);
    }
    isUndefined(value) {
        return value === undefined || value === null;
    }
    convertHexToRgbA(value) {
        value = value === undefined || value === null ? '#000000' : value;
        let hexcode;
        if (/^#([A-Fa-f0-9]{3}){1,2}$/.test(value)) {
            hexcode = value.substring(1).split('');
            if (hexcode.length === 3) {
                hexcode = [
                    hexcode[0],
                    hexcode[0],
                    hexcode[1],
                    hexcode[1],
                    hexcode[2],
                    hexcode[2],
                ];
            }
            hexcode = '0x' + hexcode.join('');
            return ('rgba(' +
                [(hexcode >> 16) & 255, (hexcode >> 8) & 255, hexcode & 255].join(',') +
                ',0.8)');
        }
        else {
            return 'rgba(0, 0, 0, 0.1)';
        }
    }
    //set parameters in URL
    setParametersInUrl(queryParams) {
        forEach(queryParams, (paramValue, paramKey) => {
            if (paramValue === '' || paramValue === null) {
                delete queryParams[paramKey];
            }
        });
        this.router.navigate([], {
            relativeTo: this.route,
            queryParams: queryParams,
        });
    }
    //append new parameter in URL
    appendNewParameterInUrl(newQueryParams) {
        this.router.navigate([], {
            relativeTo: this.route,
            queryParams: newQueryParams,
            queryParamsHandling: 'merge',
        });
    }
    filterFields(fields, search, fieldsAlreadyAdded) {
        return of(fields).pipe(map((fields) => {
            return fields.map((field) => field.asSearchableItem());
        }), map((items) => {
            return items.filter((item) => {
                return fieldsAlreadyAdded.indexOf(item.uuid) === -1;
            });
        }), map((items) => {
            return items.filter((item) => {
                return (_.isString(item.display) &&
                    item.display.toLowerCase().indexOf(search.toLowerCase()) > -1);
            });
        }));
    }
    downloadFile(fileData, fileName, fileType) {
        let file;
        let downloadLink;
        // file
        file = new Blob([fileData], { type: fileType });
        const navigator = window.navigator;
        if (navigator && navigator.msSaveOrOpenBlob) {
            navigator.msSaveOrOpenBlob(file, fileName);
        }
        else {
            // Download link
            downloadLink = document.createElement('a');
            // File name
            downloadLink.download = fileName;
            // Create a link to the file
            downloadLink.href = window.URL.createObjectURL(file);
            // Hide download link
            downloadLink.style.display = 'none';
            // Add the link to DOM
            document.body.appendChild(downloadLink);
            // Click download link
            downloadLink.click();
        }
    }
    csvToJSON(CSVcontents) {
        let contents = CSVcontents.split('\r\n');
        let columnNames = contents[0].replace(/\n|\r/g, '').split(',');
        let data = [];
        for (let i = 1; i < contents.length; i++) {
            if (contents[i] !== '' &&
                contents[i] !== null &&
                contents[i] !== undefined) {
                let rowData = contents[i].replace(/\n|\r/g, '').split(',');
                let dataObj = {};
                for (let j = 0; j < rowData.length; j++) {
                    dataObj[columnNames[j]] = rowData[j];
                }
                data.push(dataObj);
            }
        }
        return { columns: columnNames, data: data };
    }
    newAttribute() {
        return {
            // uuid: $window.UUID.generate(),
            isNew: true,
            ownsRelationship: false,
            visibility: true,
            writeable: true,
            editable: true,
            gridColumn: false,
            defaultValue: '',
            value: null,
            validation: {
                required: false,
                minlength: 0,
                // maxlength: MAX_TEXT_LENGTH
            },
            name: '(Untitled)',
            title: '(Untitled)',
            descriptions: {
                singular: '(Untitled)',
            },
            formType: 'text',
            type: 'string',
        };
    }
    convertToCamelCase(string) {
        if (!isUndefined(string)) {
            return string
                .replace(/(?:^\w|[A-Z]|\b\w)/g, function (letter, index) {
                return index === 0 ? letter.toLowerCase() : letter.toUpperCase();
            })
                .replace(/\s+/g, '');
        }
        return string;
    }
    getBaseContextPath() {
        var baseHrefElement = document.querySelector('base');
        if (baseHrefElement) {
            return baseHrefElement.getAttribute('href') || '';
        }
        return '';
    }
    getIriApiPath(IRI) {
        if (IRI.charAt(0) === '/') {
            return IRI.substring(1);
        }
        // else if (IRI.indexOf('currentUserIri') > -1){
        //   return $interpolate(IRI)({currentUserIri: currentUserIriService});
        // }
        return IRI;
    }
    getOrigin() {
        return (location.protocol +
            '://' +
            location.host +
            ':' +
            location.port +
            this.getBaseContextPath());
    }
    prependIri(value) {
        return `${API.API_3_BASE}${value}`;
    }
    formatBytes(a, b = 2) {
        if (!+a)
            return '0 Bytes';
        const c = 0 > b ? 0 : b;
        const d = Math.floor(Math.log(a) / Math.log(1024));
        return `${parseFloat((a / Math.pow(1024, d)).toFixed(c))} ${['Bytes', 'KiB', 'MiB', 'GiB', 'TiB', 'PiB', 'EiB', 'ZiB', 'YiB'][d]}`;
    }
    getTimeInterval(toDateStr, fromDateStr, ms_time) {
        let toDate, fromDate, ms;
        if (!ms_time) {
            toDate = new Date(toDateStr);
            fromDate = new Date(fromDateStr);
            ms = toDate - fromDate;
        }
        else {
            ms = ms_time;
        }
        let seconds = ms >= 1000 ? ms / 1000 : parseInt((ms / 1000).toString(), 10);
        ms = ms % 1000;
        let minutes = parseInt((seconds / 60).toString(), 10);
        seconds = parseInt((seconds % 60).toString(), 10);
        let hours = parseInt((minutes / 60).toString(), 10);
        minutes = minutes % 60;
        let returnVal = '';
        if (hours > 0) {
            returnVal += Math.round(hours) + ' hours ';
        }
        if (minutes > 0) {
            returnVal += Math.round(minutes) + ' minutes ';
        }
        if (seconds > 0) {
            returnVal += Math.round(seconds) + ' seconds ';
        }
        if (ms > 0) {
            returnVal += Math.round(ms) + ' ms';
        }
        return returnVal;
    }
    addJinjaBraces(value) {
        return `{{${value}}}`;
    }
    removeJinjaBraces(value) {
        return value.replace('{{', '').replace('}}', '').trim();
    }
    getTimeAgo(fromDateStr, toDateStr, ms_time) {
        let toDate, fromDate, ms;
        if (!ms_time) {
            toDate = toDateStr ? new Date(toDateStr) : new Date();
            fromDate = new Date(fromDateStr);
            ms = toDate - fromDate;
        }
        else {
            ms = ms_time;
        }
        let seconds = ms >= 1000 ? ms / 1000 : parseInt((ms / 1000).toString(), 10);
        ms = ms % 1000;
        let minutes = parseInt((seconds / 60).toString(), 10);
        seconds = parseInt((seconds % 60).toString(), 10);
        let hours = parseInt((minutes / 60).toString(), 10);
        minutes = minutes % 60;
        let days = parseInt((hours / 24).toString(), 10);
        let months = parseInt((days / 30).toString(), 10);
        let years = parseInt((months / 12).toString(), 10);
        let returnVal = '';
        if (ms > 0) {
            returnVal = Math.round(ms) + ' ms';
        }
        if (seconds > 0) {
            returnVal = Math.round(seconds) + ' second(s) ';
        }
        if (minutes > 0) {
            returnVal = Math.round(minutes) + ' minute(s) ';
        }
        if (hours > 0) {
            returnVal = Math.round(hours) + ' hour(s) ';
        }
        if (days > 0) {
            returnVal = Math.round(days) + ' day(s) ';
        }
        if (months > 0) {
            returnVal = Math.round(months) + ' month(s) ';
        }
        if (years > 0) {
            returnVal = Math.round(years) + ' year(s) ';
        }
        return returnVal;
    }
    typeOf(value) {
        if (_.isArray(value)) {
            return 'list';
        }
        else if (_.isNumber(value)) {
            return 'number';
        }
        else if (_.isObject(value)) {
            return 'dict';
        }
        else if (_.isString(value)) {
            return 'string';
        }
        else if (_.isBoolean(value)) {
            return 'boolean';
        }
        else {
            return 'any';
        }
    }
    getFieldTypeByValue(value) {
        let dataType = Field.BuiltInType.TEXT;
        try {
            let val = JSON.parse(value);
            if (_.isNumber(val)) {
                dataType = Field.BuiltInType.DECIMAL;
            }
            else if (_.isBoolean(val)) {
                dataType = Field.BuiltInType.CHECKBOX;
            }
        }
        catch (error) {
            if (_.isArray(value)) {
                dataType = Field.BuiltInType.ARRAY;
            }
            else if (_.isObject(value)) {
                dataType = Field.BuiltInType.JSON;
            }
            else if (new Date(value).toString() !== 'Invalid Date') {
                dataType = Field.BuiltInType.DATETIME;
            }
        }
        return dataType;
    }
    getHigherVersionNumber(version1, version2) {
        const v1split = version1.split('.').map(Number);
        const v2split = version2.split('.').map(Number);
        let index = 0;
        while (index < v1split.length) {
            if (index === v2split.length) {
                return version1;
            }
            const v1Value = v1split[index];
            const v2Value = v2split[index];
            if (v1Value === v2Value) {
                index++;
                continue;
            }
            else if (v1Value > v2Value) {
                return version1;
            }
            else if (v2Value > v1Value) {
                return version2;
            }
        }
        return version2;
    }
    copyToClipboard(value) {
        return new Promise((resolve, reject) => {
            if (navigator.clipboard) {
                navigator.clipboard.writeText(value).then(() => {
                    resolve();
                }).catch((error) => {
                    reject(error);
                });
            }
            else {
                reject('Clipboard API not supported');
            }
        });
    }
    executeMethod(methodName, componentRef) {
        try {
            componentRef[methodName]();
        }
        catch {
            console.warn(`Method ${methodName} not found!`);
        }
    }
    static { this.ɵfac = function UtilityService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || UtilityService)(i0.ɵɵinject(i1.Router), i0.ɵɵinject(i1.ActivatedRoute), i0.ɵɵinject(i2.BreadcrumbService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: UtilityService, factory: UtilityService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(UtilityService, [{
        type: Injectable,
        args: [{
                providedIn: 'root',
            }]
    }], () => [{ type: i1.Router }, { type: i1.ActivatedRoute }, { type: i2.BreadcrumbService }], null); })();
//# sourceMappingURL=data:application/json;base64,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