import { Injectable, inject } from '@angular/core';
import _ from 'lodash-es';
import { isBoolean, keys } from 'lodash-es';
import { concatMap, exhaustMap, map, of } from 'rxjs';
import { DEFAULT_GRID_ROWS, EntityService, Filter, FilterGroup, OPERATOR_MAPPINGS, Query } from "shared";
import * as i0 from "@angular/core";
import * as i1 from "@angular/common/http";
import * as i2 from "shared";
export class GridDataSourceService {
    constructor(http, apiService, dateService, getEndPathNamePipe) {
        this.http = http;
        this.apiService = apiService;
        this.dateService = dateService;
        this.getEndPathNamePipe = getEndPathNamePipe;
        this.entityService = inject(EntityService);
    }
    getDataSource(entity, baseQuery, dataInterceptor, alwaysIncludeColList = [], treeFilterField) {
        let subscription;
        return {
            getRows: (params) => {
                params.api.hideOverlay();
                const availableFieldNames = entity.attributes.filter((field) => field.gridColumn).map((field) => field.name);
                const visibleColIds = params.api.getAllDisplayedColumns().map((col) => col.getColId());
                const visibleColumnFields = availableFieldNames.filter((fieldName) => visibleColIds.indexOf(fieldName) > -1 || alwaysIncludeColList.indexOf(fieldName) > -1);
                let defaultQuery = _.cloneDeep(baseQuery);
                if (treeFilterField) { // for treedata
                    if (!defaultQuery) {
                        defaultQuery = new Query();
                    }
                    let parentId = null;
                    if (params.request.groupKeys.length > 0) {
                        parentId = this.getEndPathNamePipe.transform(params.request.groupKeys.slice(-1)[0]);
                    }
                    defaultQuery?.filters.push(new Filter(treeFilterField, 'eq', parentId));
                }
                subscription = this.loadRows(entity, params.request, defaultQuery, visibleColumnFields).pipe(exhaustMap((response) => {
                    return of(response);
                })).subscribe((response) => {
                    let modifiedData = dataInterceptor ? dataInterceptor(response['hydra:member']) : response['hydra:member'];
                    params.success({
                        rowData: this.loadLookupData(modifiedData, entity),
                        rowCount: response['hydra:totalItems']
                    });
                    if (params.api.paginationGetRowCount() === 0) {
                        params.api.showNoRowsOverlay();
                    }
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    loadLookupData(rows, entity) {
        let lookupFields = entity.attributes.filter((field) => field.formType === 'lookup');
        rows.forEach((row) => {
            lookupFields.forEach((field) => {
                // convert lookups to observables
                const uuid = row[field.name];
                if (!uuid) {
                    row[field.name] = of(field.blankDisplay || '');
                    return;
                }
                row[field.name] = this.entityService.build(field.dataSource.model).pipe(concatMap((subEntity) => this.entityService.getRecordDisplay(subEntity, uuid)));
            });
        });
        // Free up memory
        this.entityService.clearDisplayCache(true);
        return rows;
    }
    getDatasourceByParams(entity, queryParams) {
        let subscription;
        return {
            getRows: (params) => {
                params.api.hideOverlay();
                subscription = this.apiService.getList(entity, undefined, queryParams)
                    .subscribe((rowData) => {
                    params.success({
                        rowData: rowData,
                        // rowCount: 50
                    });
                    if (rowData.length === 0) {
                        params.api.showNoRowsOverlay();
                    }
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    getDatasourceByCustomRequest(request = { url: '', type: 'GET', body: {} }, queryParams, entity, dataInterceptor) {
        let subscription;
        return {
            getRows: (params) => {
                let request$ = request.type === 'GET' ? this.http.get(request.url, { params: queryParams }) : this.http.post(request.url, request.body, { params: queryParams });
                subscription = request$.subscribe((response) => {
                    let modifiedData = dataInterceptor ? dataInterceptor(response['hydra:member']) : response['hydra:member'];
                    params.success({
                        rowData: modifiedData,
                        rowCount: response['hydra:totalItems']
                    });
                });
            },
            selectedRows: [],
            destroy: () => {
                subscription?.unsubscribe();
            },
            entity: entity
        };
    }
    loadRows(entity, request, baseQuery, availableFieldNames = []) {
        let queryParams = {};
        queryParams.offset = request.startRow || 0;
        queryParams.limit = (request.endRow || DEFAULT_GRID_ROWS) - queryParams.offset;
        const query = this.convertAgGridQueryToQueryApi(request.filterModel, request.sortModel, request.groupKeys);
        let fullQuery;
        if (baseQuery) {
            fullQuery = {
                filters: [],
                logic: 'AND'
            };
            if (query.filters.length > 0) {
                fullQuery.filters.push(query.getQueryObject());
            }
            if (baseQuery.filters.length > 0) {
                fullQuery.filters.push(baseQuery.getQueryObject());
            }
            if (query.sort.length) {
                fullQuery.sort = query.sort;
            }
        }
        else {
            fullQuery = query.getQueryObject();
        }
        if (availableFieldNames.length > 0) {
            // UUID is required
            availableFieldNames.push('uuid');
            availableFieldNames = _.uniq(availableFieldNames);
            fullQuery.fields = availableFieldNames;
        }
        if (!fullQuery.sort?.length && entity.defaultSort?.length) {
            fullQuery.sort = entity.defaultSort;
        }
        if (!fullQuery.sort?.length) {
            console.warn(`${entity.title} does not have a default sort specified - this could cause infinite loops.`);
        }
        return this.apiService.query(entity, fullQuery, queryParams).pipe(map((response) => {
            return response;
        }));
    }
    convertAgGridQueryToQueryApi(filterModel, sortModel, groupKeys) {
        let query = new Query();
        keys(filterModel).forEach((fieldName) => {
            let fieldValue = filterModel[fieldName];
            if (fieldName === 'ag-Grid-AutoColumn') {
                if (groupKeys.length > 0) {
                    return; // remove tree column filter while expanding any node.
                }
                fieldName = 'name'; //TODO: this was for the Tree node column filter. mostly it will be name. Handle this gracefully.
            }
            let filter;
            if (fieldValue.type === 'blank') {
                filter = new Filter(fieldName, 'isnull', true);
            }
            else if (fieldValue.type === 'notBlank') {
                filter = new Filter(fieldName, 'isnull', false);
            }
            else if (fieldValue.filterType === 'boolean') {
                // Special boolean case
                if (fieldValue.filter === 'any') {
                    return;
                }
                filter = new Filter(fieldName, 'eq', fieldValue.filter);
            }
            else if (fieldValue.filterType === 'date' && fieldValue.filter) {
                filter = this.dateService.buildDateFilterGroup(fieldValue.filter, fieldName);
            }
            else if (fieldValue.filterType === 'set') {
                if (isBoolean(fieldValue.values[0])) {
                    filter = new Filter(fieldName, 'eq', fieldValue.values[0]);
                }
                else {
                    filter = new Filter(fieldName, 'in', fieldValue.values);
                }
            }
            else if (fieldValue.type === 'inRange') {
                filter = new FilterGroup([
                    new Filter(fieldName, 'lte', fieldValue.filterTo),
                    new Filter(fieldName, 'gte', fieldValue.filter)
                ], 'AND');
            }
            else if (fieldValue.filter) {
                let operator = OPERATOR_MAPPINGS[fieldValue.type] || 'eq';
                filter = new Filter(fieldName, operator, fieldValue.filter);
            }
            else {
                return;
            }
            query.filters.push(filter);
        });
        sortModel.forEach((sortModelItem) => {
            let fieldName = sortModelItem.colId;
            if (fieldName === 'ag-Grid-AutoColumn') {
                if (groupKeys.length > 0) {
                    return; // remove tree column filter while expanding any node.
                }
                fieldName = 'name'; //TODO: this was for the Tree node column filter. mostly it will be name. Handle this gracefully.
            }
            query.sort.push({
                field: fieldName,
                direction: sortModelItem.sort === 'asc' ? 'ASC' : 'DESC'
            });
        });
        return query;
    }
    static { this.ɵfac = function GridDataSourceService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || GridDataSourceService)(i0.ɵɵinject(i1.HttpClient), i0.ɵɵinject(i2.ApiService), i0.ɵɵinject(i2.DateService), i0.ɵɵinject(i2.GetEndPathNamePipe)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: GridDataSourceService, factory: GridDataSourceService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(GridDataSourceService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.HttpClient }, { type: i2.ApiService }, { type: i2.DateService }, { type: i2.GetEndPathNamePipe }], null); })();
//# sourceMappingURL=data:application/json;base64,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