import { Injectable } from '@angular/core';
import _ from 'lodash-es';
import { concatMap, map, of } from 'rxjs';
import { ALL_RECORDS_SIZE, API } from '../constants';
import { Query } from '../../query-filters/models/query';
import { Filter } from '../../query-filters/models/filter';
import * as i0 from "@angular/core";
import * as i1 from "../../utility/pipes/get-end-path-name.pipe";
import * as i2 from "@angular/common/http";
import * as i3 from "./entity.service";
export class ExportService {
    constructor(getEndPathNamePipe, http, entityService) {
        this.getEndPathNamePipe = getEndPathNamePipe;
        this.http = http;
        this.entityService = entityService;
    }
    /**
      * @ngdoc method
      * @name xf-ui#exportService#loadRowsForExport
      * @methodOf exportService
      *
      * @description
      * Loads exported versions of the given rows from CH.
      *
      * @param {array} rows An array of entity objects to export. Each object must have '@id' key.
      * @param {string} entityType The name of the entity (workflow).
      * @returns {Promise} Promise that will resolve with the exported hydra response
      **/
    loadRowsForExport(rows, entityType, selectedFields, skipExportFlag = false, includeVersions, childEntityName) {
        if (rows.length === 0) {
            return of([]);
        }
        let ids = rows.map((row) => {
            return row['@id'] ? this.getEndPathNamePipe.transform(row['@id']) : row.uuid;
        });
        let operator = rows.length > 1 ? 'in' : 'eq';
        let queryBody = new Query({
            logic: 'AND',
            filters: [new Filter('uuid', operator, ids.join('|'))]
        });
        if (selectedFields) {
            queryBody.selectedFields = selectedFields;
        }
        if (entityType === 'workflows' || entityType === 'workflow_collections') {
            queryBody.ignoreFields = ['createDate', 'createUser', 'modifyDate', 'modifyUser'];
        }
        let queryParams = {
            $export: !skipExportFlag,
            limit: ALL_RECORDS_SIZE
        };
        if (includeVersions) {
            queryParams.$versions = true;
        }
        if (!childEntityName) {
            queryParams.$relationships = true;
        }
        return this.entityService.build(entityType).pipe(concatMap((entity) => {
            return this.http.post(`${API.QUERY}${entityType}`, queryBody.getQueryObject(), { params: queryParams }).pipe(concatMap((data) => {
                if (!childEntityName) {
                    return of(data['hydra:member']);
                }
                const collectionUuids = _.map(data['hydra:member'], 'uuid');
                let childQueryBody = new Query({
                    logic: 'AND',
                    filters: [new Filter('collection.uuid', 'in', collectionUuids.join('|'))]
                });
                let childQueryParams = {
                    $relationships: true,
                    $export: !skipExportFlag,
                    limit: ALL_RECORDS_SIZE
                };
                if (includeVersions) {
                    childQueryParams.$versions = true;
                }
                if (childEntityName && childEntityName === 'workflows') {
                    childQueryBody.ignoreFields = ['createDate', 'createUser', 'modifyDate', 'modifyUser'];
                }
                return this.http.post(`${API.QUERY}${childEntityName}`, childQueryBody.getQueryObject(), { params: childQueryParams }).pipe(map((childData) => {
                    data['hydra:member'].forEach((collection) => {
                        let childItems = _.filter(childData['hydra:member'], (childItem) => {
                            return this.getEndPathNamePipe.transform(childItem.collection) === collection.uuid;
                        });
                        collection[childEntityName] = childItems;
                    });
                    return data['hydra:member'];
                }));
            }));
        }));
    }
    static { this.ɵfac = function ExportService_Factory(__ngFactoryType__) { return new (__ngFactoryType__ || ExportService)(i0.ɵɵinject(i1.GetEndPathNamePipe), i0.ɵɵinject(i2.HttpClient), i0.ɵɵinject(i3.EntityService)); }; }
    static { this.ɵprov = /*@__PURE__*/ i0.ɵɵdefineInjectable({ token: ExportService, factory: ExportService.ɵfac, providedIn: 'root' }); }
}
(() => { (typeof ngDevMode === "undefined" || ngDevMode) && i0.ɵsetClassMetadata(ExportService, [{
        type: Injectable,
        args: [{
                providedIn: 'root'
            }]
    }], () => [{ type: i1.GetEndPathNamePipe }, { type: i2.HttpClient }, { type: i3.EntityService }], null); })();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXhwb3J0LnNlcnZpY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi8uLi8uLi8uLi9wcm9qZWN0cy9zaGFyZWQvc3JjL2xpYi9jb3JlL3NlcnZpY2VzL2V4cG9ydC5zZXJ2aWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQU1BLE9BQU8sRUFBRSxVQUFVLEVBQUUsTUFBTSxlQUFlLENBQUM7QUFDM0MsT0FBTyxDQUFDLE1BQU0sV0FBVyxDQUFDO0FBQzFCLE9BQU8sRUFBbUMsU0FBUyxFQUFFLEdBQUcsRUFBRSxFQUFFLEVBQUUsTUFBTSxNQUFNLENBQUM7QUFDM0UsT0FBTyxFQUFFLGdCQUFnQixFQUFFLEdBQUcsRUFBRSxNQUFNLGNBQWMsQ0FBQztBQUNyRCxPQUFPLEVBQUUsS0FBSyxFQUFFLE1BQU0sa0NBQWtDLENBQUM7QUFDekQsT0FBTyxFQUFFLE1BQU0sRUFBRSxNQUFNLG1DQUFtQyxDQUFDOzs7OztBQVEzRCxNQUFNLE9BQU8sYUFBYTtJQUV4QixZQUNVLGtCQUFzQyxFQUN0QyxJQUFnQixFQUNoQixhQUE0QjtRQUY1Qix1QkFBa0IsR0FBbEIsa0JBQWtCLENBQW9CO1FBQ3RDLFNBQUksR0FBSixJQUFJLENBQVk7UUFDaEIsa0JBQWEsR0FBYixhQUFhLENBQWU7SUFDbEMsQ0FBQztJQUVMOzs7Ozs7Ozs7OztTQVdLO0lBQ0wsaUJBQWlCLENBQUMsSUFBVyxFQUFFLFVBQWtCLEVBQUUsY0FBd0IsRUFBRSxpQkFBMEIsS0FBSyxFQUFFLGVBQXlCLEVBQUUsZUFBd0I7UUFDL0osSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQ3RCLE9BQU8sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ2hCLENBQUM7UUFDRCxJQUFJLEdBQUcsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsR0FBUSxFQUFFLEVBQUU7WUFDOUIsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUM7UUFDL0UsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLFFBQVEsR0FBRyxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDN0MsSUFBSSxTQUFTLEdBQUcsSUFBSSxLQUFLLENBQUM7WUFDeEIsS0FBSyxFQUFFLEtBQUs7WUFDWixPQUFPLEVBQUUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxNQUFNLEVBQUUsUUFBUSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztTQUN2RCxDQUFDLENBQUM7UUFDSCxJQUFJLGNBQWMsRUFBRSxDQUFDO1lBQ25CLFNBQVMsQ0FBQyxjQUFjLEdBQUcsY0FBYyxDQUFDO1FBQzVDLENBQUM7UUFDRCxJQUFJLFVBQVUsS0FBSyxXQUFXLElBQUksVUFBVSxLQUFLLHNCQUFzQixFQUFFLENBQUM7WUFDeEUsU0FBUyxDQUFDLFlBQVksR0FBRyxDQUFDLFlBQVksRUFBRSxZQUFZLEVBQUUsWUFBWSxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ3BGLENBQUM7UUFDRCxJQUFJLFdBQVcsR0FBUTtZQUNyQixPQUFPLEVBQUUsQ0FBQyxjQUFjO1lBQ3hCLEtBQUssRUFBRSxnQkFBZ0I7U0FDeEIsQ0FBQztRQUNGLElBQUksZUFBZSxFQUFFLENBQUM7WUFDcEIsV0FBVyxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUM7UUFDL0IsQ0FBQztRQUNELElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztZQUNyQixXQUFXLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQztRQUNwQyxDQUFDO1FBQ0QsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLENBQzlDLFNBQVMsQ0FBQyxDQUFDLE1BQWMsRUFBRSxFQUFFO1lBQzNCLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxHQUFHLENBQUMsS0FBSyxHQUFHLFVBQVUsRUFBRSxFQUFFLFNBQVMsQ0FBQyxjQUFjLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FDMUcsU0FBUyxDQUFDLENBQUMsSUFBUyxFQUFFLEVBQUU7Z0JBQ3RCLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztvQkFDckIsT0FBTyxFQUFFLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7Z0JBQ2xDLENBQUM7Z0JBQ0QsTUFBTSxlQUFlLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7Z0JBQzVELElBQUksY0FBYyxHQUFHLElBQUksS0FBSyxDQUFDO29CQUM3QixLQUFLLEVBQUUsS0FBSztvQkFDWixPQUFPLEVBQUUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRSxJQUFJLEVBQUUsZUFBZSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO2lCQUMxRSxDQUFDLENBQUM7Z0JBRUgsSUFBSSxnQkFBZ0IsR0FBUTtvQkFDMUIsY0FBYyxFQUFFLElBQUk7b0JBQ3BCLE9BQU8sRUFBRSxDQUFDLGNBQWM7b0JBQ3hCLEtBQUssRUFBRSxnQkFBZ0I7aUJBQ3hCLENBQUM7Z0JBQ0YsSUFBSSxlQUFlLEVBQUUsQ0FBQztvQkFDcEIsZ0JBQWdCLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQztnQkFDcEMsQ0FBQztnQkFDRCxJQUFJLGVBQWUsSUFBSSxlQUFlLEtBQUssV0FBVyxFQUFFLENBQUM7b0JBQ3ZELGNBQWMsQ0FBQyxZQUFZLEdBQUcsQ0FBQyxZQUFZLEVBQUUsWUFBWSxFQUFFLFlBQVksRUFBRSxZQUFZLENBQUMsQ0FBQztnQkFDekYsQ0FBQztnQkFFRCxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsR0FBRyxDQUFDLEtBQUssR0FBRyxlQUFlLEVBQUUsRUFBRSxjQUFjLENBQUMsY0FBYyxFQUFFLEVBQUUsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FDekgsR0FBRyxDQUFDLENBQUMsU0FBYyxFQUFFLEVBQUU7b0JBQ3JCLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxVQUFlLEVBQUUsRUFBRTt3QkFDL0MsSUFBSSxVQUFVLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLEVBQUUsQ0FBQyxTQUFjLEVBQUUsRUFBRTs0QkFDdEUsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsS0FBSyxVQUFVLENBQUMsSUFBSSxDQUFDO3dCQUNyRixDQUFDLENBQUMsQ0FBQzt3QkFDSCxVQUFVLENBQUMsZUFBZ0IsQ0FBQyxHQUFHLFVBQVUsQ0FBQztvQkFDNUMsQ0FBQyxDQUFDLENBQUM7b0JBQ0gsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7Z0JBQzlCLENBQUMsQ0FBQyxDQUNILENBQUM7WUFDSixDQUFDLENBQUMsQ0FDSCxDQUFDO1FBQ0osQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNSLENBQUM7OEdBeEZVLGFBQWE7dUVBQWIsYUFBYSxXQUFiLGFBQWEsbUJBRlosTUFBTTs7aUZBRVAsYUFBYTtjQUh6QixVQUFVO2VBQUM7Z0JBQ1YsVUFBVSxFQUFFLE1BQU07YUFDbkIiLCJzb3VyY2VzQ29udGVudCI6WyIvKiBDb3B5cmlnaHQgc3RhcnRcbiAgQ29weXJpZ2h0IChDKSAyMDA4IC0gMjAyNSBGb3J0aW5ldCBJbmMuXG4gIEFsbCByaWdodHMgcmVzZXJ2ZWQuXG4gIEZPUlRJTkVUIENPTkZJREVOVElBTCAmIEZPUlRJTkVUIFBST1BSSUVUQVJZIFNPVVJDRSBDT0RFXG4gIENvcHlyaWdodCBlbmQgKi9cbmltcG9ydCB7IEh0dHBDbGllbnQgfSBmcm9tICdAYW5ndWxhci9jb21tb24vaHR0cCc7XG5pbXBvcnQgeyBJbmplY3RhYmxlIH0gZnJvbSAnQGFuZ3VsYXIvY29yZSc7XG5pbXBvcnQgXyBmcm9tICdsb2Rhc2gtZXMnO1xuaW1wb3J0IHsgT2JzZXJ2YWJsZSwgU3ViamVjdCwgY2F0Y2hFcnJvciwgY29uY2F0TWFwLCBtYXAsIG9mIH0gZnJvbSAncnhqcyc7XG5pbXBvcnQgeyBBTExfUkVDT1JEU19TSVpFLCBBUEkgfSBmcm9tICcuLi9jb25zdGFudHMnO1xuaW1wb3J0IHsgUXVlcnkgfSBmcm9tICcuLi8uLi9xdWVyeS1maWx0ZXJzL21vZGVscy9xdWVyeSc7XG5pbXBvcnQgeyBGaWx0ZXIgfSBmcm9tICcuLi8uLi9xdWVyeS1maWx0ZXJzL21vZGVscy9maWx0ZXInO1xuaW1wb3J0IHsgRW50aXR5U2VydmljZSB9IGZyb20gJy4vZW50aXR5LnNlcnZpY2UnO1xuaW1wb3J0IHsgRW50aXR5IH0gZnJvbSAnLi4vbW9kZWxzL2VudGl0eSc7XG5pbXBvcnQgeyBHZXRFbmRQYXRoTmFtZVBpcGUgfSBmcm9tICcuLi8uLi91dGlsaXR5L3BpcGVzL2dldC1lbmQtcGF0aC1uYW1lLnBpcGUnO1xuXG5ASW5qZWN0YWJsZSh7XG4gIHByb3ZpZGVkSW46ICdyb290J1xufSlcbmV4cG9ydCBjbGFzcyBFeHBvcnRTZXJ2aWNlIHtcblxuICBjb25zdHJ1Y3RvcihcbiAgICBwcml2YXRlIGdldEVuZFBhdGhOYW1lUGlwZTogR2V0RW5kUGF0aE5hbWVQaXBlLFxuICAgIHByaXZhdGUgaHR0cDogSHR0cENsaWVudCxcbiAgICBwcml2YXRlIGVudGl0eVNlcnZpY2U6IEVudGl0eVNlcnZpY2VcbiAgKSB7IH1cblxuICAvKipcbiAgICAqIEBuZ2RvYyBtZXRob2RcbiAgICAqIEBuYW1lIHhmLXVpI2V4cG9ydFNlcnZpY2UjbG9hZFJvd3NGb3JFeHBvcnRcbiAgICAqIEBtZXRob2RPZiBleHBvcnRTZXJ2aWNlXG4gICAgKlxuICAgICogQGRlc2NyaXB0aW9uXG4gICAgKiBMb2FkcyBleHBvcnRlZCB2ZXJzaW9ucyBvZiB0aGUgZ2l2ZW4gcm93cyBmcm9tIENILlxuICAgICpcbiAgICAqIEBwYXJhbSB7YXJyYXl9IHJvd3MgQW4gYXJyYXkgb2YgZW50aXR5IG9iamVjdHMgdG8gZXhwb3J0LiBFYWNoIG9iamVjdCBtdXN0IGhhdmUgJ0BpZCcga2V5LlxuICAgICogQHBhcmFtIHtzdHJpbmd9IGVudGl0eVR5cGUgVGhlIG5hbWUgb2YgdGhlIGVudGl0eSAod29ya2Zsb3cpLlxuICAgICogQHJldHVybnMge1Byb21pc2V9IFByb21pc2UgdGhhdCB3aWxsIHJlc29sdmUgd2l0aCB0aGUgZXhwb3J0ZWQgaHlkcmEgcmVzcG9uc2VcbiAgICAqKi9cbiAgbG9hZFJvd3NGb3JFeHBvcnQocm93czogYW55W10sIGVudGl0eVR5cGU6IHN0cmluZywgc2VsZWN0ZWRGaWVsZHM6IHN0cmluZ1tdLCBza2lwRXhwb3J0RmxhZzogYm9vbGVhbiA9IGZhbHNlLCBpbmNsdWRlVmVyc2lvbnM/OiBib29sZWFuLCBjaGlsZEVudGl0eU5hbWU/OiBzdHJpbmcpOiBPYnNlcnZhYmxlPGFueT4ge1xuICAgIGlmIChyb3dzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIG9mKFtdKTtcbiAgICB9XG4gICAgbGV0IGlkcyA9IHJvd3MubWFwKChyb3c6IGFueSkgPT4ge1xuICAgICAgcmV0dXJuIHJvd1snQGlkJ10gPyB0aGlzLmdldEVuZFBhdGhOYW1lUGlwZS50cmFuc2Zvcm0ocm93WydAaWQnXSkgOiByb3cudXVpZDtcbiAgICB9KTtcblxuICAgIGxldCBvcGVyYXRvciA9IHJvd3MubGVuZ3RoID4gMSA/ICdpbicgOiAnZXEnO1xuICAgIGxldCBxdWVyeUJvZHkgPSBuZXcgUXVlcnkoe1xuICAgICAgbG9naWM6ICdBTkQnLFxuICAgICAgZmlsdGVyczogW25ldyBGaWx0ZXIoJ3V1aWQnLCBvcGVyYXRvciwgaWRzLmpvaW4oJ3wnKSldXG4gICAgfSk7XG4gICAgaWYgKHNlbGVjdGVkRmllbGRzKSB7XG4gICAgICBxdWVyeUJvZHkuc2VsZWN0ZWRGaWVsZHMgPSBzZWxlY3RlZEZpZWxkcztcbiAgICB9XG4gICAgaWYgKGVudGl0eVR5cGUgPT09ICd3b3JrZmxvd3MnIHx8IGVudGl0eVR5cGUgPT09ICd3b3JrZmxvd19jb2xsZWN0aW9ucycpIHtcbiAgICAgIHF1ZXJ5Qm9keS5pZ25vcmVGaWVsZHMgPSBbJ2NyZWF0ZURhdGUnLCAnY3JlYXRlVXNlcicsICdtb2RpZnlEYXRlJywgJ21vZGlmeVVzZXInXTtcbiAgICB9XG4gICAgbGV0IHF1ZXJ5UGFyYW1zOiBhbnkgPSB7XG4gICAgICAkZXhwb3J0OiAhc2tpcEV4cG9ydEZsYWcsXG4gICAgICBsaW1pdDogQUxMX1JFQ09SRFNfU0laRVxuICAgIH07XG4gICAgaWYgKGluY2x1ZGVWZXJzaW9ucykge1xuICAgICAgcXVlcnlQYXJhbXMuJHZlcnNpb25zID0gdHJ1ZTtcbiAgICB9XG4gICAgaWYgKCFjaGlsZEVudGl0eU5hbWUpIHtcbiAgICAgIHF1ZXJ5UGFyYW1zLiRyZWxhdGlvbnNoaXBzID0gdHJ1ZTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuZW50aXR5U2VydmljZS5idWlsZChlbnRpdHlUeXBlKS5waXBlKFxuICAgICAgY29uY2F0TWFwKChlbnRpdHk6IEVudGl0eSkgPT4ge1xuICAgICAgICByZXR1cm4gdGhpcy5odHRwLnBvc3QoYCR7QVBJLlFVRVJZfSR7ZW50aXR5VHlwZX1gLCBxdWVyeUJvZHkuZ2V0UXVlcnlPYmplY3QoKSwgeyBwYXJhbXM6IHF1ZXJ5UGFyYW1zIH0pLnBpcGUoXG4gICAgICAgICAgY29uY2F0TWFwKChkYXRhOiBhbnkpID0+IHtcbiAgICAgICAgICAgIGlmICghY2hpbGRFbnRpdHlOYW1lKSB7XG4gICAgICAgICAgICAgIHJldHVybiBvZihkYXRhWydoeWRyYTptZW1iZXInXSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zdCBjb2xsZWN0aW9uVXVpZHMgPSBfLm1hcChkYXRhWydoeWRyYTptZW1iZXInXSwgJ3V1aWQnKTtcbiAgICAgICAgICAgIGxldCBjaGlsZFF1ZXJ5Qm9keSA9IG5ldyBRdWVyeSh7XG4gICAgICAgICAgICAgIGxvZ2ljOiAnQU5EJyxcbiAgICAgICAgICAgICAgZmlsdGVyczogW25ldyBGaWx0ZXIoJ2NvbGxlY3Rpb24udXVpZCcsICdpbicsIGNvbGxlY3Rpb25VdWlkcy5qb2luKCd8JykpXVxuICAgICAgICAgICAgfSk7XG5cbiAgICAgICAgICAgIGxldCBjaGlsZFF1ZXJ5UGFyYW1zOiBhbnkgPSB7XG4gICAgICAgICAgICAgICRyZWxhdGlvbnNoaXBzOiB0cnVlLFxuICAgICAgICAgICAgICAkZXhwb3J0OiAhc2tpcEV4cG9ydEZsYWcsXG4gICAgICAgICAgICAgIGxpbWl0OiBBTExfUkVDT1JEU19TSVpFXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgaWYgKGluY2x1ZGVWZXJzaW9ucykge1xuICAgICAgICAgICAgICBjaGlsZFF1ZXJ5UGFyYW1zLiR2ZXJzaW9ucyA9IHRydWU7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAoY2hpbGRFbnRpdHlOYW1lICYmIGNoaWxkRW50aXR5TmFtZSA9PT0gJ3dvcmtmbG93cycpIHtcbiAgICAgICAgICAgICAgY2hpbGRRdWVyeUJvZHkuaWdub3JlRmllbGRzID0gWydjcmVhdGVEYXRlJywgJ2NyZWF0ZVVzZXInLCAnbW9kaWZ5RGF0ZScsICdtb2RpZnlVc2VyJ107XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIHJldHVybiB0aGlzLmh0dHAucG9zdChgJHtBUEkuUVVFUll9JHtjaGlsZEVudGl0eU5hbWV9YCwgY2hpbGRRdWVyeUJvZHkuZ2V0UXVlcnlPYmplY3QoKSwgeyBwYXJhbXM6IGNoaWxkUXVlcnlQYXJhbXMgfSkucGlwZShcbiAgICAgICAgICAgICAgbWFwKChjaGlsZERhdGE6IGFueSkgPT4ge1xuICAgICAgICAgICAgICAgIGRhdGFbJ2h5ZHJhOm1lbWJlciddLmZvckVhY2goKGNvbGxlY3Rpb246IGFueSkgPT4ge1xuICAgICAgICAgICAgICAgICAgbGV0IGNoaWxkSXRlbXMgPSBfLmZpbHRlcihjaGlsZERhdGFbJ2h5ZHJhOm1lbWJlciddLCAoY2hpbGRJdGVtOiBhbnkpID0+IHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHRoaXMuZ2V0RW5kUGF0aE5hbWVQaXBlLnRyYW5zZm9ybShjaGlsZEl0ZW0uY29sbGVjdGlvbikgPT09IGNvbGxlY3Rpb24udXVpZDtcbiAgICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgICAgY29sbGVjdGlvbltjaGlsZEVudGl0eU5hbWUhXSA9IGNoaWxkSXRlbXM7XG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgcmV0dXJuIGRhdGFbJ2h5ZHJhOm1lbWJlciddO1xuICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgKTtcbiAgICAgICAgICB9KVxuICAgICAgICApO1xuICAgICAgfSkpO1xuICB9XG59XG4iXX0=